﻿using Microsoft.AspNetCore.Http;
using Microsoft.IdentityModel.Tokens;
using Newtonsoft.Json;
using Resonance.Core.Exceptions;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ConfigurationModels.Salesforce;
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.IdentityModel.Tokens.Jwt;
using System.IO;
using System.Net;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;
using System.Xml;

namespace Resonance.Core.Helpers
{
    public class AuthenticatedWebClient : WebClient
    {
        public string AuthToken { get; set; }
        public AuthenticatedWebClient(string authToken) : base()
        {
            AuthToken = authToken;
        }

        protected override WebRequest GetWebRequest(Uri address)
        {
            WebRequest request = base.GetWebRequest(address);
            request.Headers.Add("Authorization", "Bearer " + AuthToken);
            return request;
        }
    }

    public static class SalesforceHelpers
    {
        private static SalesforceCredentialsModel Credentials { get; set; }

        private static string _metricSuccessKey = "salesforce_success";
        private static string _metricErrorKey = "salesforce_error";

        static SalesforceHelpers()
        {
            string credPath = @"salesforce/prod_credentials_jwt.serialized.encrypted";
            if (Constants.AppConfig.Application.Environment  != "Production")
            {
                credPath = @"salesforce/test_credentials_jwt.serialized.encrypted";
            }
            Credentials = JsonConvert.DeserializeObject<SalesforceCredentialsModel>(AwsEncryption.Decrypt(@"crs-data-credentials", credPath));
        }

        public static void DumpUsers()
        {
            List<string> records = new List<string>();
            var salesforceInfo = SalesforceHelpers.GetAuthTokenAndUrl();

            AuthenticatedWebClient webClient = new AuthenticatedWebClient(salesforceInfo.Item1);

            Dictionary<string, string> roleLookup = new Dictionary<string, string>();
            string roleQuery = "SELECT+Id+,+Name+FROM+UserRole";
            string roleurl = string.Format(@"{0}{1}?q={2}", salesforceInfo.Item2, "/services/data/v32.0/query/", roleQuery);
            string roleResponse = webClient.DownloadString(roleurl);
            dynamic roleResponseData = JsonConvert.DeserializeObject<dynamic>(roleResponse);
            foreach (dynamic record in roleResponseData.records)
            {
                string roleID = record.Id;
                string roleName = record.Name;
                roleLookup[roleID] = roleName;
            }

                string query = "SELECT+Id+,+FederationIdentifier+,+Name+,+UserRoleId+,+ManagerId+FROM+User";
            string finalURL = string.Format(@"{0}{1}?q={2}", salesforceInfo.Item2, "/services/data/v32.0/query/", query);

            string response = webClient.DownloadString(finalURL);
            dynamic responseData = JsonConvert.DeserializeObject<dynamic>(response);
            records.Add("UserID,SSOName,Name,Role,ManagerID");
            foreach(dynamic record in responseData.records)
            {
                string roleID = record.UserRoleId;
                string role = "";
                if (roleID != null && roleID != "")
                {
                    roleLookup.TryGetValue(roleID, out role);
                }
                records.Add(string.Format("{0},{1},{2},{3},{4}", record.Id, record.FederationIdentifier, record.Name, role, record.ManagerId));
            }
            File.WriteAllLines(@"c:\test\users.csv", records);
        }

        public static void TestLogFile()
        {
            var salesforceInfo = SalesforceHelpers.GetAuthTokenAndUrlWithJWT();

            AuthenticatedWebClient webClient = new AuthenticatedWebClient(salesforceInfo.Item1);

            string query = "SELECT+Id+,+EventType+,+LogFile+,+LogDate +,+LogFileLength + FROM + EventLogFile";

            string finalURL = string.Format(@"{0}{1}?q={2}", salesforceInfo.Item2, "/services/data/v32.0/query/", query);

            string response = webClient.DownloadString(finalURL);
            dynamic responseData = JsonConvert.DeserializeObject<dynamic>(response);
            Dictionary<string, List<string>> namesAndURLs = new Dictionary<string, List<string>>();

            List<string> urls = new List<string>();

            foreach(dynamic record in responseData.records)
            //Parallel.ForEach(responseData.records, (dynamic record) =>
            {
                string url = record.attributes.url;
                urls.Add(url);
                /*string logFile = webClient.DownloadString(salesforceInfo.Item2 + url);
                dynamic logFileData = JsonConvert.DeserializeObject<dynamic>(logFile);
                string logFileUrl = logFileData.LogFile;
                string logFileText = webClient.DownloadString(salesforceInfo.Item2 + logFileUrl);
                string logFileName = logFileData.LogDate.ToString("yyyy-dd-M--HH-mm-ss");
                namesAndURLs[logFileName] =  logFileUrl;
                */
                //File.WriteAllText(Path.Combine(@"C:\test\salesforce_access", logFileName + ".txt"), logFileText);
            }
            //);
            Console.WriteLine("Getting meta data");
            Parallel.ForEach(urls, (url) =>
            {
                WebClient secondaryClient = new AuthenticatedWebClient(salesforceInfo.Item1);
                string logFile = secondaryClient.DownloadString(salesforceInfo.Item2 + url);
                dynamic logFileData = JsonConvert.DeserializeObject<dynamic>(logFile);
                string logFileUrl = logFileData.LogFile;
                string logFileType = logFileData.EventType;
                //
                string logFileName = logFileData.LogDate.ToString("yyyy-dd-M--HH-mm-ss") + "-" + logFileType;
                lock (namesAndURLs)
                {
                    if (!namesAndURLs.ContainsKey(logFileName))
                    {
                        namesAndURLs.Add(logFileName, new List<string>());
                    }
                    namesAndURLs[logFileName].Add(logFileUrl);
                }
            });
            Console.WriteLine("Downloading files");

            Parallel.ForEach(namesAndURLs, (entry) =>
            {

                WebClient secondaryClient = new AuthenticatedWebClient(salesforceInfo.Item1);
                string logFileName = entry.Key;
                List<string> logFileUrls = entry.Value;
                int counter = 1;
                foreach (string logFileUrl in logFileUrls)
                {
                    try
                    {
                        string logFileText = secondaryClient.DownloadString(salesforceInfo.Item2 + logFileUrl);
                        Console.WriteLine("Downloading " + logFileName);
                        File.WriteAllText(Path.Combine(@"C:\test\salesforce_access_11_8", logFileName + "-" + (counter++) + ".txt"), logFileText);
                        Console.WriteLine("Done Downloading " + logFileName);
                    }
                    catch(Exception)
                    {
                        Console.WriteLine("Error downloading " + logFileName);

                    }
                }
            });
        }

        public static Tuple<string, string> GetAuthTokenAndUrl()
        {
            WebClient client = new WebClient();
            NameValueCollection uploadParams = new NameValueCollection();
            uploadParams.Add("grant_type", "password");
            uploadParams.Add("client_id", Credentials.ClientID);
            uploadParams.Add("client_secret", Credentials.ClientSecret);
            uploadParams.Add("username", Credentials.UserName);
            uploadParams.Add("password", Credentials.Password);

            var response = Encoding.UTF8.GetString(client.UploadValues(Credentials.BaseURL + "/services/oauth2/token", "POST", uploadParams));

            dynamic results = JsonConvert.DeserializeObject<dynamic>(response);

            return new Tuple<string, string>(results.access_token.ToString(), results.instance_url.ToString());
        }


        public static void FromNewXmlString(this RSA rsa, string xmlString)
        {
            RSAParameters parameters = new RSAParameters();

            XmlDocument xmlDoc = new XmlDocument();
            xmlDoc.LoadXml(xmlString);

            if (xmlDoc.DocumentElement.Name.Equals("RSAKeyValue"))
            {
                foreach (XmlNode node in xmlDoc.DocumentElement.ChildNodes)
                {
                    switch (node.Name)
                    {
                        case "Modulus": parameters.Modulus = (string.IsNullOrEmpty(node.InnerText) ? null : Convert.FromBase64String(node.InnerText)); break;
                        case "Exponent": parameters.Exponent = (string.IsNullOrEmpty(node.InnerText) ? null : Convert.FromBase64String(node.InnerText)); break;
                        case "P": parameters.P = (string.IsNullOrEmpty(node.InnerText) ? null : Convert.FromBase64String(node.InnerText)); break;
                        case "Q": parameters.Q = (string.IsNullOrEmpty(node.InnerText) ? null : Convert.FromBase64String(node.InnerText)); break;
                        case "DP": parameters.DP = (string.IsNullOrEmpty(node.InnerText) ? null : Convert.FromBase64String(node.InnerText)); break;
                        case "DQ": parameters.DQ = (string.IsNullOrEmpty(node.InnerText) ? null : Convert.FromBase64String(node.InnerText)); break;
                        case "InverseQ": parameters.InverseQ = (string.IsNullOrEmpty(node.InnerText) ? null : Convert.FromBase64String(node.InnerText)); break;
                        case "D": parameters.D = (string.IsNullOrEmpty(node.InnerText) ? null : Convert.FromBase64String(node.InnerText)); break;
                    }
                }
            }
            else
            {
                throw new Exception("Invalid XML RSA key.");
            }

            rsa.ImportParameters(parameters);
        }

        public static bool CheckObjectExistance(string query)
        {
            try
            {
                var results = QueryData(query);
                return results.Count > 0;
            }
            catch(Exception)
            {
                return false;
            }
        }

        public static List<dynamic> QueryData(string query)
        {

            List<dynamic> responses = new List<dynamic>();

            var salesforceInfo = SalesforceHelpers.GetAuthTokenAndUrlWithJWT();

            AuthenticatedWebClient webClient = new AuthenticatedWebClient(salesforceInfo.Item1);

            string finalURL = string.Format(@"{0}{1}?q={2}", salesforceInfo.Item2, "/services/data/v20.0/query/", query);

            string response = null;
            try
            {
                response = webClient.DownloadString(finalURL);
                Log.Metric(_metricSuccessKey, null, true);
                if (!string.IsNullOrWhiteSpace(response))
                {
                    dynamic responseData = JsonConvert.DeserializeObject<dynamic>(response);
                    while (true)
                    {
                        responses.AddRange(responseData.records);
                        if (responseData.done.ToString() == "True")
                        {
                            break;
                        }
                        else
                        {
                            string nextRecordsURL = responseData.nextRecordsUrl;
                            string nextURL = string.Format(@"{0}{1}?q={2}", salesforceInfo.Item2, nextRecordsURL, query);
                            string nextResponse = webClient.DownloadString(nextURL);
                            responseData = JsonConvert.DeserializeObject<dynamic>(nextResponse);
                        }
                    }
                }
            }
            catch (WebException ex)
            {
                Log.Metric(_metricErrorKey, null, true);
                try
                {
                    using (var errorResponse = new StreamReader(ex?.Response?.GetResponseStream()))
                    {
                        Log.Info(errorResponse.ReadToEnd());
                    }
                }
                catch (Exception ex2)
                {
                    Log.Error(ex2);
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return responses;
        }

        // Async for User Details page :: Todo, get with Nic and see if we can remove duplication of
        public static async Task<List<dynamic>> QueryDataAsync(string query, HttpContext context = null)
        {

            List<dynamic> responses = new List<dynamic>();

            var salesforceInfo = SalesforceHelpers.GetAuthTokenAndUrlWithJWT();
            string finalURL = string.Format($@"{salesforceInfo.Item2}/services/data/v20.0/query/?q={query}");

            WebRequestResponse response = null;
            try
            {
                response = await WebRequestHelper.GetDataAsync(url: finalURL, bearer: salesforceInfo.Item1, context: context);
                if (response != null && response.IsSuccess && !string.IsNullOrWhiteSpace(response.Data))
                {
                    Log.Metric(_metricSuccessKey, null, true);
                    dynamic responseData = JsonConvert.DeserializeObject<dynamic>(response.Data);
                    while (true)
                    {
                        responses.AddRange(responseData.records);
                        if (responseData.done.ToString() == "True")
                        {
                            break;
                        }
                        else
                        {
                            string nextRecordsURL = responseData.nextRecordsUrl;
                            string nextURL = string.Format(@"{0}{1}?q={2}", salesforceInfo.Item2, nextRecordsURL, query);
                            var nextResponse = await WebRequestHelper.GetDataAsync(url: nextURL, bearer: salesforceInfo.Item1, context: context);
                            if(nextResponse != null && nextResponse.IsSuccess && !string.IsNullOrWhiteSpace(nextResponse.Data))
                            {
                                responseData = JsonConvert.DeserializeObject<dynamic>(nextResponse.Data);
                            }
                        }
                    }
                }
            }
            catch (WebException ex)
            {
                Log.Metric(_metricErrorKey, null, true);
                try
                {
                    using (var errorResponse = new StreamReader(ex?.Response?.GetResponseStream()))
                    {
                        Log.Info(errorResponse.ReadToEnd());
                    }
                }
                catch (Exception ex2)
                {
                    Log.Error(ex2);
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return responses;
        }


        public static bool UpdateRecordByExternalID(string type, string idField, string id, object payload)
        {
            try
            {   
                var salesforceInfo = GetAuthTokenAndUrlWithJWT();

                AuthenticatedWebClient webClient = new AuthenticatedWebClient(salesforceInfo.Item1);
                webClient.Headers.Add("Content-Type", "application/json");

                string finalURL = string.Format(@"{0}{1}{2}/{3}/{4}", salesforceInfo.Item2, "/services/data/v20.0/sobjects/", type, idField, id);

                string response = webClient.UploadString(finalURL, "PATCH", JsonConvert.SerializeObject(payload));
                Log.Metric(_metricSuccessKey, null, true);

                return true;
            }
            catch(WebException ex)
            {
                Log.Metric(_metricErrorKey, null, true);
                var resp = new StreamReader(ex.Response.GetResponseStream()).ReadToEnd();
                LoggingHelpers.Log.Error("Salesforce Error: " + resp);

                throw new SalesforceException(resp);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                return false;
            }
        }

        public static bool UpdateRecord(string type, string id, object payload)
        {
            try
            {
                var salesforceInfo = GetAuthTokenAndUrlWithJWT();

                AuthenticatedWebClient webClient = new AuthenticatedWebClient(salesforceInfo.Item1);
                webClient.Headers.Add("Content-Type", "application/json");

                string finalURL = string.Format(@"{0}{1}{2}/{3}", salesforceInfo.Item2, "/services/data/v20.0/sobjects/", type, id);

                string response = webClient.UploadString(finalURL, "PATCH", JsonConvert.SerializeObject(payload));
                Log.Metric(_metricSuccessKey, null, true);
                return true;
            }
            catch(WebException)
            {
                Log.Metric(_metricErrorKey, null, true);
                return false;
            }
            catch(Exception)
            {
                return false;
            }
        }

        public static string GetOwnerID(string localUserName)
        {
            try
            {
                var results = QueryData($"SELECT Id FROM User WHERE FederationIdentifier = '{localUserName}' AND IsActive = true");
                return results[0].Id;
            }
            catch (Exception)
            {

            }
            return "0051K000008duqbQAA";

        }

        public static dynamic PostData(string endpoint, object payload)
        {
            try
            {   
                var salesforceInfo = SalesforceHelpers.GetAuthTokenAndUrlWithJWT();

                AuthenticatedWebClient webClient = new AuthenticatedWebClient(salesforceInfo.Item1);
                webClient.Headers.Add("Content-Type", "application/json");

                string channelResponse = webClient.UploadString(salesforceInfo.Item2 + endpoint, "POST", JsonConvert.SerializeObject(payload));

                Log.Metric(_metricSuccessKey, null, true);

                return JsonConvert.DeserializeObject<dynamic>(channelResponse);
            }
            catch(WebException ex)
            {
                Log.Metric(_metricErrorKey, null, true);
                using (var exResponse = new StreamReader(ex.Response.GetResponseStream()))
                {
                    string error = exResponse.ReadToEnd();
                    LoggingHelpers.Log.Error("Salesforce Error: " + error);

                    throw new SalesforceException(error);
                }
            }
            catch(Exception)
            {
                return null;
            }
        }

        public static Tuple<string, string> GetAuthTokenAndUrlWithJWT()
        {
            RSA rsa = RSA.Create();
            rsa.FromNewXmlString(Credentials.KeyXML);

            var privateKey = new RsaSecurityKey(rsa);
            var rsaSigningCredentials = new SigningCredentials(privateKey, SecurityAlgorithms.RsaSha256);

            string clientID = Credentials.ClientID;
            string tokenUser = Credentials.UserName;

            var header = new JwtHeader(rsaSigningCredentials);

            var jwtPayload = new JwtPayload
            {
                {
                    "iss", clientID
                },
                {
                    "sub", tokenUser
                },
                {
                    "aud", Credentials.Audience
                },
                {
                    "exp", ((DateTimeOffset.UtcNow.ToUnixTimeMilliseconds() / 1000) + 300).ToString()
                }
            };

            var secToken = new JwtSecurityToken(header, jwtPayload);
            var handler = new JwtSecurityTokenHandler();

            WebClient client = new WebClient();
            NameValueCollection uploadParams = new NameValueCollection
            {
                { "grant_type", "urn:ietf:params:oauth:grant-type:jwt-bearer" },
                { "assertion", handler.WriteToken(secToken) }
            };


            string response = "";
            for (int retry = 0; retry < 3; retry++)
            {
                try
                {
                    response = Encoding.UTF8.GetString(client.UploadValues(Credentials.BaseURL + "/services/oauth2/token", "POST", uploadParams));
                    Log.Metric(_metricSuccessKey, null, true);
                    break;
                }
                catch(WebException ex)
                {
                    Log.Metric(_metricErrorKey, null, true);
                    using (var exResponse = new StreamReader(ex.Response.GetResponseStream()))
                    {
                        string error =  exResponse.ReadToEnd();
                        Log.Error(error);
                    }
                }
                catch(Exception)
                {

                }
            }

            dynamic results = JsonConvert.DeserializeObject<dynamic>(response);

            return new Tuple<string, string>(results.access_token.ToString(), results.instance_url.ToString());
        }

        

    }
}
