﻿using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Options;
using Resonance.Core.Exceptions;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.ErrorHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StatsDHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models.AuthModels;
using System;
using System.Diagnostics;
using System.Net;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;

namespace Resonance.Core.Middleware
{
    public class ErrorHandlerMiddlewareOptions
    {
    }

    public class ErrorHandlerMiddleware
    {
        private readonly RequestDelegate _next;
        private readonly ErrorHandlerMiddlewareOptions _options;

        public ErrorHandlerMiddleware(RequestDelegate next, IOptions<ErrorHandlerMiddlewareOptions> options)
        {
            _next = next;
            _options = options.Value;
        }

        public async Task Invoke(HttpContext context)
        {
            try
            {
                await _next(context);
            }
            catch (AuthException ex)
            {

                if (Constants.AppConfig.Application.Environment != "Production")
                {
                    Log.Warn(ex.ToString(), context: context);
                }
                context.Response.Clear();
                context.Response.StatusCode = 403;
                context.Response.Redirect("/error/403");
            }
            catch (AuthFailureException ex)
            {
                if (Constants.AppConfig.Application.Environment != "Production")
                {
                    Log.Warn(ex.ToString(), context: context);
                }
                context.Response.Clear();
                context.Response.StatusCode = 401;
                var option = new CookieOptions();
                option.Expires = DateTime.Now.AddMinutes(10);
                context.Response.Redirect("/error/401");
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: context);
                await HandleExceptionAsync(context, ex);
            }
        }

        /// <summary>
        /// Exception handler pulled from https://stackoverflow.com/questions/49241434/how-do-i-catch-unhandled-exceptions-in-asp-net-core-2-0-before-the-page-is-rend
        /// </summary>
        private static async Task HandleExceptionAsync(HttpContext context, Exception exception)
        {
            string errorCode = ErrorCodeHelper.CalculateErrorCode(context.TraceIdentifier);
            string message = string.Format($"An unhandled exception occurred; please contact the help desk with the following error code: '{errorCode}'  [{context.TraceIdentifier}]");

            Log.Error($"{exception} {message}", context: context);

            context.Response.ContentType = "text/plain";
            context.Response.StatusCode = (int)HttpStatusCode.InternalServerError;

            await context.Response.WriteAsync(message);
        }
    }
}