﻿using Amazon.CloudWatch;
using Amazon.CloudWatch.Model;
using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Options;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StatsDHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models.AuthModels;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Threading.Tasks;

namespace Resonance.Core.Middleware
{
    public class TraceMiddlewareOptions
    {
    }

    public class TraceMiddleware
    {
        private readonly RequestDelegate _next;
        private readonly TraceMiddlewareOptions _options;

        public TraceMiddleware(RequestDelegate next, IOptions<TraceMiddlewareOptions> options)
        {
            _next = next;
            _options = options.Value;
        }

        public async Task Invoke(HttpContext context)
        {
            var stopwatch = new Stopwatch();
            var request = context.Request;
            var builder = new StringBuilder();
            builder.AppendLine($"{Environment.NewLine}REQUEST");
            builder.AppendLine($"{request.Method} - {request.Path.Value}{request.QueryString}");
            builder.AppendLine($"Content-Type: {request.ContentType ?? "Not specified"}");
            builder.AppendLine($@"***");
            stopwatch.Start();
            await _next(context);
            stopwatch.Stop();

            var response = context.Response;
            string originalpath = (context.Items["originalPath"] as string) ?? "";
            AuthTokenData token = ((AuthTokenData)context.Items["AuthTokenData"]) ?? null;

            builder.AppendLine($"{Environment.NewLine}RESPONSE");
            builder.AppendLine($"Endpoint: {request.Method} - {request.Path.Value}{request.QueryString}");
            builder.AppendLine($"Status Code: {response.StatusCode}");
            builder.AppendLine($@"***");
            builder.AppendLine($@"METADATA");
            builder.AppendLine($"Host: {request.Host}");
            builder.AppendLine($@"User Ip: {HeaderHelper.GetRequestIP(context)}");
            builder.AppendLine($@"User Agent: {HeaderHelper.GetUserAgent(context)}");
            if (token != null && !string.IsNullOrWhiteSpace(token.User))
            {
                builder.AppendLine($"User: {token.User}");
            }
            if (!string.IsNullOrWhiteSpace(originalpath))
            {
                builder.AppendLine($"Original Path: {originalpath}");
            }

            builder.AppendLine($"Response elapsed time: {stopwatch.ElapsedMilliseconds}ms");

            try
            {
                var statsdname = $"{request.Path.Value}";
                if (!string.IsNullOrWhiteSpace(statsdname) && context.Request.Method != "OPTIONS")
                {
                    Log.Info($"{DateTime.Now.ToString("s")}{Environment.NewLine}{builder.ToString()}", context: context);

                    statsdname = statsdname.Length > 1
                        ? statsdname.Replace("/", "_").Replace(".", "_").Substring(1)
                        : "root";
                    bool isLoggedIn = !string.IsNullOrWhiteSpace(token?.User);
                    CloudwatchHelper.EnqueueMetricRequest("pageload", stopwatch.Elapsed.Milliseconds, context.Request.HttpContext, StandardUnit.Milliseconds);
                    CloudwatchHelper.EnqueueMetricRequest("pageload_status", context.Response.StatusCode, context.Request.HttpContext, StandardUnit.Milliseconds);
                    StatsDHelper.Counter(measurement: $"pageload", measurementType: "access", val: 1, location: statsdname, additionalTags: $",http_method={request.Method},user_id={token?.User}");
                    StatsDHelper.Counter(measurement: $"pageload", measurementType: "elapsed", val: stopwatch.Elapsed.Milliseconds, location: statsdname, additionalTags: $",http_method={request.Method},user_id={token?.User}");
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: context);
            }
            stopwatch = null;
        }
    }
}