﻿using Amazon.S3;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Resonance.Core.Models.ServiceModels.ActivityLoggerService;
using System.ComponentModel;
using System.Collections.Concurrent;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.StringHelpers;

namespace Resonance.Core.Services.ActivityLoggerService
{
    public class ActivityLoggerService : IActivityLoggerService
    {
        private static ActivityLogger _activityLogger;
        private string _baseS3Path;
        private string _name;        

        private BlockingCollection<ActivityLogData> _queue = new BlockingCollection<ActivityLogData>();

        public ActivityLoggerService(string name, string environment)
        {            
            _name = name;            
            _activityLogger = new ActivityLogger();
            _baseS3Path = $"{environment}/logs/{name}/";
            
            var worker = new BackgroundWorker();
            worker.DoWork += ProcessLogMessages;
            worker.RunWorkerAsync();
        }

        private void ProcessLogMessages(object sender, DoWorkEventArgs e)
        {            
            while (true)
            {
                using (var token = new CancellationTokenSource())
                {
                    token.CancelAfter(TimeSpan.FromSeconds(60));                    
                    List<ActivityLogData> activities = new List<ActivityLogData>();

                    try
                    {
                        while (true)
                        {
                            activities.Add(_queue.Take(token.Token));
                        }
                    }
                    catch (OperationCanceledException)
                    {
                    }
                    catch (Exception ex)
                    {
                        Log.Error(ex);
                    }
                    finally
                    {                        
                        //log to cloudwatch
                        activities.ForEach(x => _activityLogger.Info(x.Message));

                        //write csv file to S3
                        var pathname = GetTimestampFilepath(_name);
                        //WriteS3File(pathname, activities);

                        //write to rds
                        WriteFileToRds(pathname, activities);
                    }
                }
            }
        }

        private string GetTimestampFilepath(string name)
        {
            DateTime date = DateTime.UtcNow;
            string filename = name + date.ToString("_yyyyMMdd_HHmmss") + ".csv";
            string filepath = Path.Combine(_baseS3Path, filename);

            return filepath;
        }

        /* Disabled 2020-05-29 by Nathan in favor of RDS path, delete after 2020-06-30 if still in code.
        private void WriteS3File(string filepath, IList<ActivityLogData> activities)
        {
            try
            {                
                if (activities != null && activities.Count > 0)
                {
                    string data = null;

                    using (var writer = new StringWriter())
                    using (var csv = new CsvHelper.CsvWriter(writer))
                    {
                        csv.WriteRecords<ActivityLogData>(activities);

                        data = writer.GetStringBuilder().ToString();

                        S3Helper.WriteStringToS3(
                            data,
                            "resonance-activity-logs",
                            filepath,
                            false,
                            Constants.AppConfig.Application.KmsArn,
                            ServerSideEncryptionMethod.None
                            );
                    }
                }
            }
            catch (Exception e)
            {
                Log.Error(e);
            }
        }
        */

        private void WriteFileToRds(string pathname, List<ActivityLogData> activities)
        {
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText =
                        $@"
                            insert into {Constants.DatabaseSchema}microservice_twitch_atlas_activity_log
                            (
                                message_timestamp, message, controller, action, user, 
                                tracking_id, tracking_tag, tracking_date, hash
                            )
                            select @message_timestamp, @message, @controller, @action, @user, 
                            @tracking_id, @tracking_tag, @tracking_date, @hash
                        ;";
                        foreach (var activity in activities)
                        {
                            command.Parameters.Clear();
                            command.Parameters.AddWithValue("@message_timestamp", activity.Timestamp);
                            command.Parameters.AddWithValue("@message", activity.Message);
                            command.Parameters.AddWithValue("@controller", activity.Controller);
                            command.Parameters.AddWithValue("@action", activity.Action);
                            command.Parameters.AddWithValue("@user", activity.User);
                            command.Parameters.AddWithValue("@tracking_id", pathname);
                            command.Parameters.AddWithValue("@tracking_tag", Guid.NewGuid().ToString("d"));
                            command.Parameters.AddWithValue("@tracking_date", activity.Timestamp);
                            command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                            command.ExecuteNonQueryWithMeasurements("activity_log_service");
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
        }

        public void LogActivity(string msg, string controller, string action, string user)
        {
            if (msg == null)
            {
                throw new ArgumentNullException(nameof(msg));
            }

            _queue.Add(new ActivityLogData { Timestamp = DateTime.UtcNow, Message = msg, User = user, Controller = controller, Action = action });
        }

        public void LogActivity(ActivityLogData activity)
        {
            if (activity == null)
            {
                throw new ArgumentNullException(nameof(activity));
            }

            _queue.Add(activity);
        }
    }
}
