﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc.Infrastructure;
using Newtonsoft.Json;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Models.ConfigurationModels.Permissions;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Services.ActivityLoggerService;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Resonance.Core.Services.ColumnFilterService
{
	public class ColumnFilterService : IColumnFilterService
	{
		public ColumnFilterService()
		{

		}

		public IList<string> FilterOutRestrictedColumns(IList<string> permissions, IList<string> columns, string profileName, IActivityLoggerService activityLoggerService, IActionContextAccessor actionContextAccessor)
		{
			if (permissions == null)
			{
				throw new ArgumentNullException(nameof(permissions));
			}

			if (columns == null)
			{
				return null;
			}

			IList<string> removeColumnList = GetRemoveColumnList(permissions);
			IList<string> restrictedColumnList = GetRestrictedColumnList();
			if (restrictedColumnList.Any() && columns.Any(x => restrictedColumnList.Contains(x)))
			{
				string action = (string)actionContextAccessor.ActionContext.RouteData.Values["action"];
				string controller = (string)actionContextAccessor.ActionContext.RouteData.Values["controller"];
				activityLoggerService.LogActivity("Access Restricted Data", controller, action, profileName);
			}

			if (removeColumnList != null && removeColumnList.Any())
			{
				return columns.Except(removeColumnList).ToList();
			}
			else
			{
				return columns;
			}
		}

		public IList<string> GetRestrictedColumnList()
		{
			//return column name where the user does not have any of the required permissions for the column
			IList<string> removeColumnList
				= Constants.Permissions.ColumnPermissions
				.Select(c => c.ColumnName)
				.Distinct()
				.ToList()
				;
			return removeColumnList;
		}

		public IList<string> GetRemoveColumnList(IList<string> permissions)
		{

			//list of IDs of the users permissions
			IList<string> permissionIDs = Constants.Permissions.Permissions
				.Where(x => permissions.Contains(x.PermissionName))
				.Select(x => x.PermissionID)
				.ToList();

			//return column name where the user does not have any of the required permissions for the column
			IList<string> removeColumnList
				= Constants.Permissions.ColumnPermissions
					.Where(c =>
						!permissionIDs.Any(p => c.RequiredPermissionList.Contains(p))
					)

				.Select(c => c.ColumnName)
				.Distinct()
				.ToList()
				;
			return removeColumnList;
		}

        public void FilterOutRestrictedColumns(IList<string> permissions, ListingFilter filter, string profileName, IActivityLoggerService activityLoggerService, IActionContextAccessor actionContextAccessor)
        {
            if (permissions == null)
			{
				throw new ArgumentNullException(nameof(permissions));
			}

			if (filter == null)
			{
				return;
			}

			IList<string> removeColumnList = GetRemoveColumnList(permissions);
			IList<string> restrictedColumnList = GetRestrictedColumnList();
			if (restrictedColumnList.Any() && (filter.Columns.Any(x => restrictedColumnList.Contains(x)) || filter.QueryFilters.Any(x => restrictedColumnList.Contains(x.Key))))
			{
				string action = (string)actionContextAccessor.ActionContext.RouteData.Values["action"];
				string controller = (string)actionContextAccessor.ActionContext.RouteData.Values["controller"];
				activityLoggerService.LogActivity("Access Restricted Data", controller, action, profileName);
			}

			if (removeColumnList != null && removeColumnList.Any())
			{
				//filter out restricted columns
				filter.Columns = filter.Columns.Except(removeColumnList).ToArray();

				//remove any filter in filter.QueryFilters using restricted columns
				IEnumerable<QueryFilter> removeFilters = filter.QueryFilters.Where(f => removeColumnList.Contains(f.Key));
				if (removeFilters.Any())
				{
					filter.QueryFilters = filter.QueryFilters.Except(removeFilters);
				}
			}
		}
	}
}
