﻿using Amazon;
using Amazon.S3;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using System;
using System.Collections.Generic;
using System.Text;

namespace Resonance.Core.Services.CustomConfigService
{
	public class CustomConfigService : ICustomConfigService
	{
		private string _s3Bucket = "resonance-configuration";
		private string _baseS3Path;

		private string GetFullS3Path(string system, string name)
		{
			return $"{_baseS3Path}{system}/{name}.json.gz";
		}

        public CustomConfigService(string environment)
        {
            switch (environment.ToLower()) {
                case "development":
                case "staging":
                case "production":
                    break;

                default:
                    throw new ArgumentException("Invalid environment: " + environment, nameof(environment));
            }
            _baseS3Path = environment + "/" + "custom-configuration/";
        }

		public string GetConfig(string system, string name, bool load)
		{
			string json = null;

			string path = GetFullS3Path(system, name);
			using (var s3conn = new AmazonS3Client(RegionEndpoint.USWest2))
			{
				if (S3Helper.ExistsInS3(_s3Bucket, path) == true)
				{
					json = S3Helper.ReadTextFromS3(_s3Bucket, path);

					if (load)
					{
						json = Process(system, json);
					}
				}
				else
				{
					Log.Error("Unable to find config at path " + path);
				}
			}

			return json;
		}

		private string Process(string system, string data)
		{
			dynamic values = JsonConvert.DeserializeObject(data);

			foreach (var value in values)
			{
				try
				{
					string keyName = value.keyName;

					if (!string.IsNullOrEmpty(keyName))
					{
						value.keyValue = ProcessKeys(keyName, system);
					}
				}
				catch(Exception)
				{
				}
			}

			return values.ToString();
		}

		private JToken ProcessKeys(string keyName, string system)
		{
			string keyValue = null;
			var keyValuePath = $"{Constants.AppConfig.Application.Environment}/custom-configuration/{system}/{keyName}.json.gz";

			if (S3Helper.ExistsInS3(_s3Bucket, keyValuePath) == true)
			{
				keyValue = S3Helper.ReadTextFromS3(_s3Bucket, keyValuePath, true);
			}

			return keyValue != null ? JToken.Parse(keyValue) : null;
		}

		public DateTime? GetLastUpdated(string system, string name)
		{
			string path = GetFullS3Path(system, name);
			DateTime? lastUpdated = null;

			if (S3Helper.ExistsInS3(_s3Bucket, path) == true)
			{
				lastUpdated = S3Helper.LastUpdatedInS3(_s3Bucket, path);
			}

			return lastUpdated;
		}

		public void UploadConfigFile(string system, string name, string filepath)
		{
			string data = System.IO.File.ReadAllText(filepath);
			UploadConfigData(system, name, data);
		}

		public void UploadConfigData(string system, string name, string data)
		{
			string keypath = GetFullS3Path(system, name);
            string monitorKeypath = GetFullS3Path(system, "monitor");
            Log.Info($"Writing {_s3Bucket}/{keypath}");
			S3Helper.WriteStringToS3(data, _s3Bucket, keypath, true, Constants.AppConfig.Application.KmsArn, ServerSideEncryptionMethod.AWSKMS);

            //write to monitor file to indicate that system should be reloaded from front-end
            Log.Info($"Writing {_s3Bucket}/{monitorKeypath}");
            S3Helper.WriteStringToS3("", _s3Bucket, monitorKeypath, true, Constants.AppConfig.Application.KmsArn, ServerSideEncryptionMethod.AWSKMS);
		}
	}
}
