﻿using Amazon.CloudWatch;
using Microsoft.AspNetCore.Http;
using Novell.Directory.Ldap;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels.TwitchModels;
using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;

namespace Resonance.Core.Services.LdapService
{
    public class LdapService : ILdapService
    {
        /// <summary>
        /// TODO: Update this to use a2z domain as seen in the wiki article
        /// https://wiki.twitch.com/pages/viewpage.action?spaceKey=PS&title=Unified+LDAP+VPC+Endpoint
        /// TICKET: IMP-535
        /// </summary>
        private const string Host = "vpce-06b3493af8f32275d-rp1vl5qf.vpce-svc-02673a2089b1caf6d.us-west-2.vpce.amazonaws.com";
        private const string managerNamePattern = "^cn=(.*?),";
        private static Regex managerNameRegex = new Regex(managerNamePattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);

        private LdapConnection GetConnection(HttpContext context)
        {
            try
            {
                LdapConnection conn = new LdapConnection();
                int port = 636;
                conn.SecureSocketLayer = true;

                conn.Connect(Host, port);
                conn.Bind(null, null);

                return conn;
            }
            catch (Exception ex)
            {
                Log.Error(ex, "ldapservice_getconnection_error", context);
                CloudwatchHelper.EnqueueMetricRequest("ldapservice_getconnection_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public IEnumerable<UserAutocompleteEntry> GetUsersInGroup(string query, HttpContext context)
        {
            List<UserAutocompleteEntry> entries = new List<UserAutocompleteEntry>();

            try
            {
                LdapConnection conn = GetConnection(context);

                var searchBase = string.Empty;

                IEnumerable<string> filters = new[] { $"(&(objectClass=person)(memberof=cn={query},ou=Groups,dc=justin,dc=tv))" };
                HashSet<string> seenUserIDs = new HashSet<string>();

                foreach (string filter in filters)
                {
                    Log.Debug($@"Filter: '{filter}'");

                    var search = conn.Search(searchBase, LdapConnection.SCOPE_SUB, filter, new string[] { "cn", "uid", "mail", "manager" }, false);
                    while (search.HasMore())
                    {
                        var nextEntry = search.Next();
                        var attr = nextEntry.getAttribute("cn");
                        var userAttr = nextEntry.getAttribute("uid");
                        var mail = nextEntry?.getAttribute("mail")?.StringValue ?? "";
                        var managerAttr = nextEntry.getAttribute("manager");
                        var managerNameString = "";
                        if (!string.IsNullOrWhiteSpace(managerAttr?.StringValue))
                        {
                            try
                            {
                                if (managerNameRegex.IsMatch(managerAttr?.StringValue ?? ""))
                                {
                                    managerNameString = managerNameRegex.Match(managerAttr.StringValue).Groups[1].Captures[0].Value;
                                }
                            }
                            catch(Exception ex)
                            {
                                Log.Error(ex, "ldapservice_getusersingroups_error", context);
                                CloudwatchHelper.EnqueueMetricRequest("ldapservice_getusersingroups_error", 1, context, StandardUnit.Count);
                            }
                        }
                        if (userAttr != null && attr != null && !seenUserIDs.Contains(userAttr.StringValue))
                        {
                            entries.Add(new UserAutocompleteEntry()
                            {
                                FullName = attr.StringValue,
                                UserName = userAttr.StringValue,
                                Mail = mail,
                                Manager = managerNameString
                            });

                            seenUserIDs.Add(userAttr.StringValue);
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex, "ldapservice_getusersingroups_error", context);
                CloudwatchHelper.EnqueueMetricRequest("ldapservice_getusersingroups_error", 1, context, StandardUnit.Count);
            }
            return entries;
        }

        public IEnumerable<UserAutocompleteEntry> SearchUsers(string query, HttpContext context)
        {
            try
            {
                List<UserAutocompleteEntry> entries = new List<UserAutocompleteEntry>();
                LdapConnection conn = GetConnection(context);

                var searchBase = string.Empty;
                query = query
                    .Replace("\\", "\\5c")
                    .Replace("(", "\\28")
                    .Replace(")", "\\29")
                    .Replace("|", "\\7c")
                    .Replace("<", "\\3c")
                    .Replace("/", "\\2f")
                    .Replace("=", "\\3d")
                    .Replace("~", "\\7e")
                    .Replace("&", "\\26")
                    .Replace(">", "\\3e")
                    .Replace("*", "\\2a")
                    ;
                IEnumerable<string> filters = new[] { $"(&(objectClass=person)(cn={query}))", $"(&(objectClass=person)(sn={query}))" };
                HashSet<string> seenUserIDs = new HashSet<string>();

                foreach (string filter in filters)
                {
                    LdapSearchResults search = null;
                    try
                    {
                        search = conn.Search(searchBase, LdapConnection.SCOPE_SUB, filter, new string[] { "cn", "uid", "mail", "manager", "fn", "sn", "givenname" }, false);
                    }
                    catch (Exception ex)
                    {
                        Log.Error(ex, "ldapservice_searchusers_error", context);
                        CloudwatchHelper.EnqueueMetricRequest("ldapservice_searchusers_error", 1, context, StandardUnit.Count);
                    }
                    if (search != null)
                    {
                        while (search.HasMore())
                        {
                            var nextEntry = search.Next();
                            var firstname = nextEntry.getAttribute("givenname");
                            var lastname = nextEntry.getAttribute("sn");
                            var attr = nextEntry.getAttribute("cn");
                            var userAttr = nextEntry.getAttribute("uid");
                            var mail = nextEntry?.getAttribute("mail")?.StringValue ?? "";
                            var managerAttr = nextEntry.getAttribute("manager");
                            var nameString = $"{firstname.StringValue} {lastname.StringValue}";
                            var managerNameString = string.Empty;
                            if (userAttr != null && attr != null && !seenUserIDs.Contains(userAttr.StringValue))
                            {
                                try
                                {
                                    if (managerNameRegex.IsMatch(managerAttr?.StringValue ?? ""))
                                    {
                                        managerNameString = managerNameRegex.Match(managerAttr.StringValue).Groups[1].Captures[0].Value;
                                    }
                                }
                                catch (Exception ex)
                                {
                                    Log.Error(ex, "ldapservice_searchusers_error", context);
                                    CloudwatchHelper.EnqueueMetricRequest("ldapservice_searchusers_error", 1, context, StandardUnit.Count);
                                }

                                entries.Add(new UserAutocompleteEntry()
                                {
                                    FullName = nameString,
                                    UserName = userAttr.StringValue,
                                    Mail = mail,
                                    Manager = managerNameString
                                });
                                seenUserIDs.Add(userAttr.StringValue);
                            }
                        }
                    }
                }
                return entries;
            }
            catch(Exception ex)
            {
                Log.Error(ex, "ldapservice_searchusers_error", context);
                CloudwatchHelper.EnqueueMetricRequest("ldapservice_searchusers_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public string GetUserName(string user, HttpContext context)
        {
            try
            {
                user = user
                .Replace("\\", "\\5c")
                .Replace("(", "\\28")
                .Replace(")", "\\29")
                .Replace("|", "\\7c")
                .Replace("<", "\\3c")
                .Replace("/", "\\2f")
                .Replace("=", "\\3d")
                .Replace("~", "\\7e")
                .Replace("&", "\\26")
                .Replace(">", "\\3e")
                .Replace("*", "\\2a")
                ;
                LdapConnection conn = GetConnection(context);

                var searchBase = string.Empty;
                var filter = $"(&(objectClass=person)(uid={user}))";
                List<string> groups = new List<String>();

                var search = conn.Search(searchBase, LdapConnection.SCOPE_SUB, filter, new string[] { "cn" }, false);
                if (search.HasMore())
                {
                    var nextEntry = search.Next();
                    var attr = nextEntry.getAttribute("cn");
                    return attr.StringValue;
                }
                return "";
            }
            catch(Exception ex)
            {
                Log.Error(ex, "ldapservice_getusername_error", context);
                CloudwatchHelper.EnqueueMetricRequest("ldapservice_getusername_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public UserEntryWithGroups GetUserDataByName(string user, HttpContext context)
        {
            try
            {
                UserEntryWithGroups userData = null;

                user = user
                    .Replace("\\", "\\5c")
                    .Replace("(", "\\28")
                    .Replace(")", "\\29")
                    .Replace("|", "\\7c")
                    .Replace("<", "\\3c")
                    .Replace("/", "\\2f")
                    .Replace("=", "\\3d")
                    .Replace("~", "\\7e")
                    .Replace("&", "\\26")
                    .Replace(">", "\\3e")
                    .Replace("*", "\\2a")
                    ;
                LdapConnection conn = GetConnection(context);

                var searchBase = string.Empty;
                var filter = $"(&(objectClass=person)(uid={user}))";
                List<string> groups = new List<String>();

                var search = conn.Search(searchBase, LdapConnection.SCOPE_SUB, filter, new string[] { "memberof", "cn", "uid", "mail", "manager", "fn", "sn", "givenname" }, false);
                bool continueSearching = true;
                while (search.HasMore() && continueSearching)
                {
                    var nextEntry = search.Next();

                    var firstname = nextEntry.getAttribute("givenname");
                    var lastname = nextEntry.getAttribute("sn");
                    var cnattr = nextEntry.getAttribute("cn");
                    var userAttr = nextEntry.getAttribute("uid");
                    var mail = nextEntry?.getAttribute("mail")?.StringValue ?? "";
                    var managerAttr = nextEntry.getAttribute("manager");
                    var nameString = $"{firstname.StringValue} {lastname.StringValue}";
                    var managerNameString = string.Empty;
                    if (userAttr != null && cnattr != null)
                    {
                        try
                        {
                            if (managerNameRegex.IsMatch(managerAttr?.StringValue ?? ""))
                            {
                                managerNameString = managerNameRegex.Match(managerAttr.StringValue).Groups[1].Captures[0].Value;
                            }
                        }
                        catch (Exception ex)
                        {
                            Log.Error(ex);
                        }

                        if (userAttr.StringValue == user)
                        {
                            userData = new UserEntryWithGroups()
                            {
                                User = new UserAutocompleteEntry()
                                {
                                    FullName = nameString,
                                    UserName = userAttr.StringValue,
                                    Mail = mail,
                                    Manager = managerNameString
                                },
                                Groups = new List<string>()
                            };
                            var memberOfAttr = nextEntry.getAttribute("memberof").StringValueArray;
                            foreach (var dn in memberOfAttr)
                            {
                                string group = ParseGroupName(dn, context);
                                if (!string.IsNullOrEmpty(group))
                                {
                                    groups.Add(group);
                                }
                            }
                            if (groups.Count > 0)
                            {
                                userData.Groups.AddRange(groups);
                            }
                            continueSearching = false;
                        }
                    }
                }

                return userData;
            }
            catch(Exception ex)
            {
                Log.Error(ex, "ldapservice_getuserdatabyusername_error", context);
                CloudwatchHelper.EnqueueMetricRequest("ldapservice_getuserdatabyusername_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public string[] GetGroupsForUser(string user, HttpContext context)
        {
            try
            {
                user = user
                 .Replace("\\", "\\5c")
                 .Replace("(", "\\28")
                 .Replace(")", "\\29")
                 .Replace("|", "\\7c")
                 .Replace("<", "\\3c")
                 .Replace("/", "\\2f")
                 .Replace("=", "\\3d")
                 .Replace("~", "\\7e")
                 .Replace("&", "\\26")
                 .Replace(">", "\\3e")
                 .Replace("*", "\\2a")
                 ;
                LdapConnection conn = GetConnection(context);

                var searchBase = string.Empty;
                var filter = "(&(objectClass=person)(uid=" + user + "))";
                List<string> groups = new List<String>();

                var search = conn.Search(searchBase, LdapConnection.SCOPE_SUB, filter, new string[] { "memberof", "cn" }, false);
                while (search.HasMore())
                {
                    var nextEntry = search.Next();
                    var attr = nextEntry.getAttribute("memberof").StringValueArray;

                    foreach (var dn in attr)
                    {
                        string group = ParseGroupName(dn, context);
                        if (!string.IsNullOrEmpty(group))
                        {
                            groups.Add(group);
                        }
                    }
                }

                return groups.ToArray();
            }
            catch(Exception ex)
            {
                Log.Error(ex, "ldapservice_getgroupsforuser_error", context);
                CloudwatchHelper.EnqueueMetricRequest("ldapservice_getgroupsforuser_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        private string ParseGroupName(string dn, HttpContext context)
        {
            try
            {
                var result = ParseDn(dn, context);
                return result.ContainsKey("cn") ? result["cn"] : null;
            }
            catch (Exception ex)
            {
                Log.Error(ex, "ldapservice_parsegroupname_error", context);
                CloudwatchHelper.EnqueueMetricRequest("ldapservice_parsegroupname_error", 1, context, StandardUnit.Count);

                throw;
            }
        }
        
        private Dictionary<string, string> ParseDn(string dn, HttpContext context)
        {
            try
            {
                Dictionary<string, string> dict = new Dictionary<string, string>();

                var parts = dn.Split(",");

                foreach (string item in parts)
                {
                    var temp = item.Split("=");
                    if (!dict.ContainsKey(temp[0]))
                    {
                        dict.Add(temp[0], temp[1]);
                    }
                }

                return dict;
            }
            catch(Exception ex)
            {
                Log.Error(ex, "ldapservice_parsedn_error", context);
                CloudwatchHelper.EnqueueMetricRequest("ldapservice_parsedn_error", 1, context, StandardUnit.Count);

                throw;
            }
        }
    }
}
