﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading;
using Microsoft.AspNetCore.Http;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Services.LdapService;

namespace Resonance.Core.Services.UserGroupSyncerService
{
    public class LdapGroupSyncerService : IUserGroupSyncerService
    {
        private BlockingCollection<string> _queue = new BlockingCollection<string>();
        private IDictionary<string, IList<string>> _userGroups = new Dictionary<string, IList<string>>();
        private IList<string> _usernames = new List<string>();
        private ILdapService _ldapService;
        private BackgroundWorker _worker;

        public IDictionary<string, IList<string>> UserList
        {
            get
            {
                return _userGroups;
            }
        }

        public IList<string> GetGroupsByUsername(string username)
        {            
            lock (_userGroups)
            {                
                if (_userGroups.ContainsKey(username))
                {
                    return _userGroups[username];
                }
            }

            return null;
        }

        public LdapGroupSyncerService(ILdapService ldapService)
        {
            _ldapService = ldapService;
            ProcessNewUsersAsync();
        }

        public BackgroundWorker ProcessNewUsersAsync()
        {
            _worker = new BackgroundWorker();
            _worker.DoWork += ProcessNewUsers;
            _worker.RunWorkerAsync();
            _worker.WorkerSupportsCancellation = true;        

            return _worker;
        }

        private void ProcessNewUsers(object sender, DoWorkEventArgs e)
        {            
            while (!((BackgroundWorker)sender).CancellationPending)
            {
                using (var token = new CancellationTokenSource())
                {                    
                    token.CancelAfter(TimeSpan.FromSeconds(60));                    

                    try
                    {
                        while (true)
                        {
                            _usernames.Add(_queue.Take(token.Token));
                        }
                    }
                    catch (OperationCanceledException)
                    {                        
                    }
                    catch (Exception ex)
                    {
                        Log.Error(ex);
                    }
                    finally
                    {
                        UpdateLdapGroups(null);
                    }
                }
            }
        }

        private void UpdateLdapGroups(HttpContext context)
        {
            var tempUserGroups = new Dictionary<string, IList<string>>();

            _usernames = _usernames.Distinct().ToList();

            foreach (string username in _usernames)
            {
                tempUserGroups[username] = _ldapService.GetGroupsForUser(username, context);
            }            

            lock (_userGroups)
            {
                _userGroups = tempUserGroups;
            }
        }

        public void AddUserToQueue(string username)
        {
            _queue.Add(username);
        }
    }
}
