﻿using System;
using System.Diagnostics;
using System.Linq;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using Resonance.Core;
using Resonance.Core.Attributes;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;

namespace Resonance.Api.Microservices.Controllers
{
    [Route("permission")]
    public class PermissionController : Controller
    {
        private static DateTime lastRefresh = DateTime.UtcNow;
        [Obsolete]
        private static string placeholderPermission = "Amp";

        private readonly IConfiguration config;
        private static PermissionHelper permissionHelper { get; set; }

        static PermissionController()
        {
            permissionHelper = new PermissionHelper();
            permissionHelper.Initialize();
        }

        public PermissionController(IConfiguration _config)
        {
            config = _config;
        }

        [HttpPost("get-user-roles")]
        public JsonResult GetUserRoles()
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var response = new ApiListResponse<string>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetUserRoles(HttpContext);
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            stopwatch.Stop();
            Log.Verbose($@"GetUserRoles took {stopwatch.ElapsedMilliseconds}ms", context: HttpContext);
            return new JsonResult(response);
        }

        [HttpPost("get-user-permissions")]
        public JsonResult GetUserPermissions()
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var response = new ApiListResponse<string>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetUserPermissions(HttpContext);
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            stopwatch.Stop();
            Log.Verbose($@"GetUserPermissions took {stopwatch.ElapsedMilliseconds}ms", context: HttpContext);

            return new JsonResult(response);
        }

        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpGet("get-user-blocked-permissions")]
        public JsonResult GetUserBlockedPermissions()
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var response = new ApiListResponse<string>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                var userPermissions = permissionHelper.GetUserPermissions(HttpContext).ToArray();
                response.ResponseData = ConstantsPermissions.Amp.CheckPublicPermissions.Except(userPermissions);
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            stopwatch.Stop();
            Log.Verbose($@"GetUserBlockedPermissions took {stopwatch.ElapsedMilliseconds}ms", context: HttpContext);

            return new JsonResult(response);
        }

        [HttpGet("update-permission-roles")]
        public JsonResult UpdateRoles()
        {
           
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var response = new ApiListResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };
            try
            {
                var forcewait = DateTime.UtcNow.AddMinutes(-1);
                if(lastRefresh <= forcewait)
                {
                    Log.Info($@"Refreshing permissions");
                    lastRefresh = DateTime.UtcNow;
                    Constants.Permissions = permissionHelper.GetPermissionsFromS3(Constants.AppConfig.Application.Name);
                    response.Success = true;
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            stopwatch.Stop();
            Log.Verbose($@"UpdateRoles took {stopwatch.ElapsedMilliseconds}ms", context: HttpContext);

            return new JsonResult(response);
        }
    }
}