﻿using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.HealthModels;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Jobs.Amp.Maintenance;
using Resonance.Jobs.Amp.TwitchUser;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Resonance.Core.Helpers.StatsDHelpers;
using Resonance.Jobs.Amp.Request;
using Resonance.Core.Helpers.AwsHelpers;
using Amazon.CloudWatch;

namespace Resonance.Jobs.Amp
{
    public static class JobManager
    {
        private static bool monitorRunning = false;
        private static Dictionary<string, JobBase> jobs = new Dictionary<string, JobBase>();
        private static Dictionary<string, BackgroundWorker> jobQueue = new Dictionary<string, BackgroundWorker>();

        public static void Initialize()
        {
            PopulateJobs();

            var monitor = new BackgroundWorker();
            monitor.DoWork += MonitorJobs;
            monitor.RunWorkerAsync();
        }

        private static void PopulateJobs()
        {
            Log.Info($@"Populating Jobs");
            var twitchUserListingJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "twitch-user-listing");
            if (twitchUserListingJob != null)
            {
                try
                {
                    Log.Info($@"Adding twitch-user-listing job");
                    jobs.Add
                    (
                        "twitch-user-listing",
                        new TwitchUser.UserListingJob(new JobConfiguration()
                        {
                            IsActive = twitchUserListingJob.IsActive,
                            IsRunning = false,
                            LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{twitchUserListingJob.LogPath}",
                            JobName = twitchUserListingJob.JobName,
                            StatsDName = twitchUserListingJob.StatsDName,
                            NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                            S3ConfigurationBucket = $@"{twitchUserListingJob.S3ConfigurationBucket}",
                            S3ConfigurationPath = $@"{twitchUserListingJob.S3ConfigurationPath}twitch-user-listing/{Constants.ApplicationVersion}.json.gz"
                        })
                    );
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }

            var twitchGameJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "twitch-games");
            if (twitchGameJob != null)
            {
                try
                {
                    Log.Info($@"Adding twitch-games job");
                    jobs.Add
                    (
                        "twitch-games",
                        new TwitchGamesJob(new JobConfiguration()
                        {
                            IsActive = twitchGameJob.IsActive,
                            IsRunning = false,
                            LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{twitchGameJob.LogPath}",
                            JobName = twitchGameJob.JobName,
                            StatsDName = twitchGameJob.StatsDName,
                            NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                            S3ConfigurationBucket = $@"{twitchGameJob.S3ConfigurationBucket}",
                            S3ConfigurationPath = $@"{twitchGameJob.S3ConfigurationPath}twitch-games/{Constants.ApplicationVersion}.json.gz"
                        })
                    );
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }

            var salesforceSyncJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "salesforce-sync");
            if (salesforceSyncJob != null)
            {
                try
                {
                    Log.Info($@"Adding salesforce-sync job");
                    jobs.Add
                    (
                        "salesforce-sync",
                        new SyncSalesforceRequestChannelJob(new JobConfiguration()
                        {
                            IsActive = salesforceSyncJob.IsActive,
                            IsRunning = false,
                            LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{salesforceSyncJob.LogPath}",
                            JobName = salesforceSyncJob.JobName,
                            StatsDName = salesforceSyncJob.StatsDName,
                            NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                            S3ConfigurationBucket = $@"{salesforceSyncJob.S3ConfigurationBucket}",
                            S3ConfigurationPath = $@"{salesforceSyncJob.S3ConfigurationPath}salesforce-sync/{Constants.ApplicationVersion}.json.gz"
                        })
                    );
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }

            var userDetailsJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "user-details");
            if (userDetailsJob != null)
            {
                try
                {
                    Log.Info($@"Adding user-details job");
                    jobs.Add
                    (
                        "user-details",
                        new UserDetailsJob(new JobConfiguration()
                        {
                            IsActive = userDetailsJob.IsActive,
                            IsRunning = false,
                            LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{userDetailsJob.LogPath}",
                            JobName = userDetailsJob.JobName,
                            StatsDName = userDetailsJob.StatsDName,
                            NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                            S3ConfigurationBucket = $@"{userDetailsJob.S3ConfigurationBucket}",
                            S3ConfigurationPath = $@"{userDetailsJob.S3ConfigurationPath}user-details/{Constants.ApplicationVersion}.json.gz"
                        })
                    );
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }

            var twitchUserCardsJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "twitch-user-cards");
            if (twitchUserCardsJob != null)
            {
                Log.Info($@"Adding twitch-user-cards job");
                jobs.Add
                (
                    "twitch-user-cards",
                    new TwitchUser.UserCardJob(new JobConfiguration()
                    {
                        IsActive = twitchUserCardsJob.IsActive,
                        IsRunning = false,
                        LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{twitchUserCardsJob.LogPath}",
                        JobName = twitchUserCardsJob.JobName,
                        StatsDName = twitchUserCardsJob.StatsDName,
                        NextRunTime = DateTime.UtcNow.AddMinutes(60),
                        S3ConfigurationBucket = $@"{twitchUserCardsJob.S3ConfigurationBucket}",
                        S3ConfigurationPath = $@"{twitchUserCardsJob.S3ConfigurationPath}twitch-user-cards/{Constants.ApplicationVersion}.json.gz"
                    })
                );

            }

            var dataQualityJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "data-quality");
            if (dataQualityJob != null)
            {
                Log.Info($@"Adding data-quality job");
                jobs.Add
                (
                    "data-quality",
                    new DataQuality(new JobConfiguration()
                    {
                        IsActive = dataQualityJob.IsActive,
                        IsRunning = false,
                        LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{dataQualityJob.LogPath}",
                        JobName = dataQualityJob.JobName,
                        StatsDName = dataQualityJob.StatsDName,
                        NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                        S3ConfigurationBucket = $@"{dataQualityJob.S3ConfigurationBucket}",
                        S3ConfigurationPath = $@"{dataQualityJob.S3ConfigurationPath}data-quality/{Constants.ApplicationVersion}.json.gz"
                    })
                );
            }

            var scorecardJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "user-scorecard");
            if (scorecardJob != null)
            {
                Log.Info($@"Adding scorecard job");
                jobs.Add
                (
                    "user-scorecard",
                    new UserScorecardJob(new JobConfiguration()
                    {
                        IsActive = scorecardJob.IsActive,
                        IsRunning = false,
                        LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{scorecardJob.LogPath}",
                        JobName = scorecardJob.JobName,
                        StatsDName = scorecardJob.StatsDName,
                        NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                        S3ConfigurationBucket = $@"{scorecardJob.S3ConfigurationBucket}",
                        S3ConfigurationPath = $@"{scorecardJob.S3ConfigurationPath}user-scorecard/{Constants.ApplicationVersion}.json.gz"
                    })
                );
            }

            var calendarJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "user-calendar");
            if (calendarJob != null)
            {
                Log.Info($@"Adding calendar job");
                jobs.Add
                (
                    "user-calendar",
                    new UserCalendarJob(new JobConfiguration()
                    {
                        IsActive = calendarJob.IsActive,
                        IsRunning = false,
                        LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{calendarJob.LogPath}",
                        JobName = calendarJob.JobName,
                        StatsDName = calendarJob.StatsDName,
                        NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                        S3ConfigurationBucket = $@"{calendarJob.S3ConfigurationBucket}",
                        S3ConfigurationPath = $@"{calendarJob.S3ConfigurationPath}user-calendar/{Constants.ApplicationVersion}.json.gz"
                    })
                );
            }

            var weeklySummaryJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "user-weekly-summary");
            if (weeklySummaryJob != null)
            {
                Log.Info($@"Adding user weekly summary job");
                jobs.Add
                (
                    "user-weekly-summary",
                    new UserWeeklySummary(new JobConfiguration()
                    {
                        IsActive = weeklySummaryJob.IsActive,
                        IsRunning = false,
                        LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{weeklySummaryJob.LogPath}",
                        JobName = weeklySummaryJob.JobName,
                        StatsDName = weeklySummaryJob.StatsDName,
                        NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                        S3ConfigurationBucket = $@"{weeklySummaryJob.S3ConfigurationBucket}",
                        S3ConfigurationPath = $@"{weeklySummaryJob.S3ConfigurationPath}user-weekly-summary/{Constants.ApplicationVersion}.json.gz"
                    })
                );
            }

            var tableTrackingJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "table-tracking");
            if (tableTrackingJob != null)
            {
                Log.Info($@"Adding table tracking job");
                jobs.Add
                (
                    "table-tracking",
                    new TableTracking(new JobConfiguration()
                    {
                        IsActive = tableTrackingJob.IsActive,
                        IsRunning = false,
                        LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{tableTrackingJob.LogPath}",
                        JobName = tableTrackingJob.JobName,
                        StatsDName = tableTrackingJob.StatsDName,
                        NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                        S3ConfigurationBucket = $@"{tableTrackingJob.S3ConfigurationBucket}",
                        S3ConfigurationPath = $@"{tableTrackingJob.S3ConfigurationPath}table-tracking/{Constants.ApplicationVersion}.json.gz"
                    })
                );
            }
        }

        private static void MonitorJobs(object sender, DoWorkEventArgs e)
        {
            if (monitorRunning)
            {
                return;
            }
            monitorRunning = true;

            while (!CommandManager.quit)
            {
                Parallel.ForEach(jobs, new ParallelOptions() { MaxDegreeOfParallelism = 4 }, job =>
                {
                    try
                    {
                        if (job.Value.Config.IsActive && !job.Value.Config.IsRunning && (job.Value.Config.NextRunTime == null || (job.Value.Config.NextRunTime != null && job.Value.Config.NextRunTime <= DateTime.UtcNow)))
                        {
                            Log.Info($@"Running {job.Key}");
                            CloudwatchHelper.EnqueueMetricRequest($"{job.Value.Config.JobName}", 1, null, StandardUnit.Count);
                            StatsDHelper.Counter(measurement: $"jobstatus_start", measurementType: "start", val: 1, location: job.Key);
                            job.Value.Run();
                            CloudwatchHelper.EnqueueMetricRequest($"{job.Value.Config.JobName}", 1, null, StandardUnit.Count);
                            StatsDHelper.Counter(measurement: $"jobstatus_stop", measurementType: "stop", val: 1, location: job.Key);
                        }
                    }
                    catch (Exception ex)
                    {
                        Log.Error(ex);
                    }
                });
                Thread.Sleep(5000);
            }
            Log.Info($@"Quit command detected by the job manager. Waiting for all jobs to complete.");
            while (jobs.Any(x => x.Value.Config.IsRunning))
            {
                Thread.Sleep(5000);
            }
            Log.Info($@"All jobs complete. Exiting.");
            Program.GracefulQuitHandle.Set();
        }

        private static void ActivateJob(ref JobBase job)
        {
            if (!job.Config.IsActive)
            {
                Log.Info($@"{job.Config.JobName} is not active. Skipping activation process.");
                return;
            }
        }
    }
}