﻿using Amazon.CloudWatch;
using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StatsDHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.HealthModels;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Resonance.Jobs.Amp.Maintenance
{
    /// <summary>
    /// Expected Metrics:
    /// Max age of data: Warn after 2 days, critical after 3
    /// Lower population: No warn, Critical if less than 5k
    /// Website availability: Warn after 3 failures, critical after 5
    /// Duplicates Found Tolerance: Warn after 1, Critical after 2
    /// </summary>
    public class DataQuality : JobBase, IJob<long>
    {
        private static Dictionary<string, bool> runningItems { get; set; }
        private static DataQualityConfiguration jobConfig { get; set; } = null;
        private static int totalFindings { get; set; } = 0;
        private static object findingLock = new object();

        public DataQuality(JobConfiguration _config)
        {
            Config = _config;
            jobConfig = new DataQualityConfiguration();
            Log.Info($@"DataQuality configured. IsActive: {Config.IsActive}");
        }

        public override void Run()
        {
            try
            {
                totalFindings = 0;
                Log.Info($"DataRecencyJob Running");
                this.Config.IsRunning = true;

                var waitTasks = new Dictionary<string, Task>();
                var apdcsRecency = CheckAPDCSRecency();
                waitTasks.Add("CheckAPDCSRecency", apdcsRecency);
                var mysqlTwitchUserListingRecency = CheckMysqlTwitchUserListingRecency();
                waitTasks.Add("CheckMysqlTwitchUserListingRecency", mysqlTwitchUserListingRecency);
                var etlRecency = CheckMysqlETLRecency();
                waitTasks.Add("CheckMysqlETLRecency", etlRecency);
                Task.WaitAll(waitTasks.Values.ToArray(), jobConfig.TaskCompletionTimeout);
                foreach (var task in waitTasks)
                {
                    if (task.Value.IsFaulted)
                    {
                        TaskFail(task.Value, task.Key);
                    }
                    else if (task.Value.IsCompleted)
                    {
                        TaskComplete(task.Value, task.Key);
                    }
                }
            }
            catch (TaskCanceledException ex)
            {
                CloudwatchHelper.EnqueueMetricRequest("data_quality_error", 1, null, StandardUnit.Count);
                StatsDHelper.Counter(measurement: $"dataquality", measurementType: "Cancel", val: 1, location: "exception");
                Log.Error($"DataRecencyJob Cancelled: {ex}");
            }
            catch (OperationCanceledException ex)
            {
                CloudwatchHelper.EnqueueMetricRequest("data_quality_error", 1, null, StandardUnit.Count);
                StatsDHelper.Counter(measurement: $"dataquality", measurementType: "OperationCanceled", val: 1, location: "exception");
                Log.Error($"DataRecencyJob OperationvCanceled: {ex}");
            }
            catch (TimeoutException ex)
            {
                CloudwatchHelper.EnqueueMetricRequest("data_quality_error", 1, null, StandardUnit.Count);
                StatsDHelper.Counter(measurement: $"dataquality", measurementType: "Timeout", val: 1, location: "exception");
                Log.Error($"DataRecencyJob Timeout: {ex}");
            }
            catch (AggregateException ex)
            {
                CloudwatchHelper.EnqueueMetricRequest("data_quality_error", 1, null, StandardUnit.Count);
                StatsDHelper.Counter(measurement: $"dataquality", measurementType: "AggregateException", val: 1, location: "exception");
                Log.Error($"DataRecencyJob: Aggregate Exception: {ex}");
            }
            catch (Exception ex)
            {
                CloudwatchHelper.EnqueueMetricRequest("data_quality_error", 1, null, StandardUnit.Count);
                Log.Error(ex);
            }
            finally
            {
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromDays(1));
                Log.Info($"DataRecencyJob Complete");
                try
                {
                    using (var conn = DBManagerMysql.GetConnection(true))
                    {
                        using
                        (
                            var command =
                                AmpQuerySql.InsertUpdateEtlTracking
                                (
                                    source: "Redshift_Microservice_Twitch_Data_Quality",
                                    target: $"Microservice_Twitch_Data_Quality",
                                    method: "Jobs.TwitchDataQuality",
                                    timestamp: DateTime.UtcNow,
                                    rowcount: totalFindings
                                )
                        )
                        {
                            command.Connection = conn;
                            command.CommandTimeout = 0;
                            command.ExecuteNonQueryWithMeasurements("data_quality_job");
                        }
                    }
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
                totalFindings = 0;
            }
        }

        private void IncrementFindings()
        {
            lock (findingLock)
            {
                totalFindings++;
            }
        }

        private void TaskComplete(Task task, string name)
        {
            try
            {
                if (task.IsCompletedSuccessfully)
                {
                    CloudwatchHelper.EnqueueMetricRequest("data_quality_success", 1, null, StandardUnit.Count);
                    StatsDHelper.Counter(measurement: $"dataquality", measurementType: "TaskComplete", val: 1, location: "Status");
                    Log.Info($"DataQualityTask {name} Completed Successfully");

                }
                else
                {
                    CloudwatchHelper.EnqueueMetricRequest("data_quality_error", 1, null, StandardUnit.Count);
                    StatsDHelper.Counter(measurement: $"dataquality", measurementType: "TaskFail", val: 1, location: "exception");
                    Log.Error($"DataQualityTask Task {name} Failed To Complete Successfully");
                    IncrementFindings();
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                IncrementFindings();
            }
        }

        private void TaskFail(Task task, string name)
        {
            try
            {
                StatsDHelper.Counter(measurement: $"dataquality", measurementType: "TaskFail", val: 1, location: "Status");
                Log.Error($"DataQualityTask TaskFail: {task.Id}:{name}:{task.Status}:{task.CreationOptions}:{task.Exception}");
                IncrementFindings();
            }
            catch (Exception ex)
            {

                StatsDHelper.Counter(measurement: $"dataquality", measurementType: "TaskFail", val: 1, location: "exception");
                Log.Error(ex);
                IncrementFindings();
            }
        }

        private async Task<bool?> CheckAPDCSRecency()
        {
            Log.Info($"Starting CheckAPDCSRecency");
            var exceptionCount = 0;
            bool? success = null;
            try
            {
                using(var conn = DBManagerRedshift.TahoeConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 0;
                        command.CommandText = $@"select cast(datediff(day, max(isnull(day, cast('1900-01-01' as datetime))), getdate()) as int) as apdcs_age from cubes.affiliates_partners_daily_channel_summary;";
                        var age = (int)command.ExecuteScalarWithMeasurements<int>("apdcs_recency");
                        CloudwatchHelper.EnqueueMetricRequest("data_quality_success", age, null, StandardUnit.Count);
                        StatsDHelper.Counter(measurement: $"dataquality", measurementType: "mysql_data_age", val: age, location: "apdcs_age");
                        if(age > jobConfig.MaxAgeOfData)
                        {
                            IncrementFindings();
                        }
                    }
                }
                success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                exceptionCount++;
                IncrementFindings();
            }
            CloudwatchHelper.EnqueueMetricRequest("data_quality_error", exceptionCount, null, StandardUnit.Count);
            StatsDHelper.Counter(measurement: $"dataquality", measurementType: "mysql_data_age", val: exceptionCount, location: "exception");
            Log.Info($"Completed CheckAPDCSRecency");
            await Task.FromResult(0);
            return success;
        }

        private async Task<bool?> CheckMysqlTwitchUserListingRecency()
        {
            Log.Info($"Starting CheckMysqlTwitchUserListingRecency");
            bool? success = null;

            var exceptionCount = 0;

            try
            {
                foreach (var aggregationType in jobConfig.RequiredAggregations)
                {
                    using (var conn = DBManagerMysql.GetConnection(true))
                    {
                        using (var command = conn.GetCommand())
                        {
                            command.CommandTimeout = 0;
                            command.CommandText = $@"select datediff(now(), coalesce(max(maxdate), cast('1900-01-01' as datetime))) as tul_age from {Constants.DatabaseSchema}microservice_twitch_user_listing_past_{(int)aggregationType}_days;";
                            var age = (int)(long)command.ExecuteScalarWithMeasurements<long>("get_tul_age");
                            CloudwatchHelper.EnqueueMetricRequest("data_quality_true", age, null, StandardUnit.Count);
                            StatsDHelper.Counter(measurement: $"dataquality", measurementType: "mysql_data_age", val: age, location: "mysql_tul_age");
                            if(age >= jobConfig.MaxAgeOfData)
                            {
                                IncrementFindings();
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                exceptionCount++;
                IncrementFindings();
            }
            CloudwatchHelper.EnqueueMetricRequest("data_quality_erro", exceptionCount, null, StandardUnit.Count);
            StatsDHelper.Counter(measurement: $"dataquality", measurementType: "mysql_data_age", val: exceptionCount, location: "exception");
            Log.Info($"Completed CheckMysqlTwitchUserListingRecency");
            return success;
        }

        private async Task<bool?> CheckMysqlETLRecency()
        {
            Log.Info($"Starting CheckMysqlETLRecency");
            bool? success = null;

            var exceptionCount = 0;
            try
            {
                foreach (var aggregationType in jobConfig.RequiredAggregations)
                {
                    using (var conn = DBManagerMysql.GetConnection(true))
                    {
                        using (var command = conn.GetCommand())
                        {
                            command.CommandTimeout = 0;
                            command.CommandText = $@"select source, target, method, datediff(now(), LastUpdateDate) as job_etl_age, RowCount from {Constants.DatabaseSchema}microservice_etl_tracking;";
                            Log.Info($"{command.CommandText}");
                            using (var reader = await command.ExecuteReaderAsync())
                            {
                                if (reader.HasRows)
                                {
                                    while (await reader.ReadAsync())
                                    {
                                        var source = reader.GetString(0);
                                        var target = reader.GetString(1);
                                        var method = reader.GetString(2);
                                        var etlAge = reader.GetInt32(3);
                                        var rowCount = reader.GetInt64(4);
                                        CloudwatchHelper.EnqueueMetricRequest("data_quality_success", etlAge, null, StandardUnit.Count);
                                        CloudwatchHelper.EnqueueMetricRequest("data_quality_success_rowcount", rowCount, null, StandardUnit.Count);
                                        StatsDHelper.Counter(measurement: $"dataquality", measurementType: "etl_age", val: etlAge, location: "mysql_etl_age", additionalTags: $",source={source},target={target},method={method}");
                                        StatsDHelper.CounterLong(measurement: $"dataquality", measurementType: "etl_population", val: rowCount, location: "mysql_etl_population", additionalTags: $",source={source},target={target},method={method}");
                                        if(etlAge > jobConfig.MaxAgeOfData)
                                        {
                                            IncrementFindings();
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                exceptionCount++;
                IncrementFindings();
            }
            CloudwatchHelper.EnqueueMetricRequest("data_quality_error", exceptionCount, null, StandardUnit.Count);
            StatsDHelper.Counter(measurement: $"dataquality", measurementType: "etl_age", val: exceptionCount, location: "exception");
            Log.Info($"Completed CheckMysqlETLRecency");
            return success;
        }
    }
}
