﻿using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StatsDHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.HealthModels;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Models.DatabaseModels.MysqlModels;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Resonance.Jobs.Amp.Maintenance
{
    /// <summary>
    /// Expected Metrics:
    /// Max age of data: Warn after 2 days, critical after 3
    /// Lower population: No warn, Critical if less than 5k
    /// Website availability: Warn after 3 failures, critical after 5
    /// Duplicates Found Tolerance: Warn after 1, Critical after 2
    /// </summary>
    public class TableTracking : JobBase, IJob<long>
    {
        private static TableTrackingConfiguration jobConfig { get; set; } = null;

        public TableTracking(JobConfiguration _config)
        {
            Config = _config;
            jobConfig = new TableTrackingConfiguration();
            Log.Info($@"TableTracking configured. IsActive: {Config.IsActive}");
        }

        public override void Run()
        {
            try
            {
                var items = new Dictionary<string, TableTrackingMetaDataModel>();
                var itemColumns = new Dictionary<string, List<TableTrackingColumnModel>>();

                Log.Info($"TableTrackingJob Running");
                this.Config.IsRunning = true;
                WipeOldData();
                GetTableTrackingColumns(ref items, ref itemColumns);
                GetProcedures(ref items, ref itemColumns);
                InsertUpdateTableTrackingData(ref items, ref itemColumns);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromDays(1));
                Log.Info($"TableTrackingJob Complete");
                try
                {
                    using (var conn = DBManagerMysql.GetConnection(true))
                    {
                        using
                        (
                            var command =
                                AmpQuerySql.InsertUpdateEtlTracking
                                (
                                    source: "Table_Tracking_Job",
                                    target: $"maintenance_table_column_tracking",
                                    method: "Jobs.TableTracking",
                                    timestamp: DateTime.UtcNow,
                                    rowcount: 0
                                )
                        )
                        {
                            command.Connection = conn;
                            command.CommandTimeout = 0;
                            command.ExecuteNonQueryWithMeasurements("table_tracking_job");
                        }
                    }
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }
        }

        private void WipeOldData()
        {
            Log.Info($"TableTracking: Wiping olddata from staging table");
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText =
                    $@"
                        truncate table {Constants.DatabaseSchema}maintenance_staging_table_tracking;
                        truncate table {Constants.DatabaseSchema}maintenance_staging_table_tracking_definition;
                        truncate table {Constants.DatabaseSchema}maintenance_staging_table_column_tracking;
                        truncate table {Constants.DatabaseSchema}maintenance_staging_table_audit;
                    ";
                    command.ExecuteNonQueryWithMeasurements("table_tracking_job_cleanup");
                }
            }
            Log.Info($"TableTracking: Completed wiping old data from staging table");
        }

        private void InsertUpdateTableTrackingData(ref Dictionary<string, TableTrackingMetaDataModel> items, ref Dictionary<string, List<TableTrackingColumnModel>> itemColumns)
        {
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText = $@"insert into {Constants.DatabaseSchema}maintenance_staging_table_tracking (`Type`, `Schema`, `Name`, `Sql`) values (@type, @schema, @name, @sql);";
                        foreach (var table in items)
                        {
                            try
                            {
                                command.Parameters.Clear();
                                command.Parameters.AddWithValue("@type", table.Value.Type);
                                command.Parameters.AddWithValue("@schema", table.Value.Schema);
                                command.Parameters.AddWithValue("@name", table.Value.Name);
                                command.Parameters.AddWithValue("@sql", table.Value.Sql);
                                command.ExecuteNonQueryWithMeasurements("table_tracking_job");
                            }
                            catch (Exception ex)
                            {
                                Log.Error($@"TableTracking: {ex}");
                            }
                        }
                        command.Parameters.Clear();

                        command.CommandText = GetInsertUpdateSqlForTableTracking();

                        command.ExecuteNonQueryWithMeasurements("table_tracking_job");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"TableTracking: {ex}");
            }

            try
            {
                foreach (var tableColumn in itemColumns)
                {
                    try
                    {

                    }
                    catch (Exception ex)
                    {
                        Log.Error($@"TableTracking: {ex}");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"TableTracking: {ex}");
            }
        }

        private string GetInsertUpdateSqlForTableTracking()
        {
            var sql =
            $@"
                insert into {Constants.DatabaseSchema}maintenance_staging_table_audit (`Type`, `Schema`, `Name`, `AuditType`, `AuditDate`, `AuditInfo`)
                select a.`Type`, a.`Schema`, a.`Name`, 'Old' as AuditType, UTC_DATE() as `AuditDate`, a.`Sql` as `AuditInfo`
                from {Constants.DatabaseSchema}maintenance_table_tracking as a
                inner join {Constants.DatabaseSchema}maintenance_staging_table_tracking as b
	                on a.`Type` = b.`Type`
	                and a.`Schema` = b.`Schema`
	                and a.`Name` = b.`Name`
                where a.`Sql` <> b.`Sql`
                ;

                insert into {Constants.DatabaseSchema}maintenance_staging_table_audit (`Type`, `Schema`, `Name`, `AuditType`, `AuditDate`, `AuditInfo`)
                select a.`Type`, a.`Schema`, a.`Name`, 'New' as AuditType, UTC_DATE() as `AuditDate`, b.`Sql` as `AuditInfo`
                from {Constants.DatabaseSchema}maintenance_table_tracking as a
                inner join {Constants.DatabaseSchema}maintenance_staging_table_tracking as b
	                on a.`Type` = b.`Type`
	                and a.`Schema` = b.`Schema`
	                and a.`Name` = b.`Name`
                where a.`Sql` <> b.`Sql`
                ;

                insert into {Constants.DatabaseSchema}maintenance_staging_table_audit (`Type`, `Schema`, `Name`, `AuditType`, `AuditDate`, `AuditInfo`)
                select a.`Type`, a.`Schema`, a.`Name`, 'New' as AuditType, UTC_DATE() as `AuditDate`, a.`Sql` as `AuditInfo`
                from {Constants.DatabaseSchema}maintenance_staging_table_tracking as a
                where not exists
                (
	                select 1
                    from {Constants.DatabaseSchema}maintenance_table_tracking as b
                    where a.`Type` = b.`Type`
	                and a.`Schema` = b.`Schema`
	                and a.`Name` = b.`Name`
                )
                ;

                update {Constants.DatabaseSchema}maintenance_table_tracking t
                inner join {Constants.DatabaseSchema}maintenance_staging_table_tracking s
	                on t.`Type` = s.`Type`
	                and t.`Schema` = s.`Schema`
	                and t.`Name` = s.`Name`
                set
	                t.`Sql` = s.`Sql`
                where t.`Sql` <> s.`Sql`
                ;

                insert into {Constants.DatabaseSchema}maintenance_table_tracking(`Type`, `Schema`, `Name`, `Sql`)
                select `Type`, `Schema`, `Name`, `Sql`
                from {Constants.DatabaseSchema}maintenance_staging_table_tracking s
                where not exists
                (
	                select 1
	                from {Constants.DatabaseSchema}maintenance_table_tracking as t
	                where s.`Type` = t.`Type`
	                and s.`Schema` = t.`Schema`
	                and s.`Name` = t.`Name`
                );

                insert into {Constants.DatabaseSchema}maintenance_table_audit (`Type`, `Schema`, `Name`, `AuditType`, `AuditDate`, `AuditInfo`)
                select `Type`, `Schema`, `Name`, `AuditType`, `AuditDate`, `AuditInfo`
                from {Constants.DatabaseSchema}maintenance_staging_table_audit
                ;

            ";
            return sql;
        }

        private void GetTableTrackingColumns(ref Dictionary<string, TableTrackingMetaDataModel> items, ref Dictionary<string, List<TableTrackingColumnModel>> itemColumns)
        {
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText = GetTableTrackingColumnsSql();
                        command.Parameters.AddWithValue("@schema", Constants.DatabaseSchema.Substring(0, Constants.DatabaseSchema.Length - 1));
                        var columnCount = 0;
                        var tableCount = 0;
                        using (var reader = new DataReaderWithMeasurements(command, null, "get_table_tracking_columns").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    var item = new TableTrackingColumnModel()
                                    {
                                        TableType = (string)reader["table_type"],
                                        Schema = (string)reader["table_schema"],
                                        TableName = (string)reader["table_name"],
                                        ColumnName = (string)reader["column_name"],
                                        Ordinal = (long)(ulong)reader["ordinal_position"],
                                        Nullable = ((string)reader["is_nullable"] == "YES" ? true : false),
                                        Type = (string)reader["column_type"],
                                        Key = (reader["column_key"] == System.DBNull.Value ? null : (string)reader["column_key"])
                                    };
                                    var key = $@"{item.TableType}|{item.Schema}.{item.TableName}";
                                    if (!items.ContainsKey(key))
                                    {
                                        var metadata = new TableTrackingMetaDataModel()
                                        {
                                            Type = item.TableType,
                                            Schema = item.Schema,
                                            Name = item.TableName
                                        };
                                        items.Add(key, metadata);
                                        tableCount++;
                                    }
                                    if (!itemColumns.ContainsKey(key))
                                    {
                                        itemColumns.Add(key, new List<TableTrackingColumnModel>());
                                    }
                                    itemColumns[key].Add(item);
                                    columnCount++;
                                }
                            }
                        }
                        Log.Info($@"Found {tableCount.ToString("N0")} tables that contain {columnCount.ToString("N0")} total columns.");
                    }
                }

                if (items.Count > 0)
                {
                    try
                    {
                        var localTables = new Dictionary<string, string>();
                        using (var conn = DBManagerMysql.GetConnection(true))
                        {
                            foreach (var table in items)
                            {
                                using (var command = conn.GetCommand())
                                {
                                    command.CommandTimeout = 86400;
                                    command.CommandText = GetTableCreateSql(table.Value.Type, $@"{Constants.DatabaseSchema}{table.Value.Name}");
                                    using (var reader = new DataReaderWithMeasurements(command, null, "get_table_tracking_columns2").MysqlReader)
                                    {
                                        if (reader.HasRows)
                                        {
                                            while (reader.Read())
                                            {
                                                if (!localTables.ContainsKey(table.Key))
                                                {
                                                    localTables.Add(table.Key, reader.GetString(1).GzipBase64Encode());
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        foreach (var table in localTables)
                        {
                            items[table.Key].Sql = table.Value;
                        }
                    }
                    catch (Exception ex)
                    {
                        Log.Error(ex);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"TableTracking: {ex}");
            }
        }

        private string GetTableTrackingColumnsSql()
        {
            var sql =
            $@"
                select
	                b.table_type,
	                a.table_schema,
	                a.table_name,
	                a.column_name,
	                a.ordinal_position,
	                a.is_nullable,
	                a.column_type,
	                a.column_key
                from information_schema.tables as b
                left join information_schema.columns as a
                on a.table_schema = b.table_schema
                and a.table_name = b.table_name
                where a.table_schema = 'development'
                order by
	                a.table_name,
	                a.ordinal_position
            ";
            return sql;
        }

        private string GetTableCreateSql(string tableType, string table)
        {
            var sql = string.Empty;
            if (tableType == "VIEW")
            {
                sql = $"show create view {table};";
            }
            else
            {
                sql = $"show create table {table};";
            }
            return sql;
        }

        private void GetProcedures(ref Dictionary<string, TableTrackingMetaDataModel> items, ref Dictionary<string, List<TableTrackingColumnModel>> itemColumns)
        {
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText = GetProcedureListSql();
                        command.Parameters.AddWithValue("@schema", Constants.DatabaseSchema.Substring(0, Constants.DatabaseSchema.Length - 1));
                        var counter = 0;
                        using (var reader = new DataReaderWithMeasurements(command, null, "get_procedures").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    var item = new TableTrackingMetaDataModel()
                                    {
                                        Type = "PROCEDURE",
                                        Schema = (string)reader["Db"],
                                        Name = (string)reader["Name"],
                                        Sql = string.Empty
                                    };
                                    var key = $@"{item.Type}|{item.Schema}.{item.Name}";
                                    if (!items.ContainsKey(key))
                                    {
                                        items.Add(key, item);
                                        counter++;
                                    }
                                }
                            }
                        }
                        Log.Info($@"Found {counter} procedures.");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }

            if (items.Where(x => x.Value.Type == "PROCEDURE").Count() > 0)
            {
                var localProcedures = new Dictionary<string, string>();
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    foreach (var procedure in items.Where(x => x.Value.Type == "PROCEDURE"))
                    {
                        try
                        {
                            using (var command = conn.GetCommand())
                            {
                                command.CommandTimeout = 86400;
                                command.CommandText = GetProcedureCreateSql($@"{procedure.Value.Schema}.{procedure.Value.Name}");
                                using (var reader = new DataReaderWithMeasurements(command, null, "get_procedures2").MysqlReader)
                                {
                                    if (reader.HasRows)
                                    {
                                        while (reader.Read())
                                        {
                                            if (!localProcedures.ContainsKey(procedure.Key))
                                            {
                                                localProcedures.Add(procedure.Key, (reader.GetString(1)).GzipBase64Encode());
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        catch (Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }
                foreach (var procedure in localProcedures)
                {
                    items[procedure.Key].Sql = procedure.Value;
                }
            }
        }

        private string GetProcedureListSql()
        {
            var sql = "SHOW PROCEDURE STATUS where Db = @schema;";
            return sql;
        }

        private string GetProcedureCreateSql(string procedure)
        {
            var sql = $@"SHOW CREATE PROCEDURE {procedure};";
            return sql;
        }
    }
}