﻿using Resonance.Core;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Models.DatabaseModels.RequestModels;
using Resonance.Microservices.Methods;
using System;
using System.Collections.Generic;
using System.Text;
using Resonance.Microservices.Queries;
using Resonance.Core.Helpers;
using System.Linq;
using Amazon.KeyManagementService;
using Resonance.Core.Helpers.AwsHelpers;
using System.IO;
using Newtonsoft.Json;
using Resonance.Core.Models.MetaDataModels;
using Resonance.Core.ConstantData.Amp;
using Resonance.Core.Exceptions;

namespace Resonance.Jobs.Amp.Request
{
    public class SyncSalesforceRequestChannelJob : JobBase, IJob<long>
    {
        public SyncSalesforceRequestChannelJob(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                Log.Info($@"SyncSalesforceRequestChannelJob configured. IsActive: {Config.IsActive}");
            }
            catch (Exception)
            {

            }
        }

        public void SyncRequestChannels(AmpMethods ampMethod)
        {
            int channel_count = 0;
            using (var conn = DBManagerMysql.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    List<Tuple<string, string>> requestIDs = new List<Tuple<string, string>>();                    
                    command.CommandText = $"select request_id, data_key from {Constants.DatabaseSchema}microservice_twitch_request where status = '{RequestStatus.Submitted}' or status = '{RequestStatus.Finalized}' or status = '{RequestStatus.PendingFinalization}'";
                    using (var reader = new DataReaderWithMeasurements(command, null, "get_sync_request_channels").MysqlReader)
                    {
                        while (reader.Read())
                        {
                            string requestID = (string)reader["request_id"];
                            string dataKey = reader["data_key"] == System.DBNull.Value ? null : (string)reader["data_key"];
                            requestIDs.Add(new Tuple<string, string>(requestID, dataKey));
                        }
                    }
                    foreach (var request in requestIDs)
                    {
                        try
                        {
                            string requestID = request.Item1;
                            string dataKey = request.Item2;
                            if (dataKey == null)
                            {
                                continue;
                            }

                            var decryptedKey = AwsEncryption.DecryptKey(Convert.FromBase64String(dataKey), Constants.AppConfig.Application.KmsArn);

                            RequestDetails requestDetails = ampMethod.GetRequest(requestID);

                            var salesforceResults = SalesforceHelpers.QueryData($"SELECT Id,Status__c,AMP_ID__c,Preferred_Contact_Value__C,Streamer__r.External_Account_ID__c,Streamer__r.TwitchUsername__c,Datetime_Accepted_or_Denied__c,Declined_Reason__c,Nomination_Reason__c,Payment_Amount__c FROM AMP_Request_Participant__c WHERE AMP_Request__r.AMP_ID__c = '{requestID}' and Status__c != '{RequestChannelCreatorStatus.Pending}'and Status__c != '{RequestChannelCreatorStatus.Suggested}'and Status__c != '{RequestChannelCreatorStatus.Rejected}'");

                            foreach (dynamic salesforceEntry in salesforceResults)
                            {

                                string declinedReason = salesforceEntry.Declined_Reason__c;
                                string nominationReason = salesforceEntry.Nomination_Reason__c;                                

                                float? paymentAmount = null;
                                float foundAmount;

                                if (float.TryParse((string)salesforceEntry.Payment_Amount__c, out foundAmount))
                                {
                                    paymentAmount = foundAmount;
                                }

                                string status = salesforceEntry.Status__c;
                                string channelIDString = salesforceEntry.AMP_ID__c;
                                if (channelIDString == null || !channelIDString.Contains("_"))
                                {
                                    channelIDString = salesforceEntry.Streamer__r.External_Account_ID__c;
                                }
                                else
                                {
                                    channelIDString = channelIDString.Substring(channelIDString.IndexOf("_") + 1);
                                }

                                long channelID = 0;
                                if (!long.TryParse(channelIDString, out channelID))
                                {
                                    continue;
                                }
                                DateTime? responseDate = null;
                                DateTime foundDate;

                                string contactInfo = "";
                                try
                                {
                                    contactInfo = salesforceEntry.Preferred_Contact_Value__C;

                                }
                                catch (Exception)
                                {

                                }



                                if (DateTime.TryParse((string)salesforceEntry.Datetime_Accepted_or_Denied__c, out foundDate))
                                {
                                    responseDate = foundDate;
                                }

                                var existingChannel = requestDetails.Channels.Where(x => x.ChannelID == channelID).FirstOrDefault();
                                if (status == RequestChannelCreatorStatus.ApprovedSuggestion && existingChannel == null)
                                {
                                    string participantId = salesforceEntry.Id;
                                    string channelLogin = salesforceEntry.Streamer__r.TwitchUsername__c;

                                    Log.Info($"Updating channel {channelLogin} status Suggested on request {requestDetails.Request.ID}");

                                    var channelDetails = AmpQuery.GetChannelDetails(channelLogin);
                                    command.Parameters.Clear();
                                    command.CommandText = $"insert into {Constants.DatabaseSchema}microservice_twitch_request_channel (request_id, channel_id, channel_login, status, profile_image, Priority, sort_order, declined_reason, nomination_reason, payment_amount) values(@request_id, @channel_id, @channel_login, '{RequestChannelStatus.Suggested}', @profile_image, 0, 0, @declined_reason, @nomination_reason, @payment_amount)";
                                    command.Parameters.AddWithValue("request_id", requestID);
                                    command.Parameters.AddWithValue("channel_id", channelID);
                                    command.Parameters.AddWithValue("channel_login", channelLogin);
                                    command.Parameters.AddWithValue("profile_image", channelDetails.ProfileImage);
                                    command.Parameters.AddWithValue("declined_reason", declinedReason);
                                    command.Parameters.AddWithValue("nomination_reason", nominationReason);
                                    command.Parameters.AddWithValue("payment_amount", paymentAmount);

                                    command.ExecuteNonQueryWithMeasurements("sync_request_channels");
                                    //Update the external id for suggestions so we can update the record later withour own own data, rather than syncing over the salesforce id

                                    SalesforceHelpers.UpdateRecord(SalesforceNames.Amp_Request_Participant__C, participantId, new { AMP_ID__c = $"{requestID}_{channelID}" });
                                    channel_count++;
                                }
                                else if (status == RequestChannelCreatorStatus.ApprovedSuggestion && existingChannel != null && existingChannel.Status != RequestChannelStatus.Suggested)
                                {
                                    string participantId = salesforceEntry.Id;
                                    Log.Info($"Updating channel {existingChannel.ChannelLogin} status Suggested on request {requestDetails.Request.ID}");

                                    command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request_channel set status = @status, contact_info = @contact_info, response_date = @response_date, declined_reason = @declined_reason, nomination_reason = @nomination_reason, payment_amount = @payment_amount where request_id = @request_id and channel_id = @channel_id";
                                    command.Parameters.Clear();
                                    command.Parameters.AddWithValue("request_id", requestID);
                                    command.Parameters.AddWithValue("channel_id", channelID);
                                    command.Parameters.AddWithValue("response_date", responseDate);
                                    command.Parameters.AddWithValue("status", RequestChannelStatus.Suggested);
                                    command.Parameters.AddWithValue("declined_reason", declinedReason);
                                    command.Parameters.AddWithValue("nomination_reason", nominationReason);
                                    command.Parameters.AddWithValue("payment_amount", paymentAmount);

                                    string encryptedContactInfo = AwsEncryption.EncryptString(contactInfo ?? "Not Provided", decryptedKey);
                                    command.Parameters.AddWithValue("contact_info", encryptedContactInfo);

                                    command.ExecuteNonQueryWithMeasurements("sync_request_channels2");

                                    SalesforceHelpers.UpdateRecord(SalesforceNames.Amp_Request_Participant__C, participantId, new { AMP_ID__c = $"{requestID}_{channelID}" });
                                    channel_count++;
                                }
                                else if (existingChannel != null && existingChannel.CreatorStatus != status)
                                {
                                    Log.Info($"Updating channel {existingChannel.ChannelLogin} status {status} on request {requestDetails.Request.ID}");

                                    command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request_channel set creator_status = @status, contact_info = @contact_info, response_date = @response_date, declined_reason = @declined_reason, nomination_reason = @nomination_reason, payment_amount = @payment_amount where request_id = @request_id and channel_id = @channel_id";
                                    command.Parameters.Clear();
                                    command.Parameters.AddWithValue("request_id", requestID);
                                    command.Parameters.AddWithValue("channel_id", channelID);
                                    command.Parameters.AddWithValue("response_date", responseDate);
                                    command.Parameters.AddWithValue("status", status);
                                    command.Parameters.AddWithValue("declined_reason", declinedReason);
                                    command.Parameters.AddWithValue("nomination_reason", nominationReason);
                                    command.Parameters.AddWithValue("payment_amount", paymentAmount);

                                    string encryptedContactInfo = AwsEncryption.EncryptString(contactInfo ?? "Not Provided", decryptedKey);
                                    command.Parameters.AddWithValue("contact_info", encryptedContactInfo);
                                    command.ExecuteNonQueryWithMeasurements("sync_request_channels3");
                                    channel_count++;
                                }
                            }
                        }
                        catch (Exception ex)
                        {
                            Log.Error("Error updating request: " + ex.ToString());
                        }
                    }
                }
            }
            Log.Info("Done scanning salesforce");
            CloudwatchHelper.EnqueueMetricRequest("sync_salesforce_request_channel_job_channel_count", channel_count, null, Amazon.CloudWatch.StandardUnit.Count);
        }

        public void CheckAMSuggestionReadiness(AmpMethods ampMethods)
        {
            var salesforceReadyRequests = SalesforceHelpers.QueryData("select AMP_ID__c from AMP_Request__c where Past_Suggestion_Deadline__c = TRUE and Number_of_Suggestions_Pending_Approval__c = 0");
            foreach(dynamic salesforceRecord in salesforceReadyRequests)
            {
                string requestID = salesforceRecord.AMP_ID__c;
                var request = ampMethods.GetRequest(requestID, false, false);
                if(request?.Request?.Status == RequestStatus.Submitted)
                {
                    AmpQuery.UpdateRequestStatus(requestID, RequestStatus.PendingFinalization);
                    AmpQuery.RecordRequestStatusChange(requestID, request.Request.Status, RequestStatus.PendingFinalization, "Salesforce");
                    try
                    {
                        SalesforceHelpers.UpdateRecordByExternalID(SalesforceNames.Amp_Request__C, SalesforceNames.Amp_ID__C, requestID, new { Status__c = RequestStatus.PendingFinalization });
                    }
                    catch (SalesforceException ex)
                    {
                        Log.Error(ex);
                    }
                }
            }
        }

        public override void Run()
        {

            AmpMethods ampMethod = new AmpMethods();
            ampMethod.Initialize();
            try
            {
                SyncRequestChannels(ampMethod);
                CheckAMSuggestionReadiness(ampMethod);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromMinutes(5));
            }
        }
        

    }
}
