﻿using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.ConfigurationHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Models.DatabaseModels.TwitchUserListingModels;
using Resonance.Core.Services.CustomConfigService;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace Resonance.Jobs.Amp.TwitchUser
{
    class TwitchGamesJob : JobBase, IJob<long>
    {
        public TwitchGamesJob(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                Log.Info($@"TwitchGamesJob: configured. IsActive: {Config.IsActive}");
            }
            catch (Exception)
            {

            }
        }
        public override void Run()
        {
            try
            {
                string s3Bucket = ConstantsWorker.WorkerS3Bucket;
                string s3Path = $"{AppConfig.Data.Application.Environment}/amp-games";
                List<TwitchGameModel> games = new List<TwitchGameModel>();
                Log.Info($@"TwitchGamesJob: Running.");
                this.Config.IsRunning = true;
                using (var tahoeConn = DBManagerRedshift.RedshiftConnection(true, DBManagerRedshift.GetTahoeConnectionString()))
                {
                    using (var command = tahoeConn.CreateCommand())
                    {
                        command.CommandText = "select * from dbsnapshots.games";
                        using (var wrappedreader = new DataReaderWithMeasurements(command, null, "twitch_games_job"))
                        {
                            var reader = wrappedreader.NpgReader;
                            while (reader.Read())
                            {
                                TwitchGameModel model = new TwitchGameModel();
                                model.GameID = (int)reader["id"];
                                var giantBombID = reader["giantbomb_id"];
                                model.GiantBombID = giantBombID == System.DBNull.Value ? 0 : (int)giantBombID;
                                model.Name = (string)reader["Name"];
                                var images = reader["images"];
                                if (images != System.DBNull.Value)
                                {
                                    string imageData = (string)reader["images"];
                                    string jsonData = "{" + imageData.Replace("=>", ":") + "}";
                                    try
                                    {
                                        dynamic imageEntries = JsonConvert.DeserializeObject<dynamic>(jsonData);
                                        model.ImageURL = imageEntries.thumb_url;
                                        model.TinyImageURL = imageEntries.tiny_url;
                                    }
                                    catch (Exception)
                                    {
                                    }
                                }
                                games.Add(model);
                            }
                        }

                        using (var stream = new StringWriter())
                        {
                            using (var csv = new CsvHelper.CsvWriter(stream, new CsvHelper.Configuration.Configuration()
                            {
                                Delimiter = ",",
                                Quote = '"',
                                ShouldQuote = (field, context) => !field.Contains("_id"),
                                Encoding = Encoding.UTF8,
                                HasHeaderRecord = false
                            }))
                            {
                                csv.WriteRecords(games);
                                S3Helper.WriteStringToS3(stream.ToString(), s3Bucket, s3Path, false);
                            }
                        }
                    }


                }
                DBManagerMysql.ImportToMysqlFromS3
                (
                    prefixQuery: AmpQuerySql.DropTwitchGamesTempTable(),
                    primaryQuery: AmpQuerySql.LoadFileFromS3(s3Bucket, s3Path, @"temp_microservice_twitch_games"),
                    postfixQuery: AmpQuerySql.InsertTwitchGames(),
                    timeout: 86400
                );
                Log.Info("TwitchGamesJob: Done Saving.");

                Log.Info("TwitchGamesJob: Uploading games file to S3");
                if (UpdateFilterFile(new CustomConfigService(Constants.AppConfig.Application.Environment)))
                {
                    Log.Info("TwitchGamesJob: Upload successfull");
                }
                else
                {
                    Log.Error("TwitchGamesJob: Upload Error");
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"TwitchGamesJob: {ex}");
            }
            finally
            {
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromDays(1));                

                Log.Info("TwitchGamesJob: NextRun: " + this.Config.NextRunTime.Value.ToString("yyyy-MM-dd HH:mm"));
            }

        }

        public static bool UpdateFilterFile(ICustomConfigService service)
        {
            try
            {
                string key = "games";
                FilterOptionsHelper helper = new FilterOptionsHelper();
                string baseDir = Path.GetTempPath();
                string path = $"{baseDir}{key}.txt";

                if (helper.CreateOptionsFile(key, path))
                {
                    service.UploadConfigFile("filter", key, path);

                    try { File.Delete(path); }
                    catch { }

                    return true;
                }
                else
                {
                    Log.Error("TwitchGamesJob: could not CreateConfigOptionsFile");
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }

            return false;
        }
    }
}
