﻿using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Extensions;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.DateTimeHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Resonance.Jobs.Amp.TwitchUser
{
    public class UserCalendarJob : JobBase, IJob<long>
    {
        private static AmpUserCalendarJobConfiguration jobConfig { get; set; }

        public UserCalendarJob(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                jobConfig = new AmpUserCalendarJobConfiguration()
                {
                    MinimumDollarThreshold = 50
                };
                Log.Info($"UserCalendarJob: configured. IsActive: {Config.IsActive}");
            }
            catch (Exception ex)
            {
                Log.Error($@"UserCalendarJob: {ex}");
            }
        }

        public override void Run()
        {
            try
            {
                long resultCount = 0;

                Log.Info($"UserCalendarJob: Running");
                this.Config.IsRunning = true;
                if (jobConfig != null)
                {
                    try
                    {
                        var rundate = DateTime.UtcNow.Date;
                        WipeOldData();
                        var param = new Dictionary<string, dynamic>();
                        var lookback = DateTime.UtcNow.Date.AddDays(-365);
                        var runtime = DateTime.UtcNow;
                        for (var month = 0; month < 12; month++)
                        {
                            for (var channelDigit = 0; channelDigit <= 9; channelDigit++)
                            {
                                try
                                {
                                    var shiftDate = runtime.Date.AddMonths(-month);
                                    param.Clear();
                                    param.Add("@lookback", lookback.ToRedshiftDateFormat());
                                    param.Add("@mindatestring", new DateTime(shiftDate.Year, shiftDate.Month, 1).ToRedshiftDateFormat());
                                    param.Add("@maxdatestring", new DateTime(shiftDate.Year, shiftDate.Month, DateTime.DaysInMonth(shiftDate.Year, shiftDate.Month)).AddDays(1).ToRedshiftDateFormat());
                                    param.Add("@dollarthreshold", jobConfig.MinimumDollarThreshold);
                                    param.Add("@channeldigit", channelDigit);

                                    var updatefield = GetRunMonth(shiftDate);
                                    var bucket = "crs-data-export";
                                    var keypath = $"{Constants.AppConfig.Application.Environment}/twitch-user-calendar-data/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/{updatefield}/end_with_{channelDigit}/data.csv";
                                    var results = GetResultsFromTahoe(param);
                                    WriteResultsToS3(bucket, keypath, updatefield, results);
                                    UpdateCalendarFromStaging(updatefield, channelDigit);
                                    resultCount = results.Count;
                                    results.Clear();
                                    results = null;
                                }
                                catch (Exception ex)
                                {
                                    Log.Error($@"UserCalendarJob: {ex}");
                                }
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        Log.Error($@"UserCalendarJob: {ex}");
                    }

                    try
                    {
                        using (var conn = DBManagerMysql.GetConnection(true))
                        {
                            using
                            (
                                var command =
                                    AmpQuerySql.InsertUpdateEtlTracking
                                    (
                                        source: "Tahoe_Calendar_Query",
                                        target: $"Microservice_Twitch_User_Calendar",
                                        method: "Jobs.UserCalendarJob",
                                        timestamp: DateTime.UtcNow,
                                        rowcount: resultCount
                                    )
                            )
                            {
                                command.Connection = conn;
                                command.CommandTimeout = 0;
                                command.ExecuteNonQueryWithMeasurements("user_calendar_job_etl");
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        Log.Error($@"UserCalendarJob: {ex}");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserCalendarJob: {ex}");
            }
            finally
            {
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromDays(1));
            }

            Log.Info($@"UserCalendarJob: Complete. Next Run Time: {this.Config.NextRunTime.Value.ToString("yyyy-MM-dd HH:mm:ss")}");
        }

        private string GetRunMonth(DateTime shiftDate)
        {
            string updatefield = "";
            switch (shiftDate.Month)
            {
                case 1:
                {
                    updatefield = "January";
                    break;
                }
                case 2:
                {
                    updatefield = "February";
                    break;
                }
                case 3:
                {
                    updatefield = "March";
                    break;
                }
                case 4:
                {
                    updatefield = "April";
                    break;
                }
                case 5:
                {
                    updatefield = "May";
                    break;
                }
                case 6:
                {
                    updatefield = "June";
                    break;
                }
                case 7:
                {
                    updatefield = "July";
                    break;
                }
                case 8:
                {
                    updatefield = "August";
                    break;
                }
                case 9:
                {
                    updatefield = "September";
                    break;
                }
                case 10:
                {
                    updatefield = "October";
                    break;
                }
                case 11:
                {
                    updatefield = "November";
                    break;
                }
                case 12:
                {
                    updatefield = "December";
                    break;
                }
                default:
                {
                    throw new InvalidDataException("dt");
                }
            }
            return updatefield;
        }

        private List<TwitchUserCalendarQueryModel> GetResultsFromTahoe(Dictionary<string, dynamic> param)
        {
            var items = new List<TwitchUserCalendarQueryModel>();
            try
            {
                using (var conn = DBManagerRedshift.TahoeConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText = GetCalendarSql();
                        Log.Verbose($@"UserCalendarJob: {command.CommandText}");
                        foreach (var p in param)
                        {
                            command.Parameters.AddWithValue(p.Key, p.Value);
                            Log.Verbose($@"{p.Key}:{p.Value}:{p.Value.GetType()}");
                        }
                        using (var reader = new DataReaderWithMeasurements(command, null, "get_tahoe_calendar").NpgReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    var item = new TwitchUserCalendarQueryModel();
                                    item.TwitchUserID = reader.GetInt64(0);
                                    item.Day = reader.GetDateTime(1);
                                    item.HasDmcaTakedown = reader.GetBoolean(2) ? true : (bool?)null;
                                    item.HasAccountWarning = reader.GetBoolean(3) ? true : (bool?)null;
                                    item.HasAccountInfractions = reader.GetBoolean(4) ? true : (bool?)null;
                                    item.IsPermBanned = reader.GetBoolean(5) ? true : (bool?)null;
                                    item.IsTempBanned = reader.GetBoolean(6) ? true : (bool?)null;
                                    var minutesWatched = reader.GetInt64(7);
                                    item.MinutesWatched = minutesWatched > 0 ? minutesWatched : (long?)null;
                                    var minutesBroadcast = reader.GetInt64(8);
                                    item.MinutesBroadcast = minutesBroadcast > 0 ? minutesBroadcast : (long?)null;
                                    var timesCast = reader.GetInt32(9);
                                    item.TimesCast = timesCast > 0 ? timesCast : (int?)null;
                                    item.BountyStarted = reader.GetBoolean(10) ? true : (bool?)null;
                                    item.BountyComplete = reader.GetBoolean(11) ? true : (bool?)null;
                                    item.BountyCancelled = reader.GetBoolean(12) ? true : (bool?)null;
                                    item.Login = reader.GetString(13);
                                    item.DropInUrl = $@"https://dashboard.twitch.tv/u/{item.Login}/channel-analytics?end={item.Day.ToString("yyyy-MM-dd")}&start={item.Day.ToString("yyyy-MM-dd")}";
                                    items.Add(item);
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserCalendarJob: {ex}");
            }
            return items;
        }

        private void WriteResultsToS3(string bucket, string keypath, string runmonth, List<TwitchUserCalendarQueryModel> data)
        {
            var kvp = new Dictionary<long, List<TwitchUserCalendarQueryModel>>();
            var writeItems = new List<TwitchUserCalendarModel>();

            try
            {
                var items = data.GroupBy(x => $@"{x.Day.Year}-{x.Day.Month}")
                        .ToDictionary(x => x.Key, x => x.Select(y => y).ToArray());
                if (items.Keys.Count > 1)
                {
                    throw new InvalidDataException("More than 1 month detected in result set. Aborting process.");
                }

                // Adds item to list, removing the user id from the original object so that it is serialized away in the database
                foreach (var item in items.Values.SelectMany(x => x).OrderBy(x => x.Day).ToArray())
                {
                    var origID = (long)item.TwitchUserID;

                    if (!kvp.ContainsKey(origID))
                    {
                        kvp.Add(origID, new List<TwitchUserCalendarQueryModel>());
                    }
                    item.TwitchUserID = null;
                    kvp[origID].Add(item);
                }
                // Aggregate items into single line results
                foreach (var item in kvp)
                {
                    writeItems.Add(new TwitchUserCalendarModel()
                    {
                        TwitchUserID = item.Key,
                        Month = runmonth,
                        Data = JsonConvert.SerializeObject(item.Value).GzipBase64Encode(4096)
                    });
                }
                var partitionItems = writeItems.Partition(10000).ToArray();
                for (var i = 0; i < partitionItems.Length; i++)
                {
                    var modifiedkeypath = keypath.Replace("data.csv", $"data_part_{i}.csv");
                    foreach (var month in items)
                    {
                        using (var stream = new MemoryStream())
                        {
                            using (TextWriter writer = new StreamWriter(stream, Encoding.UTF8, 2048, leaveOpen: true))
                            {
                                using (var csv = new CsvHelper.CsvWriter(writer, new CsvHelper.Configuration.Configuration()
                                {
                                    Delimiter = ",",
                                    Quote = '"',
                                    ShouldQuote = (field, context) => true,
                                    Encoding = Encoding.UTF8,
                                    HasHeaderRecord = false
                                }, leaveOpen: true))
                                {
                                    csv.WriteRecords(partitionItems[i]);
                                }
                            }
                            stream.Position = 0;
                            Log.Info($@"UserCalendarJob: Uploading {bucket}/{modifiedkeypath} to s3");
                            S3Helper.UploadToS3(stream, bucket, modifiedkeypath);
                        }
                    }
                    LoadS3FileToMysqlStaging(bucket, modifiedkeypath);
                }
                partitionItems = null;
            }
            catch (Exception ex)
            {
                Log.Error($@"UserCalendarJob: {ex}");
            }
            finally
            {
                kvp.Clear();
                kvp = null;
                writeItems.Clear();
                writeItems = null;
            }
        }

        private void WipeOldData()
        {
            Log.Info($"UserCalendarJob: Wiping old user calendar data from staging table");
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"truncate table {Constants.DatabaseSchema}microservice_staging_twitch_user_calendar;";
                    command.ExecuteNonQueryWithMeasurements("user_calendar_job_cleanup");
                }
            }
            Log.Info($"UserCalendarJob: Completed wiping old user calendar data from staging table");
        }

        private string GetCalendarSql()
        {
            string sql =
            $@"
                /* BEGIN TWITCH USER CALENDAR QUERY */
                with apdcs as
                (
                    select cast(channel_id as bigint) as channel_id
                    from cubes.amp_application_population
                    where right(channel_id, 1) = @channeldigit
                    group by cast(channel_id as bigint)
                ),
                days as
                (
                    select day
                    from cubes.hours_watched_daily_by_device
                    where 
                        day >= cast(@mindatestring as date)
                        and day < cast(@maxdatestring as date)
                    group by day
                ),
                infractions as
                (
                    select
                        date_trunc('day', date) as day,
                        target_user_id as channel_id,
                        case when account_ban_duration = -1 then 1 else 0 end as is_perm_ban,
                        case when banned_until_utc is not null then 1 else 0 end as is_temp_ban
                    from tahoe.leviathan_suspension as a
                    where
                        date >= cast(@mindatestring as date)
                        and date < cast(@maxdatestring as date)
                        and right(target_user_id, 1) = @channeldigit
                ),
                dmca_takedown as
                (
                    select
                        date_trunc('day', date) as day,
                        content_poster_id as channel_id,
                        count(*) as takedown_count
                    from tahoe.dmca_takedown
                    where
                        date >= cast(@mindatestring as date)
                        and date < cast(@maxdatestring as date)
                        and right(content_poster_id, 1) = @channeldigit
                    group by
                        date_trunc('day', date),
                        channel_id
                ),
                warnings as
                (
                    select
                        date_trunc('day', date) as day,
                        target_user_id as channel_id,
                        count(*) as warning_count
                    from tahoe.aegis_warning
                    where
                        date >= cast(@mindatestring as date)
                        and date < cast(@maxdatestring as date)
                        and right(target_user_id, 1) = @channeldigit
                    group by
                        date_trunc('day', date),
                        channel_id
                ),
                minutes_watched as
                (
                    select
                        channel_id,
                        day,
                        sum(mw) as minutes_watched
                    from cubes.hours_watched_daily_by_device
                    where 
                        day >= cast(@mindatestring as date)
                        and day < cast(@maxdatestring as date)
                        and right(channel_id, 1) = @channeldigit
                    group by
                        day,
                        channel_id
                ),
                minutes_broadcast as
                (
                    select
                        channel_id,
                        day,
                        sum(minutes_broadcast_total) as minutes_broadcast_total,
                        count(*) as count_times_cast
                    from cubes.affiliates_partners_daily_channel_summary as a
                    where
                        day >= cast(@mindatestring as date)
                        and day < cast(@maxdatestring as date)
                        and coalesce(channel_id, 0) > 0
                        and right(channel_id, 1) = @channeldigit
                    group by
                        day,
                        channel_id
                ),
                bounty as
                (
                    select
                        user_id as channel_id,
                        date_trunc('day', date) as day,
                        sum(case when bounty_status = 'LIVE' then 1 else 0 end) as bounty_started,
                        sum(case when bounty_status = 'COMPLETED' then 1 else 0 end) as bounty_completed,
                        sum(case when bounty_status = 'CANCELLED' then 1 else 0 end) as bounty_cancelled
                    from tahoe.bounty_details_update
                    where
                        date >= cast(@mindatestring as date)
                        and date < cast(@maxdatestring as date)
                        and right(user_id, 1) = @channeldigit
                    group by
                        user_id,
                        date_trunc('day', date)
                )
                select
                    apdcs_days.channel_id,
                    apdcs_days.day,
                    case when sum(coalesce(dmca_takedown.takedown_count, 0)) > 0 then True else False end as has_dmca_takedown,
                    case when sum(coalesce(warnings.warning_count, 0)) > 0 then True else False end as has_warning,
                    case when max(coalesce(infractions.channel_id, 0)) > 0 then True else False end as has_infractions,
                    case when sum(coalesce(infractions.is_perm_ban, 0)) > 0 then True else False end as is_perm_ban,
                    case when sum(coalesce(infractions.is_perm_ban, 0)) = 0 and sum(coalesce(infractions.is_temp_ban, 0)) > 0 Then True else False end as is_temp_ban,
                    sum(coalesce(minutes_watched.minutes_watched, 0)) as minutes_watched,
                    sum(coalesce(minutes_broadcast_total, 0)) as minutes_broadcast,
                    sum(coalesce(count_times_cast, 0)) as times_cast,
                    case when sum(coalesce(bounty_started, 0)) > 0 then True else False end as bounty_started,
                    case when sum(coalesce(bounty_completed, 0)) > 0 then True else False end as bounty_completed,
                    case when sum(coalesce(bounty_cancelled, 0)) then True else False end as bounty_cancelled,
                    coalesce(users.login, '') as login
                from
                (
                    select *
                    from apdcs
                    cross join days
                ) as apdcs_days
                left join infractions
                    on apdcs_days.channel_id = infractions.channel_id
                    and apdcs_days.day = infractions.day
                left join minutes_watched
                    on apdcs_days.channel_id = minutes_watched.channel_id
                    and apdcs_days.day = minutes_watched.day
                left join minutes_broadcast
                    on apdcs_days.channel_id = minutes_broadcast.channel_id
                    and apdcs_days.day = minutes_broadcast.day
                left join dmca_takedown
                    on apdcs_days.channel_id = dmca_takedown.channel_id
                    and apdcs_days.day = dmca_takedown.day
                left join warnings
                    on apdcs_days.channel_id = warnings.channel_id
                    and apdcs_days.day = warnings.day
                left join bounty
                    on apdcs_days.channel_id = bounty.channel_id
                    and apdcs_days.day = bounty.day
                left join dbsnapshots.users
                    on apdcs_days.channel_id = users.id
                group by
                    apdcs_days.channel_id,
                    apdcs_days.day,
                    coalesce(users.login, '')
                ;
                /* END TWITCH USER CALENDAR QUERY */
            ";
            return sql;
        }

        private void LoadS3FileToMysqlStaging(string bucket, string keypath)
        {
            try
            {
                string fields =
                @"(
                    TwitchUserID, Month, MonthData
                )";
                if (S3Helper.ExistsInS3(bucket, $"{keypath}") == true)
                {
                    using (var conn = DBManagerMysql.GetConnection(true))
                    {
                        using (var command = AmpQuerySql.LoadFileFromS3
                            (bucket,
                            $"{keypath}",
                            $"{Constants.DatabaseSchema}microservice_staging_twitch_user_calendar",
                            ignore: "",
                            fields: fields)
                        )
                        {
                            command.CommandTimeout = 86400;
                            command.Connection = conn;
                            command.ExecuteNonQueryWithMeasurements("user_calendar_job");
                        }
                    }
                }
                else
                {
                    Log.Error($"UserCalendarJob: Unable to load s3 file to mysql staging: {bucket}/{keypath} was not found.");
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserCalendarJob: {ex}");
            }
        }

        private void UpdateCalendarFromStaging(string updatefield, int channelDigit)
        {
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            insert into {Constants.DatabaseSchema}microservice_twitch_user_calendar(TwitchUserID, Month)
                            select TwitchUserID, Month
                            from {Constants.DatabaseSchema}microservice_staging_twitch_user_calendar as a
                            where not exists
                            (
                                select 1
                                from {Constants.DatabaseSchema}microservice_twitch_user_calendar as b
                                where a.TwitchUserID = b.TwitchUserID
                                and a.Month = b.Month
                            )
                            and a.TwitchUserID > 0
                            and a.Month = '{updatefield}'
                            and right(a.TwitchUserID, 1) = @channeldigit
                            ;                 

                            update {Constants.DatabaseSchema}microservice_twitch_user_calendar t
                            inner join {Constants.DatabaseSchema}microservice_staging_twitch_user_calendar as s
                                on t.TwitchUserID = s.TwitchUserID
                                and t.Month = s.Month
                                and right(s.TwitchUserID, 1) = @channeldigit
                                and t.Month = '{updatefield}'
                                and s.Month = '{updatefield}'
                            set
                                t.MonthData = s.MonthData
                            where
                                t.TwitchUserID = s.TwitchUserID
                                and t.Month = s.Month
                                and right(s.TwitchUserID, 1) = @channeldigit
                                and s.Month = '{updatefield}'
                                and t.Month = '{updatefield}'
                                and t.TwitchUserID > 0
                            ;
                        ";
                        command.Parameters.AddWithValue("@channeldigit", channelDigit);
                        command.ExecuteNonQueryWithMeasurements("user_calendar_job");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserCalendarJob: {ex}");
            }
        }
    }
}