﻿using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Extensions;
using Resonance.Core.Helpers.AggregationHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Models.DatabaseModels.TwitchUserListingModels;
using Resonance.Microservices.Methods;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using static Resonance.Core.Constants;

namespace Resonance.Jobs.Amp.TwitchUser
{
    public class UserCardJob : JobBase, IJob<long>
    {
        private static ConcurrentDictionary<long, DateTime> cache { get; set; } = null;
        private static TwitchUserCardConfiguration jobConfig { get; set; } = null;

        public UserCardJob(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                jobConfig = new TwitchUserCardConfiguration()
                {
                    RequiredAggregates = new AggregationType[5]
                    {
                        AggregationType.Week,
                        AggregationType.ThirtyDay,
                        AggregationType.SixtyDay,
                        AggregationType.NintyDay,
                        AggregationType.BankerYear
                    }
                };
                cache = new ConcurrentDictionary<long, DateTime>();
                Log.Info($@"UserCardJob: configured. IsActive: {Config.IsActive}");
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }

        public override void Run()
        {
            try
            {
                Log.Info($@"UserCardJob: Running.");

                this.Config.IsRunning = true;
                if (jobConfig != null && jobConfig.RequiredAggregates != null && jobConfig.RequiredAggregates.Length > 0)
                {
                    Log.Info($@"UserCardJob: Preparing the following aggregates: {string.Join(",", jobConfig.RequiredAggregates)}");
                    var maxdate = DateTime.UtcNow.Date;
                    WipeOldData();
                    foreach (var aggregationType in jobConfig.RequiredAggregates)
                    {
                        var resultCount = UpdateData(aggregationType, maxdate);

                        try
                        {
                            using (var conn = DBManagerMysql.GetConnection(true))
                            {
                                using
                                (
                                    var command =
                                        AmpQuerySql.InsertUpdateEtlTracking
                                        (
                                            source: "Redshift_Microservice_Twitch_User_Cards",
                                            target: $"Microservice_Twitch_User_Cards_Past_{(int)aggregationType}_Days",
                                            method: "Jobs.UserCardsJob",
                                            timestamp: DateTime.UtcNow,
                                            rowcount: resultCount
                                        )
                                )
                                {
                                    command.Connection = conn;
                                    command.CommandTimeout = 0;
                                    command.ExecuteNonQueryWithMeasurements("user_card_job_etl");
                                }
                            }
                        }
                        catch (Exception ex)
                        {
                            Log.Error($@"UserCardJob: {ex}");
                        }

                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserCardJob: {ex}");
            }
            finally
            {
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromDays(1));
            }
            Log.Info($@"UserCardJob: Complete. Next Run Time: {this.Config.NextRunTime.Value.ToString("yyyy-MM-dd HH:mm:ss")}");
        }

        private void WipeOldData()
        {
            Log.Info($"UserCardJob: Wiping old card data from staging table");
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"truncate table {Constants.DatabaseSchema}microservice_staging_twitch_user_cards;";
                    command.ExecuteNonQueryWithMeasurements("user_card_job_cleanup");
                }
            }
            Log.Info($"UserCardJob: Completed wiping old card data from staging table");
        }

        private long UpdateData(AggregationType aggregationType, DateTime maxdate)
        {
            long resultCount = 0;

            string timeSpan = "day";
            string aggregateSnippet = string.Empty;
            DateTime mindate;
            int aggLevel = 0;
            switch (aggregationType)
            {
                case AggregationType.Week:
                {
                    mindate = maxdate.AddDays(-7);
                    aggregateSnippet = SnippetHelper.GetQuery7Snippet();
                    aggLevel = 1;
                    timeSpan = "day";
                    break;
                }
                case AggregationType.ThirtyDay:
                {
                    mindate = maxdate.AddDays(-30);
                    aggregateSnippet = SnippetHelper.GetQuery30Snippet();
                    aggLevel = 1;
                    timeSpan = "day";
                    break;
                }
                case AggregationType.SixtyDay:
                {
                    mindate = maxdate.AddDays(-60);
                    aggregateSnippet = SnippetHelper.GetQuery60Snippet();
                    aggLevel = 2;
                    timeSpan = "week";
                    break;
                }
                case AggregationType.NintyDay:
                {
                    mindate = maxdate.AddDays(-90);
                    aggregateSnippet = SnippetHelper.GetQuery90Snippet();
                    aggLevel = 2;
                    timeSpan = "week";
                    break;
                }
                case AggregationType.BankerYear:
                {
                    mindate = maxdate.AddDays(-360);
                    aggregateSnippet = SnippetHelper.GetQuery360Snippet();
                    aggLevel = 3;
                    timeSpan = "month";
                    break;
                }
                default:
                {
                    throw new ArgumentOutOfRangeException("aggregationType");
                }
            }

            Log.Info($"UserCardJob: Aggregation Snippet: {aggregateSnippet}, Mindate: {mindate.ToString("yyyy-MM-dd")}, Maxdate: {maxdate.ToString("yyyy-MM-dd")}, AggLevel: {aggLevel}, Timespan: {timeSpan}");
            string sql = string.Empty;
            var checkdate = new DateTime(1901, 1, 1);
            try
            {
                var param = new Dictionary<string, dynamic>()
                {
                    { "@aggregationtype", (int)aggregationType },
                    { "@agglevel", aggLevel },
                    { "@channeldigit", 0 },
                    { "@timespan", timeSpan },
                    { "@mindate", mindate.ToString("yyyy-MM-dd")}
                };
                for(var i = 0; i <= 9; i++)
                {
                    param["@channeldigit"] = i;
                    var loop = 0;

                    sql =
                    $@"
                        /* BEGIN TWITCH USER CARD QUERY */
                        select
                            @aggregationtype as AggregationType,
                            date_trunc(@timespan, a.day) as Day,
                            cast(channel_id as bigint) as TwitchUserID,
                            sum(cast(isnull(ccu_minutes_watched_total, 0) as bigint)) as ChannelConcurrentMinutesWatchedTotal,
                            max(cast(isnull(max_ccu_past_{(int)aggregationType}_days, 0) as int)) as MaxCCUHistorical,
                            avg(cast(isnull(avg_ccu, 0.0) as float)) as AverageCCU,
                            @agglevel as AggregationGranularity
                        from cubes.affiliates_partners_daily_channel_summary as a
                        where
                            a.day >= cast(@mindate as date)
                            and right(channel_id, 1) = @channeldigit
                            group by 
                                a.channel_id, 
                                date_trunc(@timespan, a.day)
                        /* END TWITCH USER CARD QUERY */
                    ";
                    var results = DBManagerRedshift.GetSqlData<TwitchUserCardQueryModel>(sql, parameters: param, timeout: 86400, useTahoe: true)?.ToArray();
                    
                    var result = results.GroupBy(x => x.TwitchUserID)
                    .Select(group => new
                    {
                        ChannelID = group.Key,
                        LastUpdateDate = maxdate,
                        AggregationType = (int)aggregationType,
                        MinutesWatchedData = JsonConvert.SerializeObject(group?.Select(x => new
                        {
                            x.Day,
                            Granularity = (int)x.AggregationGranularity,
                            MinutesWatched = x.ChannelConcurrentMinutesWatchedTotal
                        }).ToArray()),
                        AverageCcuData = JsonConvert.SerializeObject(group?.Select(x => new
                        {
                            x.Day,
                            Granularity = (int)x.AggregationGranularity,
                            x.AverageCCU
                        }).ToArray()),
                        ServiceID = ConstantsWorker.WorkerIdentifier,
                        BatchID = Guid.NewGuid().ToString("d")
                    })
                    ?.ToArray()
                    .Partition(10000).ToArray()
                    ;

                    if (results != null && results.Length > 0)
                    {
                        foreach (var data in result)
                        {
                            Log.Info($"UserCardJob: Found {results.Length} card results. Writing CSV to stream.");
                            var keypath = $"{Constants.AppConfig.Application.Environment}/twitch-card-data/{maxdate.ToString("yyyy")}/{maxdate.ToString("MM")}/{maxdate.ToString("dd")}/{(int)aggregationType}/{i}/card-data-{loop}.csv";
                            using (var stream = new MemoryStream())
                            {
                                using (TextWriter writer = new StreamWriter(stream, Encoding.UTF8, bufferSize: 2048, leaveOpen: true))
                                {
                                    using (var csv = new CsvHelper.CsvWriter(writer, new CsvHelper.Configuration.Configuration()
                                    {
                                        Delimiter = ",",
                                        Quote = '"',
                                        ShouldQuote = (field, context) => true,
                                        PrepareHeaderForMatch = (header, context) => header.ToLower(),
                                        Encoding = Encoding.UTF8,
                                    },
                                    leaveOpen: true))
                                    {
                                        var records = data.Select(x => new
                                        {
                                            x.ChannelID,
                                            LastUpdateDate = x.LastUpdateDate.ToRedshiftDateFormat(),
                                            x.AggregationType,
                                            x.MinutesWatchedData,
                                            x.AverageCcuData,
                                            x.ServiceID,
                                            x.BatchID
                                        }).ToArray();
                                        resultCount += records.Length;
                                        Log.Info($"UserCardJob: Adding {records.Length} records to CSV");
                                        csv.WriteRecords(records);
                                        records = null;
                                    }
                                }
                                stream.Position = 0;
                                Log.Info($"UserCardJob: Writing to S3 crs-data-export/{keypath}");
                                S3Helper.UploadToS3(stream, "crs-data-export", keypath, Constants.AppConfig.Application.KmsArn);
                                Log.Info($"UserCardJob: Successfully wrote crs-data-export/{keypath}");
                                stream.Close();
                                stream.Dispose();
                            }

                            try
                            {
                                if (S3Helper.ExistsInS3("crs-data-export", keypath) == true)
                                {
                                    Log.Info($"UserCardJob: Uploading crs-data-export/{keypath} to MySql for AggregationType: {(int)aggregationType}, Channel Digit: {i}, Loop: {loop}");
                                    try
                                    {
                                        using (var conn = DBManagerMysql.GetConnection(true))
                                        {
                                            using (var command = conn.GetCommand())
                                            {
                                                var quotechar = "\"";
                                                command.CommandText =
                                                $@"
                                                    load data from s3 's3://crs-data-export/{keypath}'
                                                    replace
                                                    into table {Constants.DatabaseSchema}microservice_staging_twitch_user_cards
                                                    character set 'utf8'
                                                    fields terminated by ','
                                                    enclosed by '{quotechar}'
                                                    lines terminated by '\r\n'
                                                    ignore 1 lines
                                                    (ChannelID, LastUpdateDate, AggregationType, MinutesWatchedData, AverageCcuData, ServiceId, BatchID)
                                                    ;
                                                ";
                                                command.ExecuteNonQueryWithMeasurements("user_card_job_load");
                                                Log.Info($"UserCardJob: Completed uploading crs-data-export/{keypath} to MySql for AggregationType: {(int)aggregationType}, Channel Digit: {i}, Loop: {loop}");
                                            }
                                        }
                                    }
                                    catch (Exception ex)
                                    {
                                        Log.Error($@"UserCardJob: {ex}");
                                    }
                                }
                            }
                            catch (Exception ex)
                            {
                                Log.Error($@"UserCardJob: {ex}");
                            }

                            loop++;
                        }

                        results = null;
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserCardJob: {ex}");
            }

            try
            {
                Log.Info($"UserCardJob: Updating AggregationType {(int)aggregationType}");

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            update {Constants.DatabaseSchema}microservice_twitch_user_listing_past_{(int)aggregationType}_days a
                            join {Constants.DatabaseSchema}microservice_staging_twitch_user_cards b 
                                on a.AggregationType = b.AggregationType
                                and a.ChannelID = b.ChannelID
                            set 
                                a.MinutesWatchedData = b.MinutesWatchedData,
                                a.AverageCCUData = b.AverageCCUData
                            where 
                                a.AggregationType = {(int)aggregationType}
                                and b.AggregationType = {(int)aggregationType}
                            ;
                        ";
                        command.ExecuteNonQueryWithMeasurements("user_card_job_update");
                    }
                }
                Log.Info($"UserCardJob: Completed updating AggregationType {(int)aggregationType}");
            }
            catch (Exception ex)
            {
                Log.Error($@"UserCardJob: {ex}");
            }
            return resultCount;
        }
    }
}
