﻿using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Extensions;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Models.DatabaseModels.RedshiftModels;
using Resonance.Core.Models.DatabaseModels.TwitchUserDetailsModels;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using MySql.Data.MySqlClient;

namespace Resonance.Jobs.Amp.TwitchUser
{
    public class UserDetailsJob : JobBase, IJob<long>
    {
        private static TwitchUserDetailsConfiguration jobConfig { get; set; } = null;


        public UserDetailsJob(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                jobConfig = new TwitchUserDetailsConfiguration()
                {
                    MinimumViewershipRequirement = 0,
                    MinimumDollarThreshold = 50
                };
                Log.Info($@"UserDetailsJob: configured. IsActive: {Config.IsActive}");
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }
        }

        public override void Run()
        {
            try
            {
                if (jobConfig != null)
                {
                    var rundate = DateTime.UtcNow.Date;

                    Log.Info($@"UserDetailsJob: Running.");
                    this.Config.IsRunning = true;
                    WipeOldData();
                    InsertMissingDetailsRecords();
                    UpdateRevenueData();
                    UpdateSubsData();
                    UpdateTimsStatus(rundate);
                    UpdateViewbotOverall(rundate);
                    UpdateViewbotData(rundate);
                    UpdatePathToAffiliateAndPartner(rundate);
                    UpdateTopCountries(rundate);
                    UpdateEmotes(rundate);
                    UpdateGameStats(rundate);
                    ReconcileData();
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }
            finally
            {
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromDays(1));
            }
            Log.Info($@"UserDetailsJob: Complete. Next Run Time: {this.Config.NextRunTime.Value.ToString("yyyy-MM-dd HH:mm:ss")}");
        }

        private void ReconcileData()
        {
            try
            {
                Log.Info($@"UserDetailsJob: Reconciling data");
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 8000;
                        command.CommandText =
                        $@"
                            delete from development.microservice_twitch_user_details
                            where not exists
                            (
	                            select ChannelID
	                            from
	                            (
		                            select ChannelID from development.microservice_twitch_user_listing_past_7_days
		                            union all
		                            select ChannelID from development.microservice_twitch_user_listing_past_30_days
		                            union all
		                            select ChannelID from development.microservice_twitch_user_listing_past_60_days
		                            union all
		                            select ChannelID from development.microservice_twitch_user_listing_past_90_days
		                            union all
		                            select ChannelID from development.microservice_twitch_user_listing_past_360_days
	                            ) as b
                                where microservice_twitch_user_details.ChannelID = b.ChannelID
	                            group by ChannelID
                            );
                        ";
                        command.ExecuteNonQueryWithMeasurements("user_details_reconcile");
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            Log.Info($@"UserDetailsJob: Reconciling data complete");
        }

        private void UpdatePathToAffiliateAndPartner(DateTime rundate)
        {
            string lookback = rundate.AddDays(-365).ToString("yyyy-MM-dd");
            var param = new Dictionary<string, dynamic>()
            {
                {"dollarthreshold", jobConfig.MinimumDollarThreshold },
                {"lookback", lookback },
            };

            var bucket = "crs-data-export";
            var keyfolderpath = $"{Constants.AppConfig.Application.Environment}/twitch-user-path-to-affiliate-partner/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/";
            Log.Info($"UserDetailsJob: Exporting Path to Affiliate / Partner data to s3");

            try
            {
                DBManagerRedshift.UnloadToS3
                (
                    bucket: bucket,
                    keyfolderpath: keyfolderpath,
                    wrappedsql: GetPathToAffiliateAndPartnerSql(param),
                    kmsarn: AppConfig.Data.Application.KmsArn,
                    timeout: 86400,
                    alternateConnection: DBManagerRedshift.GetTahoeConnectionString(),
                    options: new UnloadOptionsModel()
                    {
                        Header = "",
                        MaxFileSizeInMB = 100,
                        FileNamePart = "data_",
                        Delimiter = ",",
                        AddQuotes = "addquotes",
                        AllowOverwrite = "allowoverwrite",
                        Gzip = "",
                        Parallel = "on",
                        Escape = "escape",
                        Manifest = "manifest",
                        AutomaticallyReplaceQuotes = true
                    }
                );

                var fields = "(TwitchUserID,PathToAffiliateDate,PathToPartnerDate)";
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = AmpQuerySql.LoadFileFromS3
                    (
                        bucket: bucket,
                        keypath: $"{keyfolderpath}data_manifest",
                        table: $"{Constants.DatabaseSchema}microservice_staging_twitch_user_path_to",
                        fields: fields
                    ))
                    {
                        command.CommandTimeout = 86400;
                        command.Connection = conn;
                        command.ExecuteNonQueryWithMeasurements("user_details_job_path_to_load");

                        command.CommandText = GetPathToAffiliatePartnerUpdateSql();
                        command.ExecuteNonQueryWithMeasurements("user_details_job_path_to_update");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }

        private string GetPathToAffiliatePartnerUpdateSql()
        {
            var sql =
            $@"
                update {Constants.DatabaseSchema}microservice_twitch_user_details as t
                inner join {Constants.DatabaseSchema}microservice_staging_twitch_user_path_to as s
                    on t.ChannelID = s.TwitchUserID
                set
                    t.PathToAffiliateDate = s.PathToAffiliateDate,
                    t.PathToPartnerDate = s.PathToPartnerDate
                ;
            ";
            return sql;
        }

        private string GetPathToAffiliateAndPartnerSql(Dictionary<string, dynamic> param)
        {
            var sql =
            $@"
                with apdcs as
                (
                    select channel_id
                    from cubes.affiliates_partners_daily_channel_summary
                    group by channel_id
                ), path_affiliate as
                (
                  select channel_id, min(server_timestamp) as path_to_affiliate_date 
                  from spade.quest_completed as a
                  where 
                    quest_name = 'path_to_affiliate'
                  group by channel_id
                ),
                path_partner as
                (
                  select channel_id, min(server_timestamp) as path_to_partner_date 
                  from spade.quest_completed as a
                  where
                    quest_name = 'path_to_partner'
                  group by channel_id 
                )
                select apdcs.channel_id, path_affiliate.path_to_affiliate_date, path_partner.path_to_partner_date
                from apdcs
                left join path_affiliate
                on apdcs.channel_id = path_affiliate.channel_id
                left join path_partner
                on apdcs.channel_id = path_partner.channel_id
            ";
            return sql;
        }

        private void WipeOldData()
        {
            Log.Info($"UserDetailsJob: Wiping old data from staging table");
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText =
                    $@"
                        truncate table {Constants.DatabaseSchema}microservice_staging_twitch_user_tims_status;
                        truncate table {Constants.DatabaseSchema}microservice_staging_twitch_viewbot;
                        truncate table {Constants.DatabaseSchema}microservice_staging_twitch_viewbot_day;
                        truncate table {Constants.DatabaseSchema}microservice_staging_twitch_user_path_to;
                        truncate table {Constants.DatabaseSchema}microservice_staging_twitch_user_top_countries;
                        truncate table {Constants.DatabaseSchema}microservice_staging_twitch_user_emotes;
                        truncate table {Constants.DatabaseSchema}microservice_staging_twitch_user_game_stats;
                    ";
                    command.ExecuteNonQueryWithMeasurements("user_details_cleanup");
                }
            }
            Log.Info($"UserDetailsJob: Completed wiping old data from staging table");
        }

        private void InsertMissingDetailsRecords()
        {
            try
            {
                Log.Info($@"UserDetailsJob: Inserting / Updating Twitch User Details Missing Records");
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            update {Constants.DatabaseSchema}microservice_twitch_user_details as t
                            inner join {Constants.DatabaseSchema}microservice_twitch_user_listing_past_360_days as s
                                on t.ChannelID = s.ChannelID
                            set
                                t.MaxDate = s.MaxDate,
                                t.Login = s.Login,
                                t.CCUTier = s.CCUTier,
                                t.LastBroadcastDate = s.LastBroadcastDate,
                                t.UserType = s.UserType,
                                t.SubsTotal = s.SubsTotal,
                                t.FollowerCountTotal = s.FollowerCountTotal,
                                t.ProfileImage = s.ProfileImage,
                                t.AccountManagerUserName = s.AccountManager,
                                t.AccountManagerFullName = concat(s.AccountManagerFirstName, ' ', s.AccountManagerLastName),
                                t.MinutesWatchedData = s.MinutesWatchedData,
                                t.AverageCcuData = s.AverageCcuData
                            ;

                            insert into {Constants.DatabaseSchema}microservice_twitch_user_details
                            (
                                MaxDate, ChannelID, Login, CCUTier, LastBroadcastDate, UserType, SubsTotal,
                                FollowerCountTotal, ProfileImage, AccountManagerUserName, AccountManagerFullName,
                                MinutesWatchedData, AverageCcuData
                            )
                            select
                                MaxDate,
                                ChannelID,
                                Login,
                                CCUTier,
                                LastBroadcastDate,
                                UserType,
                                SubsTotal,
                                FollowerCountTotal,
                                ProfileImage,
                                AccountManager,
                                concat(AccountManagerFirstName, ' ', AccountManagerLastName),
                                MinutesWatchedData,
                                AverageCcuData
                            from {Constants.DatabaseSchema}microservice_twitch_user_listing_past_360_days as a
                            where not exists
                            (
                                select 1
                                from {Constants.DatabaseSchema}microservice_twitch_user_details as b
                                where a.ChannelID = b.ChannelID
                            );
                        ";
                        command.ExecuteNonQueryWithMeasurements("insert_update_details_rows", prepare: false, echoQuery: true);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }
        }

        private void UpdateTimsStatus(DateTime rundate)
        {
            try
            {
                string lookback = rundate.AddDays(-365).ToString("yyyy-MM-dd");
                var param = new Dictionary<string, dynamic>()
                {
                    {"dollarthreshold", jobConfig.MinimumDollarThreshold },
                    {"lookback", lookback },
                };
                var bucket = "crs-data-export";
                var keyfolderpath = $"{Constants.AppConfig.Application.Environment}/twitch-user-details-tims-status/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/";

                Log.Info($"UserDetailsJob: Exporting TIMS data to s3");
                DBManagerRedshift.UnloadToS3
                (
                    bucket: bucket,
                    keyfolderpath: keyfolderpath,
                    wrappedsql: GetUserTimsStatus(),
                    kmsarn: AppConfig.Data.Application.KmsArn,
                    timeout: 86400,
                    alternateConnection: DBManagerRedshift.GetTahoeConnectionString(),
                    options: new UnloadOptionsModel()
                    {
                        Header = "",
                        MaxFileSizeInMB = 100,
                        FileNamePart = "data_",
                        Delimiter = ",",
                        AddQuotes = "addquotes",
                        AllowOverwrite = "allowoverwrite",
                        Gzip = "",
                        Parallel = "on",
                        Escape = "escape",
                        Manifest = "manifest",
                        AutomaticallyReplaceQuotes = true
                    }
                );

                string fields = "(channel_id,form_type,isusperson,isuseractionrequired,status,withholdingrate,service_withholding_rate,royalty_withholding_rate)";
                Log.Info($"UserDetailsJob: Importing TIMS data to MySql from S3");
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = AmpQuerySql.LoadFileFromS3
                    (
                        bucket,
                        $"{keyfolderpath}data_manifest",
                        $"{Constants.DatabaseSchema}microservice_staging_twitch_user_tims_status",
                        fields: fields
                    ))
                    {
                        command.CommandTimeout = 86400;
                        command.Connection = conn;
                        command.ExecuteNonQueryWithMeasurements("user_details_job_tims_load");

                        command.CommandText = GetTimsUpdateSql();
                        command.ExecuteNonQueryWithMeasurements("user_details_job_tims_update");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }
        }

        private string GetTimsUpdateSql()
        {
            var sql =
            $@"
                update {Constants.DatabaseSchema}microservice_twitch_user_details as a
                inner join {Constants.DatabaseSchema}microservice_staging_twitch_user_tims_status as b
                    on a.ChannelID = b.channel_id
                set TimsFormType = b.form_type,
                    TimsIsUsPerson = b.isusperson,
                    TimsIsUserActionRequired = b.isuseractionrequired,
                    TimsStatus = b.status,
                    TimsWithholdingRate = b.withholdingrate,
                    TimsDataSourced = UTC_DATE(),
                    TimsServiceWithholdingRate = b.service_withholding_rate,
                    TimsRoyaltyWithholdingRate = b.royalty_withholding_rate
                ;
            ";
            return sql;
        }

        private long UpdateSubsData()
        {

            long resultCount = 0;
            DateTime maxDate = DateTime.UtcNow;
            DateTime minDate = maxDate.AddYears(-1);
            string timeSpan = "month";
            string sql = string.Empty;
            var checkdate = new DateTime(1901, 1, 1);
            try
            {
                var param = new Dictionary<string, dynamic>()
                {
                    { "@channeldigit", 0 },
                    { "@timespan", timeSpan },
                    { "@mindate", minDate.ToString("yyyy-MM-dd")}
                };
                for (var i = 0; i <= 9; i++)
                {
                    param["@channeldigit"] = i;
                    var loop = 0;

                    sql =
                    $@"
                        /* BEGIN TWITCH USER DETAILS SUB QUERY */
                        select
                            date_trunc(@timespan, a.day) as Day,
                            cast(channel_id as bigint) as TwitchUserID,
                            sum(subs_sold_total) as SubsTotal,
                            sum(subs_sold_prime) as PrimeSubsTotal,
                            sum(subs_sold_paid_total) as PaidSubsTotal,
                            sum(subs_sold_paid_non_gifted) as PaidNonGiftedSubs,
                            sum(subs_sold_paid_gifted) as PaidGiftedSubs,
                            sum(subs_sold_499_total) as Tier1SubsTotal,
                            sum(subs_sold_999_total) as Tier2SubsTotal,
                            sum(subs_sold_2499_total) as Tier3SubsTotal
                        from cubes.affiliates_partners_daily_channel_summary as a
                        where
                            a.day >= cast(@mindate as date)
                            and right(channel_id, 1) = @channeldigit
                            group by 
                                a.channel_id, 
                                date_trunc(@timespan, a.day)
                        /* END TWITCH USER DETAILS SUB QUERY */
                    ";
                    var results = DBManagerRedshift.GetSqlData<TwitchUserDetailsSubGraphQueryModel>(sql, parameters: param, timeout: 86400, useTahoe: true)?.ToArray();

                    var result = results.GroupBy(x => x.TwitchUserID)
                    .Select(group => new
                    {
                        ChannelID = group.Key,
                        LastUpdateDate = maxDate,
                        MonthlySubsData = JsonConvert.SerializeObject(group?.Select(x => new
                        {
                            x.Day,
                            SubsTotal = x.SubsTotal,
                            PrimeSubsTotal = x.PrimeSubsTotal,
                            PaidSubsTotal = x.PaidSubsTotal,
                            PaidNonGiftedSubs = x.PaidNonGiftedSubs,
                            PaidGiftedSubs = x.PaidGiftedSubs,
                            Tier1SubsTotal = x.Tier1SubsTotal,
                            Tier2SubsTotal = x.Tier2SubsTotal,
                            Tier3SubsTotal = x.Tier3SubsTotal
                        }).ToArray()),
                        ServiceID = ConstantsWorker.WorkerIdentifier,
                        BatchID = Guid.NewGuid().ToString("d")
                    })
                    ?.ToArray()
                    .Partition(10000).ToArray()
                    ;

                    if (results != null && results.Length > 0)
                    {
                        foreach (var data in result)
                        {
                            Log.Info($"UserDetailsJob: Found {results.Length} subs results. Writing CSV to stream.");
                            var keypath = $"{Constants.AppConfig.Application.Environment}/twitch-details-data/{maxDate.ToString("yyyy")}/{maxDate.ToString("MM")}/{maxDate.ToString("dd")}/{i}/subs-data-{loop}.csv";
                            using (var stream = new MemoryStream())
                            {
                                using (TextWriter writer = new StreamWriter(stream, Encoding.UTF8, bufferSize: 2048, leaveOpen: true))
                                {
                                    using (var csv = new CsvHelper.CsvWriter(writer, new CsvHelper.Configuration.Configuration()
                                    {
                                        Delimiter = ",",
                                        Quote = '"',
                                        ShouldQuote = (field, context) => true,
                                        PrepareHeaderForMatch = (header, context) => header.ToLower(),
                                        Encoding = Encoding.UTF8,
                                    },
                                    leaveOpen: true))
                                    {
                                        var records = data.Select(x => new
                                        {
                                            x.ChannelID,
                                            LastUpdateDate = x.LastUpdateDate.ToRedshiftDateFormat(),
                                            x.MonthlySubsData,
                                            x.ServiceID,
                                            x.BatchID
                                        }).ToArray();
                                        resultCount += records.Length;
                                        Log.Info($"UserDetailsJob: Adding {records.Length} records to CSV");
                                        csv.WriteRecords(records);
                                        records = null;
                                    }
                                }
                                stream.Position = 0;
                                Log.Info($"UserDetailsJob: Writing to S3 crs-data-export/{keypath}");
                                S3Helper.UploadToS3(stream, "crs-data-export", keypath, Constants.AppConfig.Application.KmsArn);
                                Log.Info($"UserDetailsJob: Successfully wrote crs-data-export/{keypath}");
                                stream.Close();
                                stream.Dispose();
                            }

                            try
                            {
                                if (S3Helper.ExistsInS3("crs-data-export", keypath) == true)
                                {
                                    Log.Info($"UserDetailsJob: Uploading crs-data-export/{keypath} to MySql for Channel Digit: {i}, Loop: {loop}");
                                    try
                                    {
                                        using (var conn = DBManagerMysql.GetConnection(true))
                                        {
                                            using (var command = conn.GetCommand())
                                            {
                                                var quotechar = "\"";
                                                command.CommandText =
                                                $@"
                                                    load data from s3 's3://crs-data-export/{keypath}'
                                                    replace
                                                    into table {Constants.DatabaseSchema}microservice_staging_twitch_user_details_subs
                                                    character set 'utf8'
                                                    fields terminated by ','
                                                    enclosed by '{quotechar}'
                                                    lines terminated by '\r\n'
                                                    ignore 1 lines
                                                    (ChannelID, LastUpdateDate, MonthlySubsData, ServiceId, BatchID)
                                                    ;
                                                ";
                                                command.ExecuteNonQueryWithMeasurements("user_details_staging_load");
                                                Log.Info($"UserDetailsJob: Completed uploading crs-data-export/{keypath} to MySql for Channel Digit: {i}, Loop: {loop}");
                                            }
                                        }
                                    }
                                    catch (Exception ex)
                                    {
                                        Log.Error($@"UserDetailsJob: {ex}");
                                    }
                                }
                            }
                            catch (Exception ex)
                            {
                                Log.Error($@"UserDetailsJob: {ex}");
                            }

                            loop++;
                        }

                        results = null;
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }

            try
            {

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            update {Constants.DatabaseSchema}microservice_twitch_user_details a
                            join {Constants.DatabaseSchema}microservice_staging_twitch_user_details_subs b 
                                on a.ChannelID = b.ChannelID
                            set 
                                a.MonthlySubsData = b.MonthlySubsData
                            ;
                        ";
                        command.ExecuteNonQueryWithMeasurements("user_details_update");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }
            return resultCount;
        }

        private long UpdateRevenueData()
        {

            long resultCount = 0;
            DateTime maxDate = DateTime.UtcNow;
            DateTime minDate = maxDate.AddYears(-1);
            string timeSpan = "month";
            string sql = string.Empty;
            var checkdate = new DateTime(1901, 1, 1);
            try
            {
                var param = new Dictionary<string, dynamic>()
                {
                    { "@channeldigit", 0 },
                    { "@timespan", timeSpan },
                    { "@mindate", minDate.ToString("yyyy-MM-dd")}
                };
                for (var i = 0; i <= 9; i++)
                {
                    param["@channeldigit"] = i;
                    var loop = 0;

                    sql =
                    $@"
                        /* BEGIN TWITCH USER DETAILS REVENUE QUERY */
                        select
                            date_trunc(@timespan, a.day) as Day,
                            cast(channel_id as bigint) as TwitchUserID,
                            sum(creator_bounty_board_revenue) as BountyBoardRevenue,
                            sum(creator_ad_revenue) as AdRevenue,
                            sum(creator_sub_revenue_total) as SubsRevenue,
                            sum(creator_bits_revenue_total) as BitsRevenue
                        from cubes.affiliates_partners_daily_channel_summary as a
                        where
                            a.day >= cast(@mindate as date)
                            and right(channel_id, 1) = @channeldigit
                            group by 
                                a.channel_id, 
                                date_trunc(@timespan, a.day)
                        /* END TWITCH USER DETAILS REVENUE QUERY */
                    ";
                    var results = DBManagerRedshift.GetSqlData<TwitchUserDetailsRevenueGraphQueryModel>(sql, parameters: param, timeout: 86400, useTahoe: true)?.ToArray();

                    var result = results.GroupBy(x => x.TwitchUserID)
                    .Select(group => new
                    {
                        ChannelID = group.Key,
                        LastUpdateDate = maxDate,
                        RevenueData = JsonConvert.SerializeObject(group?.Select(x => new
                        {
                            x.Day,
                            BountyBoardRevenue = x.BountyBoardRevenue,
                            AdRevenue = x.AdRevenue,
                            SubsRevenue = x.SubsRevenue,
                            BitsRevenue = x.BitsRevenue,
                        }).ToArray()),
                        ServiceID = ConstantsWorker.WorkerIdentifier,
                        BatchID = Guid.NewGuid().ToString("d")
                    })
                    ?.ToArray()
                    .Partition(10000).ToArray()
                    ;

                    if (results != null && results.Length > 0)
                    {
                        Log.Info($"UserDetailsJob: Found {results.Length} revenue results. Writing CSV to stream.");

                        foreach (var data in result)
                        {
                            var keypath = $"{Constants.AppConfig.Application.Environment}/twitch-details-data/{maxDate.ToString("yyyy")}/{maxDate.ToString("MM")}/{maxDate.ToString("dd")}/{i}/revenue-data-{loop}.csv";
                            using (var stream = new MemoryStream())
                            {
                                using (TextWriter writer = new StreamWriter(stream, Encoding.UTF8, bufferSize: 2048, leaveOpen: true))
                                {
                                    using (var csv = new CsvHelper.CsvWriter(writer, new CsvHelper.Configuration.Configuration()
                                    {
                                        Delimiter = ",",
                                        Quote = '"',
                                        ShouldQuote = (field, context) => true,
                                        PrepareHeaderForMatch = (header, context) => header.ToLower(),
                                        Encoding = Encoding.UTF8,
                                    },
                                    leaveOpen: true))
                                    {
                                        var records = data.Select(x => new
                                        {
                                            x.ChannelID,
                                            LastUpdateDate = x.LastUpdateDate.ToRedshiftDateFormat(),
                                            x.RevenueData,
                                            x.ServiceID,
                                            x.BatchID
                                        }).ToArray();
                                        resultCount += records.Length;
                                        Log.Info($"UserDetailsJob: Adding {records.Length} records to CSV");
                                        csv.WriteRecords(records);
                                        records = null;
                                    }
                                }
                                stream.Position = 0;
                                Log.Info($"UserDetailsJob: Writing to S3 crs-data-export/{keypath}");
                                S3Helper.UploadToS3(stream, "crs-data-export", keypath, Constants.AppConfig.Application.KmsArn);
                                Log.Info($"UserDetailsJob: Successfully wrote crs-data-export/{keypath}");
                                stream.Close();
                                stream.Dispose();
                            }

                            try
                            {
                                if (S3Helper.ExistsInS3("crs-data-export", keypath) == true)
                                {
                                    Log.Info($"UserDetailsJob: Uploading crs-data-export/{keypath} to MySql for Channel Digit: {i}, Loop: {loop}");
                                    try
                                    {
                                        using (var conn = DBManagerMysql.GetConnection(true))
                                        {
                                            using (var command = conn.GetCommand())
                                            {
                                                var quotechar = "\"";
                                                command.CommandText =
                                                $@"
                                                    load data from s3 's3://crs-data-export/{keypath}'
                                                    replace
                                                    into table {Constants.DatabaseSchema}microservice_staging_twitch_user_details_revenue
                                                    character set 'utf8'
                                                    fields terminated by ','
                                                    enclosed by '{quotechar}'
                                                    lines terminated by '\r\n'
                                                    ignore 1 lines
                                                    (ChannelID, LastUpdateDate, RevenueData, ServiceId, BatchID)
                                                    ;
                                                ";
                                                command.ExecuteNonQueryWithMeasurements("user_details_staging_revenue_load");
                                                Log.Info($"UserDetailsJob: Completed uploading crs-data-export/{keypath} to MySql for Channel Digit: {i}, Loop: {loop}");
                                            }
                                        }
                                    }
                                    catch (Exception ex)
                                    {
                                        Log.Error($@"UserDetailsJob: {ex}");
                                    }
                                }
                            }
                            catch (Exception ex)
                            {
                                Log.Error($@"UserDetailsJob: {ex}");
                            }

                            loop++;
                        }

                        results = null;
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }

            try
            {

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            update {Constants.DatabaseSchema}microservice_twitch_user_details a
                            join {Constants.DatabaseSchema}microservice_staging_twitch_user_details_revenue b 
                                on a.ChannelID = b.ChannelID
                            set 
                                a.RevenueData = b.RevenueData
                            ;
                        ";
                        command.ExecuteNonQueryWithMeasurements("user_Details_revenue_update");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }
            return resultCount;
        }

        private string GetUserTimsStatus()
        {
            var sql =
            $@"
                with apdcs as
                (
                    select cast(channel_id as bigint) as channel_id
                    from cubes.amp_application_population
                    group by cast(channel_id as bigint)
                ), royalty as
                (
                    select channel_id, effectivefrom, formtype, isusperson, isuseractionrequired, status, a.withholdingrate
                    from
                    (
                        select
                            c.id as channel_id,
                            a.effectivefrom,
                            a.formtype,
                            a.isusperson,
                            a.isuseractionrequired,
                            a.status,
                            a.withholdingrate,
                            row_number() over(partition by a.accountid order by a.effectivefrom desc) as sort
                        from paymentsdata.tax_status_updates_v1 as a
                        left join paymentsdata.payout_entities_v0 as b
                            on a.accountid = concat(cast(b.payoutentityid as varchar), ':ROY')
                        left join dbsnapshots.users as c
                            on b.ownerchannelid = c.id
                    ) as a
                    where sort = 1
                ), service as
                (
                    select channel_id, effectivefrom, formtype, isusperson, isuseractionrequired, status, a.withholdingrate
                    from
                    (
                        select
                            c.id as channel_id,
                            a.effectivefrom,
                            a.formtype,
                            a.isusperson,
                            a.isuseractionrequired,
                            a.status,
                            a.withholdingrate,
                            row_number() over(partition by a.accountid order by a.effectivefrom desc) as sort
                        from paymentsdata.tax_status_updates_v1 as a
                        left join paymentsdata.payout_entities_v0 as b
                            on a.accountid = concat(cast(b.payoutentityid as varchar), ':SER')
                        left join dbsnapshots.users as c
                            on b.ownerchannelid = c.id
                    ) as a
                    where sort = 1
                )
                select 
                    apdcs.channel_id,
                    case
                        when service.effectivefrom > royalty.effectivefrom
                        then service.formtype
                        else royalty.formtype
                    end as form_type,
                    case 
                        when case
                            when service.effectivefrom > royalty.effectivefrom
                            then service.isusperson
                            else royalty.isusperson
                        end = 'true'
                        then 1 else 0
                    end as isusperson,
                    case
                        when service.isuseractionrequired = 'true' or royalty.isuseractionrequired = 'true'
                        then 1 else 0 
                    end as isuseractionrequired,
                    case
                        when service.effectivefrom > royalty.effectivefrom
                        then service.status
                        else royalty.status
                    end as status,
                    case
                        when service.effectivefrom > royalty.effectivefrom
                        then service.withholdingrate
                        else royalty.withholdingrate
                    end as withholdingrate,
                    service.withholdingrate as service_withholding_rate,
                    royalty.withholdingrate as royalty_withholding_rate
                from apdcs
                left join royalty
                    on apdcs.channel_id = royalty.channel_id
                left join service
                    on apdcs.channel_id = service.channel_id
                ;
            ";
            return sql;
        }

        private void UpdateViewbotOverall(DateTime rundate)
        {
            try
            {
                var bucket = "crs-data-export";
                var keyfolderpath = $"{Constants.AppConfig.Application.Environment}/twitch-viewbot-overall-data/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/";
                DBManagerRedshift.UnloadToS3
                (
                    bucket: bucket,
                    keyfolderpath: keyfolderpath,
                    wrappedsql: GetViewbotOverallSql(rundate),
                    kmsarn: AppConfig.Data.Application.KmsArn,
                    timeout: 86400,
                    alternateConnection: DBManagerRedshift.GetTahoeConnectionString(),
                    options: new UnloadOptionsModel()
                    {
                        Header = "",
                        MaxFileSizeInMB = 100,
                        FileNamePart = "data_",
                        Delimiter = ",",
                        AddQuotes = "addquotes",
                        AllowOverwrite = "allowoverwrite",
                        Gzip = "",
                        Parallel = "on",
                        Escape = "escape",
                        Manifest = "manifest",
                        AutomaticallyReplaceQuotes = true
                    }
                );

                string fields =
                @"
                    (
                        ChannelID, BeforeAvg, MergedAvg, BestGuessAvg
                    )
                ";
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = AmpQuerySql.LoadFileFromS3
                    (
                        bucket: bucket,
                        keypath: $"{keyfolderpath}data_manifest",
                        table: $"{Constants.DatabaseSchema}microservice_staging_twitch_viewbot",
                        fields: fields
                    )
                    )
                    {
                        command.CommandTimeout = 86400;
                        command.Connection = conn;
                        command.ExecuteNonQueryWithMeasurements("user_details_staging_viewbot_load");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }

            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            update
                                {Constants.DatabaseSchema}microservice_twitch_viewbot t
                            inner join {Constants.DatabaseSchema}microservice_staging_twitch_viewbot s
                                on t.ChannelID = s.ChannelID
                            set
                                t.BeforeAvg = s.BeforeAvg,
                                t.MergedAvg = s.MergedAvg,
                                t.BestGuessAvg = s.BestGuessAvg
                            ;

                            insert into {Constants.DatabaseSchema}microservice_twitch_viewbot (ChannelID, BeforeAvg, MergedAvg, BestGuessAvg)
                            select ChannelID, BeforeAvg, MergedAvg, BestGuessAvg
                            from {Constants.DatabaseSchema}microservice_staging_twitch_viewbot as a
                            where not exists
                            (
                                select 1
                                from {Constants.DatabaseSchema}microservice_twitch_viewbot as b
                                where a.ChannelID = b.ChannelID
                            )
                            and a.ChannelID > 0
                            ;
                        ";
                        command.ExecuteNonQueryWithMeasurements("user_details_viewbot_update");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }
        }

        private string GetViewbotOverallSql(DateTime rundate)
        {
            string lookback = rundate.AddDays(-365).ToString("yyyy-MM-dd");
            DateTime mindate = rundate.AddDays(-60);
            string mindatestring = mindate.ToString("yyyy-MM-dd");
            string maxdatestring = rundate.ToString("yyyy-MM-dd");
            var sql =
            $@"
                /* Viewbot Overall Query */
                with apdcs as
                (
                    select cast(channel_id as bigint) as channel_id
                    from cubes.amp_application_population
                    group by cast(channel_id as bigint)
                ),
                viewbot as
                (
                  select
                    a.channel_id,
                    coalesce(avg(unfiltered), 0.0) as before_avg,
                    coalesce(avg(site_ccu), 0.0) as site_avg,
                    coalesce(avg(beta_ccu), 0.0) as merged_avg,
                    coalesce(avg(best_guess), 0.0) as best_avg,
                    coalesce(avg(suspected_bot_ratio_sessions), 0.0) as suspected_bot_ratio_sessions_avg,
                    coalesce(avg(suspected_bot_ratio_ips), 0.0) as suspected_bot_ratio_ips_avg 
                  from cubes.amp_viewbot_channel_best_guess_over_time_by_hour as a
                  where
                    timepart >= cast('{mindatestring}' as datetime)
                    and timepart < cast('{maxdatestring}' as datetime)
                  and exists
                  (
                    select 1
                    from apdcs as b
                    where a.channel_id = b.channel_id
                  )
                  group by
                    channel_id
                )
                select
                    a.channel_id as ChannelID,
                    a.before_avg as BeforeAvg,
                    a.merged_avg as MergedAvg,
                    a.best_avg as BestGuessAvg
                from viewbot as a
                ;
                /* End Viewbot Overall Query */
            ";
            return sql;
        }

        private void UpdateViewbotData(DateTime rundate)
        {
            Log.Info($@"UserDetailsJob: Updating Viewbot Data for {rundate.ToString("yyyy-MM-dd")}");
            var items = new Dictionary<long, List<TwitchViewbotDayModel>>();
            var bucket = "crs-data-export";
            for (var channelDigit = 0; channelDigit <= 9; channelDigit++)
            {
                items.Clear();

                try
                {
                    var keyfolderpath = $"{Constants.AppConfig.Application.Environment}/twitch-viewbot-day-data/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/";

                    using (var conn = DBManagerRedshift.TahoeConnection(true))
                    {
                        using (var command = conn.GetCommand())
                        {
                            command.CommandTimeout = 86400;
                            command.CommandText = GetViewbotDataSql(rundate, channelDigit);
                            Log.Verbose($@"UserDetailsJob: {command.CommandText}");
                            foreach (MySqlParameter p in command.Parameters)
                            {
                                command.Parameters.AddWithValue(p.ParameterName, p.Value);
                                Log.Verbose($@"{p.ParameterName}:{p.Value}:{p.Value.GetType()}");
                            }
                            using (var reader = new DataReaderWithMeasurements(command, null, "get_viewbot_data").NpgReader)
                            {
                                if (reader.HasRows)
                                {
                                    while (reader.Read())
                                    {
                                        var twitchUserID = long.Parse(reader.GetString(0));

                                        var item = new TwitchViewbotDayModel()
                                        {
                                            Day = reader.GetDateTime(1),
                                            BeforeGuessAvg = (float)reader.GetDouble(2),
                                            MergedGuessAvg = (float)reader.GetDouble(3),
                                            BestGuessAvg = (float)reader.GetDouble(4),
                                        };

                                        if (!items.ContainsKey(twitchUserID))
                                        {
                                            items.Add(twitchUserID, new List<TwitchViewbotDayModel>());
                                        }
                                        items[twitchUserID].Add(item);
                                    }
                                }
                            }
                        }
                    }

                    if (items.Count > 0)
                    {
                        var data = items.ToDictionary(x => x.Key, x => JsonConvert.SerializeObject(x.Value).GzipBase64Encode());
                        WriteViewbotDataToS3(bucket, $@"{keyfolderpath}ends_with_{channelDigit}/", channelDigit, data);
                    }
                }
                catch (Exception ex)
                {
                    Log.Error($@"UserDetailsJob: {ex}");
                }
            }
            items.Clear();
            items = null;
        }

        private string GetViewbotDataSql(DateTime rundate, int channelDigit)
        {
            string lookback = rundate.AddDays(-365).ToString("yyyy-MM-dd");
            DateTime mindate = rundate.AddDays(-60);
            string mindatestring = mindate.ToString("yyyy-MM-dd");
            string maxdatestring = rundate.ToString("yyyy-MM-dd");
            var sql =
            $@"
                with apdcs as
                (
                    select cast(channel_id as bigint) as channel_id
                    from cubes.amp_application_population
                    group by cast(channel_id as bigint)
                )
                select
                    a.channel_id as ChannelID,
                    date_trunc('day', a.timepart) as Day,
                    coalesce(avg(unfiltered), 0.0) as BeforeGuessAvg,
                    coalesce(avg(beta_ccu), 0.0) as MergedGuessAvg,
                    coalesce(avg(best_guess), 0.0) as BestGuessAvg
                from cubes.amp_viewbot_channel_best_guess_over_time_by_hour as a
                inner join apdcs as b
                    on a.channel_id = b.channel_id
                where
                    timepart >= cast('{mindatestring}' as datetime)
                    and timepart < cast('{maxdatestring}' as datetime)
                    and right(a.channel_id, 1) = {channelDigit}
                group by 
                    a.channel_id,
                    date_trunc('day', a.timepart)
                ;
                /* End Viewbot Data Query */
            ";
            return sql;
        }

        private void WriteViewbotDataToS3(string bucket, string keyfolderpath, int channelDigit, Dictionary<long, string> data)
        {
            try
            {
                var partitionItems = data
                    .Select(x => new
                    {
                        TwitchUserID = x.Key,
                        Data = x.Value
                    })
                    .ToArray()
                    .Partition(10000)
                    .ToArray();

                Log.Info($@"UserDetailsJob: Writing Viewbot Data {bucket}/{keyfolderpath} To S3 - Batches: {partitionItems.Length}");
                for (var i = 0; i < partitionItems.Length; i++)
                {
                    var keypath = $@"{keyfolderpath}data_part_{i}.csv";

                    using (var stream = new MemoryStream())
                    {
                        using (TextWriter writer = new StreamWriter(stream, Encoding.UTF8, 2048, leaveOpen: true))
                        {
                            using (var csv = new CsvHelper.CsvWriter(writer, new CsvHelper.Configuration.Configuration()
                            {
                                Delimiter = ",",
                                Quote = '"',
                                ShouldQuote = (field, context) => true,
                                Encoding = Encoding.UTF8,
                                HasHeaderRecord = false
                            }, leaveOpen: true))
                            {
                                csv.WriteRecords(partitionItems[i]);
                            }
                        }
                        stream.Position = 0;
                        Log.Info($@"UserDetailsJob: Uploading {bucket}/{keypath} to s3");
                        S3Helper.UploadToS3(stream, bucket, keypath);

                        LoadViewbotDataS3FileToMysqlStaging(bucket, keypath, channelDigit);
                    }
                }

                partitionItems = null;
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }
            finally
            {
                data.Clear();
            }
        }

        private void LoadViewbotDataS3FileToMysqlStaging(string bucket, string keypath, int channelDigit)
        {
            string fields = $@"(ChannelID, Data)";
            if (S3Helper.ExistsInS3(bucket, keypath) == true)
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = AmpQuerySql.LoadFileFromS3
                        (bucket,
                        $"{keypath}",
                        $"{Constants.DatabaseSchema}microservice_staging_twitch_viewbot_day",
                        ignore: "",
                        fields: fields)
                    )
                    {
                        command.Parameters.AddWithValue($@"@channeldigit", channelDigit);
                        command.CommandTimeout = 86400;
                        command.Connection = conn;
                        command.ExecuteNonQueryWithMeasurements("staging_viewbot_load_by_digit");
                    }

                    Log.Info($@"UserDetailsJob: Updating day data for microservice_twitch_viewbot channel digit {channelDigit}");
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            update {Constants.DatabaseSchema}microservice_twitch_viewbot t
                            inner join {Constants.DatabaseSchema}microservice_staging_twitch_viewbot_day s
                                on t.ChannelID = s.ChannelID
                                and right(s.ChannelID, 1) = @channeldigit
                            set
                                t.Data = s.Data
                            where
                                t.ChannelID = s.ChannelID
                                and right(s.ChannelID, 1) = @channeldigit
                                and t.ChannelID > 0
                        ";
                        command.Parameters.AddWithValue("@channelDigit", channelDigit);
                        command.ExecuteNonQueryWithMeasurements("update_microservice_twitch_viewbot");
                    }
                }
            }
        }

        private void UpdateTopCountries(DateTime rundate)
        {
            try
            {
                Log.Info($@"UserDetailsJobs: Loading Top Countries");
                var bucket = "crs-data-export";
                var keyfolderpath = $"{Constants.AppConfig.Application.Environment}/twitch-top-country-data/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/";

                DBManagerRedshift.UnloadToS3
                (
                    bucket: bucket,
                    keyfolderpath: keyfolderpath,
                    wrappedsql: GetChannelTopCountrySql(rundate),
                    kmsarn: AppConfig.Data.Application.KmsArn,
                    timeout: 86400,
                    alternateConnection: DBManagerRedshift.GetTahoeConnectionString(),
                    options: new UnloadOptionsModel()
                    {
                        Header = "",
                        MaxFileSizeInMB = 100,
                        FileNamePart = "data_",
                        Delimiter = ",",
                        AddQuotes = "addquotes",
                        AllowOverwrite = "allowoverwrite",
                        Gzip = "",
                        Parallel = "on",
                        Escape = "escape",
                        Manifest = "manifest",
                        AutomaticallyReplaceQuotes = true
                    }
                );

                string fields =
                @"
                    (channel_id, country_code, country_name, total_count, total_count_known, total_count_unknown, total_known_percent, total_unknown_percent, country_percent, rank, last_update_date)
                ";

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = AmpQuerySql.LoadFileFromS3
                    (
                        bucket: bucket,
                        keypath: $"{keyfolderpath}data_manifest",
                        table: $"{Constants.DatabaseSchema}microservice_staging_twitch_user_top_countries",
                        fields: fields
                    )
                    )
                    {
                        command.CommandTimeout = 86400;
                        command.Connection = conn;
                        command.ExecuteNonQueryWithMeasurements("user_details_top_countries");
                    }

                    Log.Info($@"UserDetailsJob: Updating microservice_twitch_user_top_countries");
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            replace into {Constants.DatabaseSchema}microservice_twitch_user_top_countries
                            (
                                channel_id, 
                                country_code, 
                                country_name, 
                                total_count, 
                                total_count_known, 
                                total_count_unknown, 
                                total_known_percent, 
                                total_unknown_percent, 
                                country_percent, 
                                rank, 
                                last_update_date
                            )
                            select
                                channel_id, 
                                country_code, 
                                country_name, 
                                total_count, 
                                total_count_known, 
                                total_count_unknown, 
                                total_known_percent, 
                                total_unknown_percent, 
                                country_percent, 
                                rank, 
                                last_update_date
                            from {Constants.DatabaseSchema}microservice_staging_twitch_user_top_countries
                            ;
                        ";
                        command.ExecuteNonQueryWithMeasurements("user_details_update_top_countries");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserDetailsJob: {ex}");
            }
            Log.Info($@"UserDetailsJobs: Completed Top Countries");
        }

        private string GetChannelTopCountrySql(DateTime rundate)
        {
            var sql =
            $@"
                with base_channels as
                (
                  select channel_id::bigint as channel_id
                  from cubes.amp_channel_country_ranking_top5
                  group by channel_id
                ),
                ranks as
                (
                  select channel_id, 1::int as rank from base_channels
                  union all select channel_id, 2::int as rank from base_channels
                  union all select channel_id, 3::int as rank from base_channels
                  union all select channel_id, 4::int as rank from base_channels
                  union all select channel_id, 5::int as rank from base_channels
                )
                select
                  cast(a.channel_id as bigint) as channel_id, 
                  coalesce(b.country_code, 'Unknown') as country_code, 
                  coalesce(b.country_name, 'Unknown') as country_name, 
                  b.total_count, 
                  b.total_count_known, 
                  b.total_count_unknown, 
                  b.total_known_percent, 
                  b.total_unknown_percent, 
                  b.country_percent, 
                  a.rank, 
                  current_timestamp as last_update_date
                from ranks as a
                left join cubes.amp_channel_country_ranking_top5 as b
                  on a.channel_id = b.channel_id
                  and a.rank = b.rank
                ;            
            ";
            return sql;
        }

        private void UpdateEmotes(DateTime rundate)
        {
            try
            {
                Log.Info($@"UserDetailsJobs: Loading Emotes");
                var bucket = "crs-data-export";
                var keyfolderpath = $"{Constants.AppConfig.Application.Environment}/twitch-user-emotes/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/";

                DBManagerRedshift.UnloadToS3
                (
                    bucket: bucket,
                    keyfolderpath: keyfolderpath,
                    wrappedsql: GetChannelEmotesSql(),
                    kmsarn: AppConfig.Data.Application.KmsArn,
                    timeout: 86400,
                    alternateConnection: DBManagerRedshift.GetTahoeConnectionString(),
                    options: new UnloadOptionsModel()
                    {
                        Header = "",
                        MaxFileSizeInMB = 100,
                        FileNamePart = "data_",
                        Delimiter = ",",
                        AddQuotes = "addquotes",
                        AllowOverwrite = "allowoverwrite",
                        Gzip = "",
                        Parallel = "on",
                        Escape = "escape",
                        Manifest = "manifest",
                        AutomaticallyReplaceQuotes = true
                    }
                );

                string fields =
                @"
                    (ChannelID, Regex, TierType, ProductType, IconType, BaseUrl, SmallUrlFragment, MediumUrlFragment, LargeUrlFragment)
                ";

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = AmpQuerySql.LoadFileFromS3
                    (
                        bucket: bucket,
                        keypath: $"{keyfolderpath}data_manifest",
                        table: $"{Constants.DatabaseSchema}microservice_staging_twitch_user_emotes",
                        fields: fields
                    )
                    )
                    {
                        command.CommandTimeout = 86400;
                        command.Connection = conn;
                        command.ExecuteNonQueryWithMeasurements("user_details_emotes");
                    }

                    Log.Info($@"UserDetailsJob: Updating microservice_twitch_user_top_countries");
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            set session transaction isolation level read uncommitted;
                            replace into {Constants.DatabaseSchema}microservice_twitch_user_emotes
                            (
                                ChannelID,
                                Regex,
                                TierType,
                                ProductType,
                                IconType,
                                BaseUrl,
                                SmallUrlFragment,
                                MediumUrlFragment,
                                LargeUrlFragment
                            )
                            select
                                ChannelID,
                                Regex,
                                TierType,
                                ProductType,
                                IconType,
                                BaseUrl,
                                SmallUrlFragment,
                                MediumUrlFragment,
                                LargeUrlFragment
                            from {Constants.DatabaseSchema}microservice_staging_twitch_user_emotes
                            ;

                            delete a
                            from {Constants.DatabaseSchema}microservice_twitch_user_emotes as a
                            where not exists
                            (
                                select 1
                                from {Constants.DatabaseSchema}microservice_staging_twitch_user_emotes as b
                                where a.ChannelID = b.ChannelID
                                and a.Regex = b.Regex
                            );
                        ";
                        command.ExecuteNonQueryWithMeasurements("user_details_update_emotes");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }
        
        private string GetChannelEmotesSql()
        {
            var sql =
            $@"
                with emotes as
                (
                    select
                        emoticons.ticket_product_owner_id as channel_id,
                        case
                            when tickets.ticket_type = 'chansub' and tickets.interval_unit = 'month' and tickets.recurring = true and tickets.default_price = 999 and right (tickets.short_name,5) = '_2000' then 'tier_2'
                            when tickets.ticket_type = 'chansub' and tickets.interval_unit = 'month' and tickets.recurring = true and tickets.default_price = 2499 and right (tickets.short_name,5) = '_3000' then 'tier_3'
                            when tickets.ticket_type = 'chansub' and tickets.interval_unit = 'month' and tickets.recurring = true and tickets.default_price = 499 then 'tier_1'
                            else 'custom'
                        end AS tier_type,
                        emoticons.ticket_product_type,
                        'emote' as icon_type,
                        substring(emoticons.emoticon_url, 0, (length(emoticons.emoticon_url) + 2) - strpos(reverse(emoticons.emoticon_url), '/')) as base_url,
                        '1.0' as small_url_fragment,
                        '2.0' as medium_url_fragment,
                        '3.0' as large_url_fragment,
                        emoticons.regex_display
                    from cubes.emoticons_enriched as emoticons
                    inner join dbsnapshots.ticket_products as tickets
                        on emoticons.ticket_product_id = tickets.id
                    where
                        emoticons.ticket_product_availability_state = 'active'
                        and emoticons.state = 'active'
                )
                select
                    cast(emotes.channel_id as bigint) as channel_id,
                    emotes.regex_display,
                    emotes.tier_type,
                    emotes.ticket_product_type as product_type,
                    emotes.icon_type,
                    emotes.base_url,
                    emotes.small_url_fragment,
                    emotes.medium_url_fragment,
                    emotes.large_url_fragment
                from emotes
                where exists
                (
                    select 1
                    from cubes.amp_application_population
                    where emotes.channel_id = amp_application_population.channel_id
                )
                ;
            ";
            return sql;
        }

        private void UpdateGameStats(DateTime rundate)
        {
            try
            {
                Log.Info($@"UserDetailsJobs: Loading Game Stats");
                var bucket = "crs-data-export";
                var keyfolderpath = $"{Constants.AppConfig.Application.Environment}/twitch-channel-game-stats/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/";

                var aggType = new int[]
                {
                    7,
                    30,
                    60,
                    90,
                    360
                };
                foreach(var aggregationType in aggType)
                {
                    var aggregation_keyfolderpath = $@"{keyfolderpath}{aggregationType}/";
                    DBManagerRedshift.UnloadToS3
                    (
                        bucket: bucket,
                        keyfolderpath: aggregation_keyfolderpath,
                        wrappedsql: GetChannelGameStatsSql(aggregationType),
                        kmsarn: AppConfig.Data.Application.KmsArn,
                        timeout: 86400,
                        alternateConnection: DBManagerRedshift.GetTahoeConnectionString(),
                        options: new UnloadOptionsModel()
                        {
                            Header = "",
                            MaxFileSizeInMB = 100,
                            FileNamePart = "data_",
                            Delimiter = ",",
                            AddQuotes = "addquotes",
                            AllowOverwrite = "allowoverwrite",
                            Gzip = "",
                            Parallel = "on",
                            Escape = "escape",
                            Manifest = "manifest",
                            AutomaticallyReplaceQuotes = true
                        }
                    );

                    string fields =
                    @"
                        (ChannelID, AggregationType, MinDate, MaxDate, UniqueDaysFound, UniqueDaysPossible, GameName, GameId, GiantbombId, MinutesBroadcastTotal, MinutesWatchedTotal, GameStartCount, PeakViewership, AverageViewership)
                    ";

                    using (var conn = DBManagerMysql.GetConnection(true))
                    {
                        using (var command = AmpQuerySql.LoadFileFromS3
                        (
                            bucket: bucket,
                            keypath: $"{aggregation_keyfolderpath}data_manifest",
                            table: $"{Constants.DatabaseSchema}microservice_staging_twitch_user_game_stats",
                            fields: fields
                        ))
                        {
                            command.CommandTimeout = 86400;
                            command.Connection = conn;
                            command.ExecuteNonQueryWithMeasurements("user_details_staging_game_stats");
                        }

                        Log.Info($@"UserDetailsJob: Updating microservice_twitch_user_game_stats");
                        using (var command = conn.GetCommand())
                        {
                            command.CommandTimeout = 86400;
                            command.CommandText =
                            $@"
                                set session transaction isolation level read uncommitted;
                                replace into {Constants.DatabaseSchema}microservice_twitch_user_game_stats
                                (
                                    ChannelID, 
                                    AggregationType, 
                                    MinDate, 
                                    MaxDate, 
                                    UniqueDaysFound, 
                                    UniqueDaysPossible, 
                                    GameName, 
                                    GameId, 
                                    GiantbombId, 
                                    MinutesBroadcastTotal, 
                                    MinutesWatchedTotal, 
                                    GameStartCount, 
                                    PeakViewership, 
                                    AverageViewership
                                )
                                select
                                    ChannelID, 
                                    AggregationType, 
                                    MinDate, 
                                    MaxDate, 
                                    UniqueDaysFound, 
                                    UniqueDaysPossible, 
                                    GameName, 
                                    GameId, 
                                    GiantbombId, 
                                    MinutesBroadcastTotal, 
                                    MinutesWatchedTotal, 
                                    GameStartCount, 
                                    PeakViewership, 
                                    AverageViewership
                                from {Constants.DatabaseSchema}microservice_staging_twitch_user_game_stats
                                ;

                                delete a
                                from {Constants.DatabaseSchema}microservice_twitch_user_game_stats as a
                                where MaxDate < (select max(MaxDate) from {Constants.DatabaseSchema}microservice_staging_twitch_user_game_stats)
                                and AggregationType = @aggregationType
                                ;
                            ";
                            command.Parameters.AddWithValue("@aggregationType", aggregationType);
                            command.ExecuteNonQueryWithMeasurements("user_details_update_aggregation_type");
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }

        private string GetChannelGameStatsSql(int aggregationType)
        {
            var sql =
            $@"
                select
                    channel_id as ChannelID,
                    @aggregationType as AggregationType,
                    dateadd(day, -@aggregationType, date_trunc('day', getdate())) as MinDate,
                    date_trunc('day', getdate()) as MaxDate,
                    count(distinct day) as UniqueDaysFound,
                    @aggregationType as UniqueDaysPossible,
                    game_name as GameName,
                    game_id as GameId,
                    giantbomb_id as GiantbombId,
                    sum(minutes_broadcast_total) as MinutesBroadcastTotal,
                    sum(minutes_watched_total) as MinutesWatchedTotal,
                    sum(game_start_count) as GameStartCount,
                    max(peak_viewership) as PeakViewership,
                    (sum(minutes_watched_total::float) / sum(item_count_to_find_average_with_minutes_watched_total_over_time)::float) as average_viewership  
                from cubes.amp_channel_game_statistics
                where
                    day >= dateadd(day, -@aggregationType, date_trunc('day', getdate()))
                    and day < date_trunc('day', getdate())
                group by
                    channel_id,
                    dateadd(day, -@aggregationType, date_trunc('day', getdate())),
                    date_trunc('day', getdate()),
                    game_name,
                    game_id,
                    giantbomb_id
                ;
            ";
            sql = sql.Replace("@aggregationType", aggregationType.ToString());
            return sql;
        }
    }
}
