﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using Npgsql;
using Resonance.Core;
using Resonance.Core.Extensions;
using Resonance.Core.Helpers.AggregationHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Models.DatabaseModels.TwitchUserListingModels;
using Resonance.Microservices.Queries;
using static Resonance.Core.Constants;

namespace Resonance.Jobs.Amp.TwitchUser
{
    public class UserListingJob : JobBase, IJob<long>
    {
        private static ConcurrentDictionary<long, DateTime> cache { get; set; } = null;
        private static TwitchUserListingConfiguration jobConfig { get; set; } = null;

        public UserListingJob(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                jobConfig = new TwitchUserListingConfiguration()
                {
                    MinimumViewershipRequirement = 0,
                    RequiredAggregates = new AggregationType[5]
                    {
                        AggregationType.Week,
                        AggregationType.ThirtyDay,
                        AggregationType.SixtyDay,
                        AggregationType.NintyDay,
                        AggregationType.BankerYear
                    },
                    MinimumDollarThreshold = 50
                };
                cache = new ConcurrentDictionary<long, DateTime>();
                Log.Info($@"UserListingJob: configured. IsActive: {Config.IsActive}");
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }

        public override void Run()
        {
            try
            {
                Log.Info($@"UserListingJob: Running.");

                this.Config.IsRunning = true;
                if(jobConfig != null && jobConfig.RequiredAggregates != null && jobConfig.RequiredAggregates.Length > 0)
                {
                    var rundate = DateTime.UtcNow.Date;
                    WipeOldData();

                    Log.Info($@"UserListingJob: Preparing the following aggregates: {string.Join(",", jobConfig.RequiredAggregates)}");
                    foreach (var aggregationType in jobConfig.RequiredAggregates)
                    {
                        string superlookback = rundate.AddDays(-1825).ToString("yyyy-MM-dd");
                        DateTime mindate = rundate.AddDays(-(int)aggregationType);
                        string lookback = rundate.AddDays(-365).ToString("yyyy-MM-dd");
                        string mindatestring = mindate.ToString("yyyy-MM-dd");
                        string maxdatestring = rundate.ToString("yyyy-MM-dd");
                        string maxday = DateTime.UtcNow.ToString("yyyy-MM-dd");

                        var param = new Dictionary<string, dynamic>()
                        {
                            {"@aggregationtype", (int)aggregationType},
                            {"@mindatestring", mindatestring },
                            {"@maxdatestring", maxdatestring },
                            {"@dollarthreshold", jobConfig.MinimumDollarThreshold },
                            {"@superlookback", superlookback },
                            {"@lookback", lookback },
                            {"@channeldigit", 0 }
                        };

                        var results = new List<TwitchUserListingModel>();

                        var sqlScripts = GetExecuteSql(aggregationType, rundate);


                        string prefixSql = sqlScripts.Item1;
                        string mainSql = sqlScripts.Item2;
                        foreach (var paramEntry in param)
                        {
                            prefixSql = prefixSql.Replace(paramEntry.Key, paramEntry.Value.ToString());
                            mainSql = mainSql.Replace(paramEntry.Key, paramEntry.Value.ToString());
                        }

                        var bucket = "crs-data-export";
                        var keypath = $"{Constants.AppConfig.Application.Environment}/twitch-user-listings/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/";
                        DBManagerRedshift.UnloadToS3
                        (
                            bucket: bucket,
                            keyfolderpath: keypath,
                            wrappedsql: mainSql,
                            kmsarn: AppConfig.Data.Application.KmsArn,
                            timeout: 86400,
                            prefixsql: prefixSql,
                            alternateConnection: DBManagerRedshift.GetTahoeConnectionString(),
                            options: new Core.Models.DatabaseModels.RedshiftModels.UnloadOptionsModel()
                            {
                                Header = "",
                                MaxFileSizeInMB = 100,
                                FileNamePart = "data_",
                                Delimiter = ",",
                                AddQuotes = "addquotes",
                                AllowOverwrite = "allowoverwrite",
                                Gzip = "",
                                Parallel = "on",
                                Escape = "escape",
                                Manifest = "manifest",
                                AutomaticallyReplaceQuotes = true
                            }
                        );

                        string fields =
                        @"
                            (
                                MinDate,MaxDate,AggregationType,ChannelID,Login,CcuTier,LastBroadcastDate,UserType,UserTypeChanged,
                                MaxCcu,AverageCcu,MinutesWatchedTotal,ChannelConcurrentsMinutesWatchedTotal,MinutesBroadcastTotal,
                                ChannelConcurrentsMinutesBroadcastTotal,UniqueDaysBroadcast,FollowerCountTotal,FollowerCountChangeTotal,
                                ChatMessagesTotal,SubsSoldTotal,SubsSoldPrimeTotal,SubsSoldTier1Total,SubsSoldTier2Total,SubsSoldTier3Total,
                                CreatorSubRevenueTotal,CreatorPrimeSubRevenueTotal,CreatorAdRevenueTotal,CreatorBitsRevenueTotal,CreatorFuelRevenueTotal,
                                CreatorBountyBoardRevenueTotal,CreatorRevenueTotal,VideoPlaysTotal,PrimaryBroadcastLanguage,PrimaryBroadcastCountry,
                                TopGameBroadcast,SubsTotal,ProfileImage,Region,AccountManager,AccountManagerFirstName,AccountManagerLastName,ValueScoreOverallWeighted,GameCount,
                                MostViewedPlatform,PlatformPercentWatchedOther,PlatformPercentWatchedMobile,PlatformPercentWatchedChromecast,
                                PlatformPercentWatchedWeb,PlatformPercentWatchedConsole,PlatformPercentWatchedDesktop,IsPremium, TimsUserActionRequired
                            )
                        ";
                        using (var conn = DBManagerMysql.GetConnection(true))
                        {
                            using (var command = AmpQuerySql.LoadFileFromS3
                            (
                                bucket: bucket,
                                keypath: $"{keypath}data_manifest",
                                table: $"{Constants.DatabaseSchema}microservice_staging_twitch_user_listing",
                                fields: fields
                            )
                            )
                            {
                                command.CommandTimeout = 86400;
                                command.Connection = conn;
                                command.ExecuteNonQueryWithMeasurements("user_listing_staging");
                            }
                        }



                        try
                        {
                            using (var conn = DBManagerMysql.GetConnection(true))
                            {
                                using
                                (
                                    var command =
                                        AmpQuerySql.InsertUpdateEtlTracking
                                        (
                                            source: "Redshift_Microservice_Twitch_User_Listing",
                                            target: $"Microservice_Twitch_User_Listing_Past_{(int)aggregationType}_Days",
                                            method: "Jobs.UserListingJob",
                                            timestamp: DateTime.UtcNow,
                                            rowcount: results.Count
                                        )
                                )
                                {
                                    command.Connection = conn;
                                    command.CommandTimeout = 0;
                                    command.ExecuteNonQueryWithMeasurements("update_user_listing_etl");
                                }
                                results.Clear();
                                results = null;
                            }
                        }
                        catch (Exception ex)
                        {
                            Log.Error($@"UserListingJob: {ex}");
                        }

                        UpdateTwitchUserListingByAggregationTypeFromStaging(aggregationType);

                        try
                        {
                            // IMP-475: Delete old data to make sure we're following privacy compliance
                            using (var conn = DBManagerMysql.GetConnection(true))
                            {
                                using (var command = conn.GetCommand())
                                {
                                    command.CommandText =
                                    $@"
                                        delete from {Constants.DatabaseSchema}microservice_twitch_user_listing_past_{(int)aggregationType}_days
                                        where not exists
                                        (
	                                        select 1
                                            from {Constants.DatabaseSchema}microservice_staging_twitch_user_listing
                                            where microservice_twitch_user_listing_past_{(int)aggregationType}_days.ChannelID = microservice_staging_twitch_user_listing.ChannelID
                                            and microservice_staging_twitch_user_listing.aggregationtype = @aggregationType
                                        )
                                        ;
                                    ";
                                    command.Parameters.AddWithValue("@aggregationType", (int)aggregationType);
                                    command.ExecuteNonQueryWithMeasurements("purge_amp_user_listing", true);
                                }
                            }
                        }
                        catch (Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserListingJob: {ex}");
            }
            finally
            {
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromDays(1));
            }
            Log.Info($@"UserListingJob: Complete. Next Run Time: {this.Config.NextRunTime.Value.ToString("yyyy-MM-dd HH:mm:ss")}");
        }

        private void WipeOldData()
        {
            Log.Info($"UserListingJob: Wiping old user listing data from staging table");
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"truncate table {Constants.DatabaseSchema}microservice_staging_twitch_user_listing;";
                    command.ExecuteNonQueryWithMeasurements("user_listing_staging_cleanup");
                }
            }
            Log.Info($"UserListingJob: Completed wiping old user listing data from staging table");
        }

        private void UpdateTwitchUserListingByAggregationTypeFromStaging(AggregationType aggregationType)
        {
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            update 
                                {Constants.DatabaseSchema}microservice_twitch_user_listing_past_{(int)aggregationType}_days t
                            inner join {Constants.DatabaseSchema}microservice_staging_twitch_user_listing as s
                                on t.AggregationType = s.AggregationType
	                            and t.ChannelID = s.ChannelID
	                        set
		                        t.MinDate = s.MinDate,
		                        t.MaxDate = s.MaxDate,
		                        t.Login = s.Login,
		                        t.CCUTier = s.CCUTier,
		                        t.LastBroadcastDate = s.LastBroadcastDate,
		                        t.UserType = s.UserType,
		                        t.UserTypeChanged = s.UserTypeChanged,
		                        t.MaxCCU = s.MaxCCU,
		                        t.AverageCCU = s.AverageCCU,
		                        t.MinutesWatchedTotal = s.MinutesWatchedTotal,
		                        t.ChannelConcurrentsMinutesWatchedTotal = s.ChannelConcurrentsMinutesWatchedTotal,
		                        t.MinutesBroadcastTotal = s.MinutesBroadcastTotal,
		                        t.ChannelConcurrentsMinutesBroadcastTotal = s.ChannelConcurrentsMinutesBroadcastTotal,
		                        t.UniqueDaysBroadcast = s.UniqueDaysBroadcast,
		                        t.FollowerCountTotal = s.FollowerCountTotal,
		                        t.FollowerCountChangeTotal = s.FollowerCountChangeTotal,
		                        t.ChatMessagesTotal = s.ChatMessagesTotal,
		                        t.SubsSoldTotal = s.SubsSoldTotal,
		                        t.SubsSoldPrimeTotal = s.SubsSoldPrimeTotal,
		                        t.SubsSoldTier1Total = s.SubsSoldTier1Total,
		                        t.SubsSoldTier2Total = s.SubsSoldTier2Total,
		                        t.SubsSoldTier3Total = s.SubsSoldTier3Total,
		                        t.CreatorSubRevenueTotal = s.CreatorSubRevenueTotal,
		                        t.CreatorPrimeSubRevenueTotal = s.CreatorPrimeSubRevenueTotal,
		                        t.CreatorAdRevenueTotal = s.CreatorAdRevenueTotal,
		                        t.CreatorBitsRevenueTotal = s.CreatorBitsRevenueTotal,
		                        t.CreatorFuelRevenueTotal = s.CreatorFuelRevenueTotal,
		                        t.CreatorBountyBoardRevenueTotal = s.CreatorBountyBoardRevenueTotal,
		                        t.CreatorRevenueTotal = s.CreatorRevenueTotal,
		                        t.VideoPlaysTotal = s.VideoPlaysTotal,
		                        t.PrimaryBroadcastLanguage = s.PrimaryBroadcastLanguage,
		                        t.PrimaryBroadcastCountry = s.PrimaryBroadcastCountry,
		                        t.TopGameBroadcast = s.TopGameBroadcast,
		                        t.SubsTotal = s.SubsTotal,
		                        t.ProfileImage = s.ProfileImage,
		                        t.Region = s.Region,
		                        t.AccountManager = s.AccountManager,
                                t.AccountManagerFirstName = s.AccountManagerFirstName,
                                t.AccountManagerLastName = s.AccountManagerLastName,
		                        t.ValueScoreOverallWeighted = s.ValueScoreOverallWeighted,
		                        t.GameCount = s.GameCount,
                                t.MostViewedPlatform = s.MostViewedPlatform,
                                t.PlatformPercentWatchedOther = s.PlatformPercentWatchedOther,
                                t.PlatformPercentWatchedMobile = s.PlatformPercentWatchedMobile,
                                t.PlatformPercentWatchedChromecast = s.PlatformPercentWatchedChromecast,
                                t.PlatformPercentWatchedWeb = s.PlatformPercentWatchedWeb,
                                t.PlatformPercentWatchedConsole = s.PlatformPercentWatchedConsole,
                                t.PlatformPercentWatchedDesktop = s.PlatformPercentWatchedDesktop,
                                t.IsPremium = s.IsPremium,
                                t.TimsUserActionRequired = s.TimsUserActionRequired
	                        where 
		                        t.Aggregationtype = @aggregationtype
		                        and s.AggregationType = @aggregationtype
	                        ;
    
                            insert into {Constants.DatabaseSchema}microservice_twitch_user_listing_past_{(int)aggregationType}_days 
                            (
	                            MinDate, MaxDate, AggregationType, ChannelID, Login, CCUTier, LastBroadcastDate, UserType,
	                            UserTypeChanged, MaxCCU, AverageCCU, MinutesWatchedTotal, ChannelConcurrentsMinutesWatchedTotal,
	                            MinutesBroadcastTotal, ChannelConcurrentsMinutesBroadcastTotal, UniqueDaysBroadcast, FollowerCountTotal,
	                            FollowerCountChangeTotal, ChatMessagesTotal, SubsSoldTotal, SubsSoldPrimeTotal, SubsSoldTier1Total,
	                            SubsSoldTier2Total, SubsSoldTier3Total, CreatorSubRevenueTotal, CreatorPrimeSubRevenueTotal,
	                            CreatorAdRevenueTotal, CreatorBitsRevenueTotal, CreatorFuelRevenueTotal, CreatorBountyBoardRevenueTotal,
	                            CreatorRevenueTotal, VideoPlaysTotal, PrimaryBroadcastLanguage, PrimaryBroadcastCountry, TopGameBroadcast,
	                            SubsTotal, ProfileImage, Region, AccountManager, AccountManagerFirstName, AccountManagerLastName, ValueScoreOverallWeighted, GameCount, MostViewedPlatform, 
                                PlatformPercentWatchedOther, PlatformPercentWatchedMobile, PlatformPercentWatchedChromecast, PlatformPercentWatchedWeb,
                                PlatformPercentWatchedConsole, PlatformPercentWatchedDesktop, IsPremium, TimsUserActionRequired
                            )
                            select 
                                MinDate, MaxDate, AggregationType, ChannelID, Login, CCUTier, LastBroadcastDate, UserType,
                                UserTypeChanged, MaxCCU, AverageCCU, MinutesWatchedTotal, ChannelConcurrentsMinutesWatchedTotal,
                                MinutesBroadcastTotal, ChannelConcurrentsMinutesBroadcastTotal, UniqueDaysBroadcast, FollowerCountTotal,
                                FollowerCountChangeTotal, ChatMessagesTotal, SubsSoldTotal, SubsSoldPrimeTotal, SubsSoldTier1Total,
                                SubsSoldTier2Total, SubsSoldTier3Total, CreatorSubRevenueTotal, CreatorPrimeSubRevenueTotal,
                                CreatorAdRevenueTotal, CreatorBitsRevenueTotal, CreatorFuelRevenueTotal, CreatorBountyBoardRevenueTotal,
                                CreatorRevenueTotal, VideoPlaysTotal, PrimaryBroadcastLanguage, PrimaryBroadcastCountry, TopGameBroadcast,
                                SubsTotal, ProfileImage, Region, AccountManager, AccountManagerFirstName, AccountManagerLastName, ValueScoreOverallWeighted, GameCount, MostViewedPlatform, 
                                PlatformPercentWatchedOther, PlatformPercentWatchedMobile, PlatformPercentWatchedChromecast, PlatformPercentWatchedWeb,
                                PlatformPercentWatchedConsole, PlatformPercentWatchedDesktop, IsPremium, TimsUserActionRequired
                            from {Constants.DatabaseSchema}microservice_staging_twitch_user_listing as a
                            where AggregationType = @aggregationtype
                            and not exists
                            (
	                            select 1
                                from {Constants.DatabaseSchema}microservice_twitch_user_listing_past_{(int)aggregationType}_days as b
                                where a.AggregationType = b.AggregationType
                                and a.ChannelID = b.ChannelID
                            )
                            ;
                        ";
                        command.Parameters.AddWithValue("@aggregationtype", (int)aggregationType);
                        command.ExecuteNonQueryWithMeasurements("user_listing_aggregation_update");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserListingJob: {ex}");
            }
        }

        private Tuple<string, string> GetExecuteSql(AggregationType aggregationType, DateTime maxdate)
        {
            DateTime mindate = maxdate.AddDays(-(int)aggregationType);

            string aggregateSnippet = string.Empty;
            switch (aggregationType)
            {
                case AggregationType.Week:
                {
                    mindate = maxdate.AddDays(-7);
                    aggregateSnippet = SnippetHelper.GetQuery7Snippet();
                    break;
                }
                case AggregationType.ThirtyDay:
                {
                    mindate = maxdate.AddDays(-30);
                    aggregateSnippet = SnippetHelper.GetQuery30Snippet();
                    break;
                }
                case AggregationType.SixtyDay:
                {
                    mindate = maxdate.AddDays(-60);
                    aggregateSnippet = SnippetHelper.GetQuery60Snippet();
                    break;
                }
                case AggregationType.NintyDay:
                {
                    mindate = maxdate.AddDays(-90);
                    aggregateSnippet = SnippetHelper.GetQuery90Snippet();
                    break;
                }
                case AggregationType.BankerYear:
                {
                    mindate = maxdate.AddDays(-360);
                    aggregateSnippet = SnippetHelper.GetQuery360Snippet();
                    break;
                }
                default:
                {
                    throw new ArgumentOutOfRangeException("aggregationType");
                }
            }
            Log.Info($"UserListingJob: Aggregate Snippet: {aggregateSnippet}");

            var prefixSql = 
            $@"
                drop table if exists population;
                create temp table population (channel_id bigint, maxdate date, login varchar(64));
                insert into population (channel_id, maxdate, login)
                select a.channel_id, a.max_date, b.login
                from
                (
                    select 
                        cast(channel_id as bigint) as channel_id, 
                        max(max_date) as max_date
                    from cubes.amp_application_population as a
                    group by cast(channel_id as bigint)
                ) as a
                left join dbsnapshots.users as b
                    on a.channel_id = b.id;

                drop table if exists total_minutes_watched;
                create temp table total_minutes_watched(channel_id bigint not null, minutes_watched float);
                insert into total_minutes_watched select hours_watched_daily_by_device.channel_id,sum(hours_watched_daily_by_device.mw)  from cubes.hours_watched_daily_by_device inner join population on population.channel_id = hours_watched_daily_by_device.channel_id where day >= cast('@mindatestring' as date) group by hours_watched_daily_by_device.channel_id;


                drop table if exists total_platform_minutes_watched;
                create temp table total_platform_minutes_watched( minutes_watched float,channel_id bigint not null, platform varchar(32));
                insert into total_platform_minutes_watched select sum(hours_watched_daily_by_device.mw), hours_watched_daily_by_device.channel_id, case platform
                when 'other' then 'other'
                when 'ios' then 'mobile'
                when 'android' then 'mobile'
                when 'mobile_web' then 'mobile'
                when 'xbox360' then 'console'
                when 'xboxone' then 'console'
                when 'playstation' then 'console'
                when 'web' then 'web'
                when 'desktop' then 'desktop'
                when 'chromecast' then 'chromecast'
                end from cubes.hours_watched_daily_by_device inner join population on population.channel_id = hours_watched_daily_by_device.channel_id inner join total_minutes_watched on total_minutes_watched.channel_id = population.channel_id where day >= cast('@mindatestring' as date) group by 2,3;

                drop table if exists platform_minutes_watched;
                create temp table platform_minutes_watched( percent_minutes_watched float,channel_id bigint not null, platform varchar(32));
                insert into platform_minutes_watched select (total_platform_minutes_watched.minutes_watched / total_minutes_watched.minutes_watched) * 100.0, total_platform_minutes_watched.channel_id, platform from total_platform_minutes_watched inner join total_minutes_watched on total_minutes_watched.channel_id = total_platform_minutes_watched.channel_id;         
                drop table if exists most_watched_platform;
                create temp table most_watched_platform(channel_id bigint not null, platform varchar(32));
                insert into most_watched_platform select channel_id, platform from platform_minutes_watched where percent_minutes_watched != 0 and percent_minutes_watched = (select max(pmw2.percent_minutes_watched) from platform_minutes_watched pmw2 where pmw2.channel_id = platform_minutes_watched.channel_id);

                drop table if exists platform_watched_percents;
                create temp table platform_watched_percents(channel_id bigint not null, other float, console float, chromecast float, desktop float, web float, mobile float);
                insert into platform_watched_percents select channel_id,(select sum(mwp2.percent_minutes_watched) from platform_minutes_watched mwp2 where mwp2.channel_id = most_watched_platform.channel_id and mwp2.platform = 'other'),(select sum(mwp2.percent_minutes_watched) from platform_minutes_watched mwp2 where mwp2.channel_id = most_watched_platform.channel_id and mwp2.platform in ('console')),(select sum(mwp2.percent_minutes_watched) from platform_minutes_watched mwp2 where mwp2.channel_id = most_watched_platform.channel_id and mwp2.platform = 'chromecast'),(select sum(mwp2.percent_minutes_watched) from platform_minutes_watched mwp2 where mwp2.channel_id = most_watched_platform.channel_id and mwp2.platform = 'desktop'),(select sum(mwp2.percent_minutes_watched) from platform_minutes_watched mwp2 where mwp2.channel_id = most_watched_platform.channel_id and mwp2.platform = 'web'),(select sum(mwp2.percent_minutes_watched) from platform_minutes_watched mwp2 where mwp2.channel_id = most_watched_platform.channel_id and mwp2.platform in ('mobile')) from most_watched_platform group by channel_id;

                drop table if exists top_games_broadcast;
                create temp table top_games_broadcast (channel_id bigint not null, top_game_broadcast varchar(1024) null);
                insert into top_games_broadcast (channel_id, top_game_broadcast)
                select
                    channel_id, top_game_broadcast
                from
                (
                    select 
                        channel_id, 
                        coalesce(primary_broadcast_game, top_live_game_watched) as top_game_broadcast, 
                        sum(minutes_broadcast_total) as time_broadcast,
                        count(*) as count_times_cast,
                        sum(coalesce(unique_follows, 0) + coalesce(chat_messages_sent, 0) + coalesce(unique_chatters, 0) + coalesce(ccu_minutes_broadcast_total, 0) + coalesce(avg_ccu, 0)) as tie_breaker,
                        row_number() over(partition by channel_id order by count(*) desc, sum(minutes_broadcast_total) desc, sum(coalesce(unique_follows, 0) + coalesce(chat_messages_sent, 0) + coalesce(unique_chatters, 0) + coalesce(ccu_minutes_broadcast_total, 0) + coalesce(avg_ccu, 0)) desc) as sort
                    from cubes.affiliates_partners_daily_channel_summary as a
                    where
                        day >= cast('@mindatestring' as date)
                        and day < cast('@maxdatestring' as date)
                        and coalesce(channel_id, 0) > 0
                        and coalesce(coalesce(primary_broadcast_game, top_live_game_watched), '') <> ''
                    and exists
                    (
                        select 1
                        from population as b
                        where a.channel_id = b.channel_id 
                    )
                    group by channel_id, coalesce(primary_broadcast_game, top_live_game_watched)  
                ) as a
                where sort = 1;
            ";

            var sql =
            $@"
                /* MAIN SQL */
                with selection as
                (
                    select
                    channel_id as ChannelID,
                    max(day) as maxdate,
                    sum(case when is_affiliate = true then 1 else 0 end)  as DaysAffiliateCount,
                    sum(case when is_partner = true then 1 else 0 end) as DaysPartneredCount,
                    sum(case when coalesce(minutes_broadcast_total, 0) > 0 then 1 else 0 end) as UniqueDaysBroadcastTotal,
                    sum(coalesce(minutes_watched_total, 0)) as MinutesWatchedTotal,
                    sum(coalesce(ccu_minutes_watched_total, 0)) as ChannelConcurrentMinutesWatchedTotal,
                    sum(coalesce(minutes_broadcast_total, 0)) as MinutesBroadcastTotal,
                    sum(coalesce(ccu_minutes_broadcast_total, 0)) as ChannelConcurrentsMinutesBroadcastTotal,
                    sum(coalesce(change_in_total_follower_count, 0)) as FollowerCountChangeTotal,
                    sum(coalesce(subs_sold_total, 0)) as SubsSoldTotal,
                    sum(coalesce(subs_sold_prime, 0)) as PrimeSubsSoldTotal,
                    sum(coalesce(subs_sold_499_total, 0)) as SubsSoldTier1Total,
                    sum(coalesce(subs_sold_999_total, 0)) as SubsSoldTier2Total,
                    sum(coalesce(subs_sold_2499_total, 0)) as SubsSoldTier3Total,
                    sum(coalesce(creator_ad_revenue, 0.0)) as CreatorAdRevenueTotal,
                    sum(coalesce(creator_sub_revenue_total, 0.0)) as CreatorSubRevenueTotal,
                    sum(coalesce(creator_sub_revenue_prime, 0.0)) as CreatorPrimeSubRevenueTotal,
                    sum(coalesce(creator_bits_revenue_total, 0.0)) as CreatorBitsRevenueTotal,
                    sum(coalesce(creator_fuel_revenue, 0.0)) as CreatorFuelRevenueTotal,
                    sum(coalesce(creator_bounty_board_revenue, 0.0)) as CreatorBountyBoardRevenueTotal,
                    sum(coalesce(video_plays_total, 0)) as VideoPlaysTotal,
                    sum(coalesce(chat_messages_sent, 0)) as ChatMessagesTotal,
                    sum(
                        coalesce(creator_ad_revenue, 0.0)
                        + coalesce(creator_sub_revenue_total, 0.0)
                        + coalesce(creator_bits_revenue_total, 0.0)
                        + coalesce(creator_fuel_revenue, 0.0)
                        + coalesce(creator_bounty_board_revenue, 0.0)
                    ) as CreatorRevenueTotal
                    from cubes.affiliates_partners_daily_channel_summary as a
                    where 
                    day >= cast('@mindatestring' as date)
                    and day < cast('@maxdatestring' as date)
                    and coalesce(channel_id, 0) > 0
                    and exists
                    (
                        select 1
                        from population as b
                        where a.channel_id = b.channel_id 
                    )
                    group by channel_id
                ), metadata as
                (
                    select 
                        a.channel_id as ChannelID, 
                        coalesce(a.channel, z.login) as Login, 
                        case when a.primary_broadcast_language{aggregateSnippet} = '' then null else a.primary_broadcast_language{aggregateSnippet} end as PrimaryBroadcastLanguage, 
                        case when a.primary_broadcast_country{aggregateSnippet} = '' then null else a.primary_broadcast_country{aggregateSnippet} end as PrimaryBroadcastCountry,
                        coalesce(a.total_follower_count, 0) as FollowerCountTotal,
                        coalesce(a.ccu_tier{aggregateSnippet}, '') as CCUTier,
                        coalesce(a.max_ccu{aggregateSnippet}, 0) as MaxCCUTotal,
                        coalesce(a.avg_ccu{aggregateSnippet}, 0.0) as AverageCCUTotal,
                        coalesce(a.primary_broadcast_game, '') as TopGameBroadcast,
                        coalesce(a.value_score_rank_overall_weighted{aggregateSnippet}, 0) as ValueScoreOverallWeighted,
                        coalesce(a.marked_as_premium, false) as IsPremium
                    from cubes.affiliates_partners_daily_channel_summary as a
                    left join dbsnapshots.users as z
                        on a.channel_id = z.id
                    where 
                    day >= cast('@mindatestring' as date)
                    and day < cast('@maxdatestring' as date)
                    and exists
                    (
                        select 1
                        from population as b
                        where a.channel_id = b.channel_id
                        and a.day = b.maxdate
                    )
                ), broadcast_language as
                (
                    select channel_id, PrimaryBroadcastLanguage
                    from
                    (
                        select channel_id, coalesce(primary_broadcast_language_past_360_days, '') as PrimaryBroadcastLanguage, row_number() over(partition by channel_id order by day desc) as sort
                        from cubes.affiliates_partners_daily_channel_summary as a
                        where 
                        day = (select max(day) from cubes.affiliates_partners_daily_channel_summary as c where a.channel_id = c.channel_id and coalesce(primary_broadcast_language_past_360_days, '') <> '')
                        and exists
                        (
                            select 1
                            from population as b
                            where a.channel_id = b.channel_id
                        )
                        and coalesce(primary_broadcast_language_past_360_days, '') <> ''
                    ) as a
                    where sort = 1
                ), broadcast_country as
                (
                    select channel_id, PrimaryBroadcastCountry
                    from
                    (
                        select channel_id, coalesce(primary_broadcast_country_past_360_days, '') as PrimaryBroadcastCountry, row_number() over(partition by channel_id order by day desc) as sort
                        from cubes.affiliates_partners_daily_channel_summary as a
                        where 
                        day = (select max(day) from cubes.affiliates_partners_daily_channel_summary as c where a.channel_id = c.channel_id and coalesce(primary_broadcast_country_past_360_days, '') <> '')
                        and exists
                        (
                            select 1
                            from population as b
                            where a.channel_id = b.channel_id
                        )
                        and coalesce(primary_broadcast_country_past_360_days, '') <> ''
                    ) as a
                    where sort = 1
                ), total_subs as (
                    SELECT count(*) as TotalSubs, ticket_product_owner_id as ChannelID
		                    FROM dbsnapshots.tickets inner join dbsnapshots.ticket_products on tickets.ticket_product_id = dbsnapshots.ticket_products.id
		                    AND (access_start is NULL or access_start < CURRENT_DATE)
		                    AND (access_end is NULL or access_end > CURRENT_DATE) AND created_by != 'admin' 
                    and exists
                    (
                        select 1
                        from population as b
                        where ticket_product_owner_id = b.channel_id 
                    )
                    group by ticket_product_owner_id
                ), account_owner as
                (
                    select
                        channel_id,
                        account_owner_ldap_username as account_owner,
                        account_owner_first_name,
                        account_owner_last_name
                    from salesforce_partnerships.partner_owners
                ), last_broadcast as
                (
                    select
                    a.channel_id as ChannelID,
                    max(date) as LastBroadcastDate
                    from tahoe.minute_broadcast as a
                    where date >= cast('@lookback' as date)
                    and date < cast('@maxdatestring' as date)
                    and exists
                    (
                        select 1
                        from population as b
                        where a.channel_id = b.channel_id
                    )
                    group by 
                    a.channel_id
                ), count_games as
                (
                    select channel_id, count(*) as game_count
                    from
                    (
                        select channel_id, channel_category_id 
                        from tahoe.liveline_channel as a 
                        where date >= cast('@mindatestring' as date)
                        and date < cast('@maxdatestring' as date)
                        and exists
                        (
                            select 1
                            from population as b
                            where a.channel_id = b.channel_id
                        )
                    group by channel_id, channel_category_id
                    ) as a
                    group by channel_id
                ), royalty as
                (
	                select channel_id, effectivefrom, isuseractionrequired
	                from
	                (
		                select
			                c.id as channel_id,
			                a.effectivefrom,
			                a.isuseractionrequired,
			                row_number() over(partition by a.accountid order by a.effectivefrom desc) as sort
		                from paymentsdata.tax_status_updates_v1 as a
		                left join paymentsdata.payout_entities_v0 as b
			                on a.accountid = concat(cast(b.payoutentityid as varchar), ':ROY')
		                left join dbsnapshots.users as c
			                on b.ownerchannelid = c.id
	                ) as a
	                where sort = 1
                ), service as
                (
	                select channel_id, effectivefrom, isuseractionrequired
	                from
	                (
		                select
			                c.id as channel_id,
			                a.effectivefrom,
			                a.isuseractionrequired,
			                row_number() over(partition by a.accountid order by a.effectivefrom desc) as sort
		                from paymentsdata.tax_status_updates_v1 as a
		                left join paymentsdata.payout_entities_v0 as b
			                on a.accountid = concat(cast(b.payoutentityid as varchar), ':SER')
		                left join dbsnapshots.users as c
			                on b.ownerchannelid = c.id
	                ) as a
	                where sort = 1
                )
                select *
                from
                (
                    select
                        cast('@mindatestring' as datetime) as MinDate,
                        cast('@maxdatestring' as datetime) as MaxDate,
                        cast(@aggregationtype as int) as AggregationType,
                        cast(a.ChannelID as bigint) as TwitchUserID,
                        coalesce(population.login, coalesce(b.Login, '')) as Login,
                        coalesce(b.CCUTier, '') as CcuTier,
                        cast(coalesce(c.LastBroadcastDate, '1900-01-01') as datetime) as LastBroadcastDate,
                        case
                            when a.DaysPartneredCount > 0 then 'partner'
                            when a.DaysAffiliateCount > 0 then 'affiliate'
                            else ''
                        end as UserType,
                        case when cast(case
                            when a.DaysPartneredCount <> a.DaysAffiliateCount and a.DaysAffiliateCount > 0 and a.DaysPartneredCount > 0 then true
                            else false
                        end as boolean) = true then 1 else 0 end as UserTypeChanged,
                        cast(coalesce(b.MaxCCUTotal, 0) as int) as MaxCcu,
                        cast(coalesce(b.AverageCCUTotal, 0.0) as float) as AverageCcu,
                        cast(coalesce(a.MinutesWatchedTotal, 0) as bigint) as MinutesWatchedTotal,
                        cast(coalesce(a.ChannelConcurrentMinutesWatchedTotal, 0) as bigint) as ChannelConcurrentsMinutesWatchedTotal,
                        cast(coalesce(a.MinutesBroadcastTotal, 0) as bigint) as MinutesBroadcastTotal,
                        cast(coalesce(a.ChannelConcurrentsMinutesBroadcastTotal, 0) as bigint) as ChannelConcurrentsMinutesBroadcastTotal,
                        cast(coalesce(a.UniqueDaysBroadcastTotal, 0) as int) as UniqueDaysBroadcast,
                        cast(coalesce(b.FollowerCountTotal, 0) as int) as FollowerCountTotal,
                        cast(coalesce(a.FollowerCountChangeTotal, 0) as int) as FollowerCountChangeTotal,
                        cast(coalesce(a.ChatMessagesTotal, 0) as int) as ChatMessagesTotal,
                        cast(coalesce(a.SubsSoldTotal, 0) as int) as SubsSoldTotal,
                        cast(coalesce(a.PrimeSubsSoldTotal, 0) as int) as SubsSoldPrimeTotal,
                        cast(coalesce(a.SubsSoldTier1Total, 0) as int) as SubsSoldTier1Total,
                        cast(coalesce(a.SubsSoldTier2Total, 0) as int) as SubsSoldTier2Total,
                        cast(coalesce(a.SubsSoldTier3Total, 0) as int) as SubsSoldTier3Total,         
                        cast(coalesce(a.CreatorSubRevenueTotal, 0) as float) as CreatorSubRevenueTotal,
                        cast(coalesce(a.CreatorPrimeSubRevenueTotal, 0) as float) as CreatorPrimeSubRevenueTotal,
                        cast(coalesce(a.CreatorAdRevenueTotal, 0) as float) as CreatorAdRevenueTotal,
                        cast(coalesce(a.CreatorBitsRevenueTotal, 0) as float) as CreatorBitsRevenueTotal,
                        cast(coalesce(a.CreatorFuelRevenueTotal, 0) as float) as CreatorFuelRevenueTotal,
                        cast(coalesce(a.CreatorBountyBoardRevenueTotal, 0) as float) as CreatorBountyBoardRevenueTotal,
                        cast(coalesce(a.CreatorRevenueTotal, 0) as float) as CreatorRevenueTotal,
                        cast(coalesce(a.VideoPlaysTotal, 0) as bigint) as VideoPlaysTotal,
                        case 
                            when coalesce(trim(b.PrimaryBroadcastLanguage), '') <> '' then b.PrimaryBroadcastLanguage
                            when coalesce(trim(i.PrimaryBroadcastLanguage), '') <> '' then i.PrimaryBroadcastLanguage
                            else ''
                        end as PrimaryBroadcastLanguage,
                        case
                            when coalesce(trim(b.PrimaryBroadcastCountry), '') <> '' then b.PrimaryBroadcastCountry
                            when coalesce(trim(j.PrimaryBroadcastCountry), '') <> '' then j.PrimaryBroadcastCountry
                            else ''
                        end as PrimaryBroadcastCountry,
                        coalesce(d.top_game_broadcast, '') as TopGameBroadcast,
                        coalesce(g.TotalSubs, 0) as SubsTotal,
                        case 
			                when coalesce(profile_image, 'empty_profile_image') = 'empty_profile_image' then 'empty_profile_image'
			                when regexp_instr(profile_image,'^(profile_image_[\\d]+)$', 1) then profile_image
			                when 
				                regexp_instr(profile_image,'(:new_format: true)', 1) then 
				                coalesce(replace(regexp_substr(profile_image, ':uid: ([a-z0-9\-]+)', 1), ':uid: ', ''), '') 
				                || '-profile_image-70x70.' 
				                || coalesce(replace(regexp_substr(profile_image, ':format: ([a-z0-9\-]+)', 1), ':format: ', ''), '')
			                when regexp_instr(profile_image,'(:sizes:)', 1) then
				                coalesce(b.Login, '') 
				                || '-profile_image-' 
				                || coalesce(replace(regexp_substr(profile_image, ':uid: ([a-z0-9\-]+)', 1), ':uid: ', ''), '')  
				                || '-70x70.' 
				                || coalesce(replace(regexp_substr(profile_image, ':format: ([a-z0-9\-]+)', 1), ':format: ', ''), '')
			                else 'empty_profile_image'
		                end as ProfileImage,
                        coalesce(f.metaregion,'') as Region,
                        coalesce(h.account_owner, '') as AccountManager,
                        coalesce(h.account_owner_first_name, '') as AccountManagerFirstName,
                        coalesce(h.account_owner_last_name, '') as AccountManagerLastName,
                        cast(coalesce(b.ValueScoreOverallWeighted, 0) as float) as ValueScoreOverallWeighted,
                        coalesce(k.game_count, 0) as GameCount,
                        coalesce(mwp.platform,'') as MostViewedPlatform,
                        coalesce(pwp.other,'') as PlatformPercentWatchedOther,
                        coalesce(pwp.mobile,'') as PlatformPercentWatchedMobile,
                        coalesce(pwp.chromecast,'') as PlatformPercentWatchedChromecast,
                        coalesce(pwp.web,'') as PlatformPercentWatchedWeb,
                        coalesce(pwp.console,'') as PlatformPercentWatchedConsole,
                        coalesce(pwp.desktop,'') as PlatformPercentWatchedDesktop,
                        case when cast(coalesce(b.IsPremium, false) as boolean) = true then 1 else 0 end as IsPremium,
                        case
                            when service.isuseractionrequired = 'true' or royalty.isuseractionrequired = 'true'
                            then 1 else 0 
                        end as TimsUserActionRequired,
                        row_number() over (partition by a.ChannelID, cast(@aggregationtype as int) order by c.LastBroadcastDate desc, a.CreatorRevenueTotal desc, case when a.DaysPartneredCount > 0 then 'partner' when a.DaysAffiliateCount > 0 then 'affiliate' else '' end desc) as sort
                    from selection as a
                    left join metadata as b
                        on a.ChannelID = b.ChannelID
                    left join last_broadcast as c
                        on a.ChannelID = c.ChannelID
                    left join top_games_broadcast as d
                        on a.ChannelID = d.channel_id
                    left join dbsnapshots.users as e
                        on a.ChannelID = e.id
                    left join total_subs as g on g.ChannelID = a.ChannelID
                    left join account_owner as h on a.ChannelID = h.channel_id
                    left join broadcast_language as i on a.ChannelID = i.channel_id
                    left join broadcast_country as j on a.ChannelID = j.channel_id
                    left join metadata.country_to_region as f on f.country = case
                            when coalesce(trim(b.PrimaryBroadcastCountry), '') <> '' then b.PrimaryBroadcastCountry
                            when coalesce(trim(j.PrimaryBroadcastCountry), '') <> '' then j.PrimaryBroadcastCountry
                            else ''
                        end
                    left join count_games as k on a.ChannelID = k.channel_id
                    left join most_watched_platform as mwp on mwp.channel_id = a.ChannelID
                    left join platform_watched_percents as pwp on pwp.channel_id = a.ChannelID
                    left join service on a.ChannelID = service.channel_id
					left join royalty on a.ChannelID = royalty.channel_id
                    left join population on a.ChannelID = population.channel_id
                ) as a
                where sort = 1;
                /* END TWITCH USER LISTING QUERY */
            ";
            return new Tuple<string,string>(prefixSql, sql);
        }
    }
}
