﻿using Resonance.Core;
using Resonance.Core.Extensions;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Resonance.Jobs.Amp.TwitchUser
{
    public class UserScorecardJob : JobBase, IJob<long>
    {
        private static AmpUserScorecardJobConfiguration jobConfig { get; set; }

        public UserScorecardJob(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                jobConfig = new AmpUserScorecardJobConfiguration()
                {

                };
                Log.Info($"UserScorecardJob: configured. IsActive: {Config.IsActive}");
            }
            catch(Exception ex)
            {
                Log.Error($@"UserScorecardJob: {ex}");
            }
        }

        public override void Run()
        {
            try
            {
                Log.Info($"UserScorecardJob: Running");
                this.Config.IsRunning = true;
                if (jobConfig != null)
                {
                    long resultCount = 0;
                    var rundate = DateTime.UtcNow.Date;
                    WipeOldData();
                    var param = new Dictionary<string, dynamic>();
                    for (var channelDigit = 0; channelDigit <= 9; channelDigit++)
                    {
                        param.Clear();
                        param.Add("channelDigit", channelDigit);
                        var bucket = "crs-data-export";
                        var keypath = $"{Constants.AppConfig.Application.Environment}/twitch-user-scorecard-data/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/ends_with_{channelDigit}/";
                        DBManagerRedshift.UnloadToS3
                        (
                            bucket: bucket, 
                            keyfolderpath: keypath, 
                            wrappedsql: GetScorecardSql(channelDigit), 
                            kmsarn: AppConfig.Data.Application.KmsArn,
                            timeout: 86400,
                            alternateConnection: DBManagerRedshift.GetTahoeConnectionString(),
                            options: new Core.Models.DatabaseModels.RedshiftModels.UnloadOptionsModel()
                            {
                                Header = "header",
                                MaxFileSizeInMB = 100,
                                FileNamePart = "data_",
                                Delimiter = ",",
                                AddQuotes = "addquotes",
                                AllowOverwrite = "allowoverwrite",
                                Gzip = "",
                                Parallel = "on",
                                Escape = "escape",
                                Manifest = "manifest",
                                AutomaticallyReplaceQuotes = true
                            }
                        );
                        LoadS3FileToMysqlStaging(bucket, keypath);
                        UpdateScorecardFromStaging(channelDigit);
                    }

                    try
                    {
                        using (var conn = DBManagerMysql.GetConnection(true))
                        {
                            using
                            (
                                var command =
                                    AmpQuerySql.InsertUpdateEtlTracking
                                    (
                                        source: "Tahoe_Scorecard_Query",
                                        target: $"Microservice_Twitch_User_Scorecard",
                                        method: "Jobs.UserScorecardJob",
                                        timestamp: DateTime.UtcNow,
                                        rowcount: resultCount
                                    )
                            )
                            {
                                command.Connection = conn;
                                command.CommandTimeout = 0;
                                command.ExecuteNonQueryWithMeasurements("update_scorecard_etl");
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        Log.Error($@"UserScorecardJob: {ex}");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserScorecardJob: {ex}");
            }
            finally
            {
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromDays(1));
            }

            Log.Info($@"UserScorecardJob Complete. Next Run Time: {this.Config.NextRunTime.Value.ToString("yyyy-MM-dd HH:mm:ss")}");
        }

        private void WipeOldData()
        {
            Log.Info($"UserScorecardJob: Wiping old user scorecard data from staging table");
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"truncate table {Constants.DatabaseSchema}microservice_staging_twitch_user_auto_scorecard;";
                    command.ExecuteNonQueryWithMeasurements("user_scorecard_staging_cleanup");
                }
            }
            Log.Info($"UserScorecardJob: Completed wiping old user scorecard data from staging table");
        }

        private string GetScorecardSql(int channelDigit)
        {
            string sql =
            $@"
                /* BEGIN USER SCORECARD QUERY */
                with apdcs as
                (
                    select
                        day,
                        channel_id,
                        channel
                    from cubes.affiliates_partners_daily_channel_summary
                    where day = (select max(day) from cubes.affiliates_partners_daily_channel_summary)
                    and right(channel_id, 1) = {channelDigit}
                ),
                users as
                (
                    select
                        id as user_id,
                        case when coalesce(profile_image, 'empty_profile_image') <> 'empty_profile_image'
                            then True
                            else False
                        end as HasProfileImage
                    from dbsnapshots.users as a
                    where exists
                    (
                        select 1
                        from apdcs as b
                        where a.id = b.channel_id
                    )
                    and right(id, 1) = {channelDigit}
                ),
                events as
                (
                    select
                        channel_id,
                        case when count(*) > 0 then True else False end as HasUsedEvents
                    from tahoe.oracle_event_server as a
                    where action = 'create'
                    and exists
                    (
                        select 1
                        from apdcs as b
                        where a.channel_id = b.channel_id
                    )
                    and right(channel_id, 1) = {channelDigit}
                    group by channel_id
                ),
                panels as
                (
                    select
                        user_id,
                        case when count(*) >= 3 then True else False end as MeetsOrExceedsChannelPanelCount
                    from dbsnapshots.panels as a
                    where exists
                    (
                        select 1
                        from apdcs as b
                        where a.user_id = b.channel_id
                    )
                    and right(user_id, 1) = {channelDigit}
                    group by user_id
                ),
                chat_rules as
                (
                    select 
                        user_id,
                        case when count(*) > 0
                            then True
                            else False
                        end as HasUsedRules
                    from tahoe.chat_channel_moderation as a
                    where action = 'chat_rules'
                    and exists
                    (
                        select 1
                        from apdcs as b
                        where a.user_id = b.channel_id
                    )
                    and right(user_id, 1) = {channelDigit}
                    group by user_id
                ),
                collections as
                (
                    select
                        channel_id,
                        case when count(*) > 0 then True else False end as MeetsOrExceedsCollectionsCount
                    from tahoe.collection_create as a
                    where exists
                    (
                        select 1
                        from apdcs as b
                        where a.channel_id = b.channel_id
                    )
                    and right(channel_id, 1) = {channelDigit}
                    group by channel_id
                ),
                automod as
                (
                    select
                        channel_id,
                        AutomodEnabled
                    from
                    (
                         select
                            channel_id,
                            am_active as AutomodEnabled,
                            row_number() over(partition by channel_id order by am_active desc) as sort
                        from cubes.channel_automod_settings as a
                        where exists
                        (
                            select 1
                            from apdcs as b
                            where a.channel_id = b.channel_id
                        )
                        and right(channel_id, 1) = {channelDigit}
                        and date = (select max(date) from cubes.channel_automod_settings)
                    ) as a
                    where sort = 1
                ),
                vods as
                (
                    select
                        owner_id,
                        case when sum(case when broadcast_type = 'archive' then 1 else 0 end) > 0 then True else False end as MeetsOrExceedsVodCount,
                        case when sum(case when broadcast_type = 'highlight' then 1 else 0 end) > 0 then True else False end as MeetsOrExceedsHighlightCount,
                        case when sum(case when broadcast_type = 'upload' then 1 else 0 end) > 0 then True else False end as MeetsOrExceedsUploadCount
                    from dbsnapshots.vods as a
                    where exists
                    (
                        select 1
                        from apdcs as b
                        where a.owner_id = b.channel_id
                    )
                    and right(owner_id, 1) = {channelDigit}
                    group by owner_id
                ),
                emotes as
                (
                    select
                        ticket_product_owner_id,
                        case when count(*) > 0 then True else False end as MeetsOrExceedsEmoteCount
                    from cubes.emoticons_enriched as a
                    where exists
                    (
                        select 1
                        from apdcs as b
                        where a.ticket_product_owner_id = b.channel_id
                    )
                    and ticket_product_availability_state = 'active'
                    and right(ticket_product_owner_id, 1) = {channelDigit}
                    group by ticket_product_owner_id
                ),
                cheermotes as
                (
                    select
                        channel_id,
                        case when count(*) > 0 then True else False end as MeetsOrExceedsCheermoteCount
                    from tahoe.bits_partner_cheermote_detail as a
                    where exists
                    (
                        select 1
                        from apdcs as b
                        where a.channel_id = b.channel_id
                    )
                    and right(channel_id, 1) = {channelDigit}
                    group by channel_id
                ),
                clips as
                (
                    select
                        channel_id,
                        case when count(*) > 0 then True else False end as MeetsOrExceedsClipCount
                    from tahoe.create_clip as a
                    where exists
                    (
                        select 1
                        from apdcs as b
                        where a.channel_id = b.channel_id
                    )
                    and right(channel_id, 1) = {channelDigit}
                    group by channel_id
                ),
                tenure_badges as
                (
                    select
                        channel_id,
                        case when count(*) > 0 then True else False end as MeetsOrExceedsTenureBadgeCount
                    from tahoe.tenure_badges as a
                    where exists
                    (
                        select 1
                        from apdcs as b
                        where a.channel_id = b.channel_id
                    )
                    and right(channel_id, 1) = {channelDigit}
                    group by channel_id
                ),
                extension_install as
                (
                    select
                        channel_id,
                        case when count(*) >= 2 then True else False end as MeetsOrExceedsExtensionInstallCount
                    from tahoe.extension_install as a
                    where exists
                    (
                        select 1
                        from apdcs as b
                        where a.channel_id = b.channel_id
                    )
                    and right(channel_id, 1) = {channelDigit}
                    group by channel_id
                ),
                moderators as
                (
                    select
                        channel_id,
                        case when count(*) > 0 then True else False end as MeetsOrExceedsModeratorCount
                    from dbsnapshots.moderators as a
                    where exists
                    (
                        select 1
                        from apdcs as b
                        where a.channel_id = b.channel_id
                    )
                    and right(channel_id, 1) = {channelDigit}
                    group by channel_id
                ),
                tags as
                (
                    select
                        reverse(substring(reverse(content_id), 0, charindex('_', reverse(content_id)))) as channel_id,
                        case when count(*) > 0 then True else False end as MeetsOrExceedsTagCount
                    from dbsnapshots.graffiti_content_tags
                    where 
                        content_id like 'live_channel%'
                    group by
                        reverse(substring(reverse(content_id), 0, charindex('_', reverse(content_id))))
                )
                select
                    apdcs.channel_id as TwitchUserID,
                    apdcs.channel as Login,
                    coalesce(apdcs.day, cast('1900-01-01' as datetime)) as DataSourced,
                    case 
                        when coalesce(users.HasProfileImage, False) = True
                        then 1
                        else 0
                    end as HasProfileImage,
                    case 
                        when coalesce(events.HasUsedEvents, False)  = True
                        then 1
                        else 0
                    end as HasUsedEvents,
                    case 
                        when coalesce(panels.MeetsOrExceedsChannelPanelCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsChannelPanelCount,
                    case
                        when coalesce(chat_rules.HasUsedRules, False) = True
                        then 1
                        else 0
                    end as HasUsedRules,
                    case
                        when coalesce(collections.MeetsOrExceedsCollectionsCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsCollectionsCount,
                    case
                        when coalesce(automod.AutomodEnabled, False) = True
                        then 1
                        else 0
                    end as AutomodEnabled,
                    case
                        when coalesce(vods.MeetsOrExceedsVodCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsVodCount,
                    case
                        when coalesce(vods.MeetsOrExceedsHighlightCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsHighlightCount,
                    case
                        when coalesce(vods.MeetsOrExceedsUploadCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsUploadCount,
                    case
                        when coalesce(emotes.MeetsOrExceedsEmoteCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsEmoteCount,
                    case
                        when coalesce(cheermotes.MeetsOrExceedsCheermoteCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsCheermoteCount,
                    case
                        when coalesce(clips.MeetsOrExceedsClipCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsClipCount,
                    case
                        when coalesce(tenure_badges.MeetsOrExceedsTenureBadgeCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsTenureBadgeCount,
                    case
                        when coalesce(extension_install.MeetsOrExceedsExtensionInstallCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsExtensionInstallCount,
                    case
                        when coalesce(moderators.MeetsOrExceedsModeratorCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsModeratorCount,
                    case
                        when coalesce(tags.MeetsOrExceedsTagCount, False) = True
                        then 1
                        else 0
                    end as MeetsOrExceedsTagCount
                from apdcs
                left join users
                    on apdcs.channel_id = users.user_id
                left join events
                    on apdcs.channel_id = events.channel_id
                left join panels
                    on apdcs.channel_id = panels.user_id
                left join chat_rules
                    on apdcs.channel_id = chat_rules.user_id
                left join collections
                    on apdcs.channel_id = collections.channel_id
                left join automod
                    on apdcs.channel_id = automod.channel_id
                left join vods
                    on apdcs.channel_id = vods.owner_id
                left join emotes
                    on apdcs.channel_id = emotes.ticket_product_owner_id
                left join cheermotes
                    on apdcs.channel_id = cheermotes.channel_id
                left join clips
                    on apdcs.channel_id = clips.channel_id
                left join tenure_badges
                    on apdcs.channel_id = tenure_badges.channel_id
                left join extension_install
                    on apdcs.channel_id = extension_install.channel_id
                left join moderators
                    on apdcs.channel_id = moderators.channel_id
                left join tags
                    on apdcs.channel_id = tags.channel_id
                ; 
                /* END USER SCORECARD QUERY */
            ";
            return sql;
        }

        private void LoadS3FileToMysqlStaging(string bucket, string keypath)
        {
            try
            {
                string fields =
                @"(
                    TwitchUserID, Login, DataSourced, HasProfileImage, HasUsedEvents,
                    MeetsOrExceedsChannelPanelCount, HasUsedRules, MeetsOrExceedsCollectionsCount,
                    AutomodEnabled, MeetsOrExceedsVodCount, MeetsOrExceedsHighlightCount,
                    MeetsOrExceedsUploadCount, MeetsOrExceedsEmoteCount,
                    MeetsOrExceedsCheermoteCount, MeetsOrExceedsClipCount, MeetsOrExceedsTenureBadgeCount,
                    MeetsOrExceedsExtensionInstallCount, MeetsOrExceedsModeratorCount, MeetsOrExceedsTagCount
                )";
                if (S3Helper.ExistsInS3(bucket, $"{keypath}data_manifest") == true)
                {
                    using (var conn = DBManagerMysql.GetConnection(true))
                    {
                        using (var command = AmpQuerySql.LoadFileFromS3
                            (bucket,
                            $"{keypath}data_manifest",
                            $"{Constants.DatabaseSchema}microservice_staging_twitch_user_auto_scorecard",
                            ignore: "ignore 1 lines",
                            fields: fields)
                        )
                        {
                            command.CommandTimeout = 86400;
                            command.Connection = conn;
                            command.ExecuteNonQueryWithMeasurements("user_scorecard_staging_load");
                        }
                    }
                }
                else
                {
                    Log.Error($"UserScorecardJob: Unable to load s3 file to mysql staging: {bucket}/{keypath} was not found.");
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserScorecardJob: {ex}");
            }
        }

        private void UpdateScorecardFromStaging(int channelDigit)
        {
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            update
                                {Constants.DatabaseSchema}microservice_twitch_user_scorecard t
                            inner join {Constants.DatabaseSchema}microservice_staging_twitch_user_auto_scorecard as s
                                on t.TwitchUserID = s.TwitchUserID
                            set
                                t.Login = s.Login,
                                t.DataSourced = s.DataSourced,
                                t.HasProfileImage = s.HasProfileImage,
                                t.HasUsedEvents = s.HasUsedEvents,
                                t.MeetsOrExceedsChannelPanelCount = s.MeetsOrExceedsChannelPanelCount,
                                t.HasUsedRules = s.HasUsedRules,
                                t.MeetsOrExceedsCollectionsCount = s.MeetsOrExceedsCollectionsCount,
                                t.AutomodEnabled = s.AutomodEnabled,
                                t.MeetsOrExceedsVodCount = s.MeetsOrExceedsVodCount,
                                t.MeetsOrExceedsHighlightCount = s.MeetsOrExceedsHighlightCount,
                                t.MeetsOrExceedsUploadCount = s.MeetsOrExceedsUploadCount,
                                t.MeetsOrExceedsEmoteCount = s.MeetsOrExceedsEmoteCount,
                                t.MeetsOrExceedsCheermoteCount = s.MeetsOrExceedsCheermoteCount,
                                t.MeetsOrExceedsClipCount = s.MeetsOrExceedsClipCount,
                                t.MeetsOrExceedsTenureBadgeCount = s.MeetsOrExceedsTenureBadgeCount,
                                t.MeetsOrExceedsExtensionInstallCount = s.MeetsOrExceedsExtensionInstallCount,
                                t.MeetsOrExceedsModeratorCount = s.MeetsOrExceedsModeratorCount,
                                t.MeetsOrExceedsTagCount = s.MeetsOrExceedsTagCount
                            where
                                t.TwitchUserID = s.TwitchUserID
                                and right(t.TwitchUserID, 1) = @channelDigit
                                and right(s.TwitchUserID, 1) = @channelDigit
                                and t.TwitchUserID > 0
                            ;

                            insert into {Constants.DatabaseSchema}microservice_twitch_user_scorecard
                            (
                                TwitchUserID, Login, DataSourced, HasProfileImage, HasUsedEvents, MeetsOrExceedsChannelPanelCount, HasUsedRules,
                                MeetsOrExceedsCollectionsCount, AutomodEnabled, MeetsOrExceedsVodCount, MeetsOrExceedsHighlightCount, MeetsOrExceedsUploadCount,
                                MeetsOrExceedsEmoteCount, MeetsOrExceedsCheermoteCount, MeetsOrExceedsClipCount, MeetsOrExceedsTenureBadgeCount,
                                MeetsOrExceedsExtensionInstallCount, MeetsOrExceedsModeratorCount, MeetsOrExceedsTagCount
                            )
                            select
                                TwitchUserID, Login, DataSourced, HasProfileImage, HasUsedEvents, MeetsOrExceedsChannelPanelCount, HasUsedRules,
                                MeetsOrExceedsCollectionsCount, AutomodEnabled, MeetsOrExceedsVodCount, MeetsOrExceedsHighlightCount, MeetsOrExceedsUploadCount,
                                MeetsOrExceedsEmoteCount, MeetsOrExceedsCheermoteCount, MeetsOrExceedsClipCount, MeetsOrExceedsTenureBadgeCount,
                                MeetsOrExceedsExtensionInstallCount, MeetsOrExceedsModeratorCount, MeetsOrExceedsTagCount
                            from {Constants.DatabaseSchema}microservice_staging_twitch_user_auto_scorecard as a
                            where not exists
                            (
                                select 1
                                from {Constants.DatabaseSchema}microservice_twitch_user_scorecard as b
                                where a.TwitchUserID = b.TwitchUserID
                                and right(b.TwitchUserID, 1) = @channelDigit
                            )
                            and right(a.TwitchUserID, 1) = @channelDigit
                            and a.TwitchUserID > 0
                            ;                       
                        ";
                        command.Parameters.AddWithValue("@channelDigit", channelDigit);
                        command.ExecuteNonQueryWithMeasurements("user_scorecard_channel_digit_update");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserScorecardJob: {ex}");
            }
        }
    }
}
