﻿using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Extensions;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace Resonance.Jobs.Amp.TwitchUser
{
    public class UserWeeklySummary : JobBase, IJob<long>
    {
        private static TwitchUserWeeklySummaryConfiguration jobConfig { get; set; }

        public UserWeeklySummary(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                jobConfig = new TwitchUserWeeklySummaryConfiguration()
                {
                    MinimumDollarThreshold = 50
                };
                Log.Info($"UserWeeklySummary: configured. IsActive: {Config.IsActive}");
            }
            catch(Exception ex)
            {
                Log.Error($@"UserWeeklySummary: {ex}");
            }
        }

        public override void Run()
        {
            try
            {
                Log.Info($"UserWeeklySummary: Running");
                this.Config.IsRunning = true;
                if (jobConfig != null)
                {
                    long resultCount = 0;
                    WipeOldData();
                    var rundate = DateTime.UtcNow.Date;
                    string mindatestring = rundate.AddDays(-365).ToString("yyyy-MM-dd");
                    var param = new Dictionary<string, dynamic>()
                    {
                        {"@mindatestring", mindatestring },
                        {"@dollarthreshold", jobConfig.MinimumDollarThreshold },
                        {"@channeldigit", 0 }
                    };

                    var sql = GetUserWeeklySummarySql();
                    Log.Verbose($@"UserWeeklySummary: Executing {sql}");
                    foreach(var p in param)
                    {
                        Log.Verbose($@"UserWeeklySummary: Param {p.Key}={p.Value}::{p.Value.GetType()}");
                    }
                    for (var channelDigit = 0; channelDigit <= 9; channelDigit++)
                    {
                        param["@channeldigit"] = channelDigit;
                        Log.Verbose($@"UserWeeklySummary: Param @channeldigit={param["@channeldigit"]}::{param["@channeldigit"].GetType()}");
                        var bucket = "crs-data-export";
                        var keyfolderpath = $"{Constants.AppConfig.Application.Environment}/twitch-user-weekly-summary/{rundate.ToString("yyyy")}/{rundate.ToString("MM")}/{rundate.ToString("dd")}/ends_with_{channelDigit}/";
                        var results = new Dictionary<long, List<TwitchUserWeeklySummaryQueryModel>>();

                        try
                        {
                            using (var conn = DBManagerRedshift.TahoeConnection(true))
                            {
                                using (var command = conn.GetCommand())
                                {
                                    command.CommandTimeout = 86400;
                                    command.CommandText = sql;
                                    foreach(var p in param)
                                    {
                                        command.Parameters.AddWithValue(p.Key, p.Value);
                                    }
                                    using (var reader = new DataReaderWithMeasurements(command, null, "get_user_weekly_summary").NpgReader)
                                    {
                                        if (reader.HasRows)
                                        {
                                            while (reader.Read())
                                            {
                                                long twitchUserID = reader.GetInt64(0);
                                                if (!results.ContainsKey(twitchUserID))
                                                {
                                                    results.Add(twitchUserID, new List<TwitchUserWeeklySummaryQueryModel>());
                                                }
                                                var item = new TwitchUserWeeklySummaryQueryModel();
                                                item.Week = reader.GetDateTime(1);
                                                item.TotalChatMessages = reader.GetInt64(2);
                                                item.UniqueChatters = reader.GetInt32(3);
                                                item.SubsTotal = reader.GetInt32(4);
                                                item.MinutesWatched = reader.GetInt64(5);
                                                item.HostedMinutesWatched = reader.GetInt64(6);
                                                results[twitchUserID].Add(item);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }

                        if(results.Count > 0)
                        {
                            resultCount += results.Count;
                            WriteTwitchUserWeeklySummaryToS3(channelDigit, bucket, keyfolderpath, results);
                            results.Clear();
                        }
                    }

                    try
                    {
                        using (var conn = DBManagerMysql.GetConnection(true))
                        {
                            using
                            (
                                var command =
                                    AmpQuerySql.InsertUpdateEtlTracking
                                    (
                                        source: "Microservice_User_Weekly_Summary_Job",
                                        target: $"Microservice_User_Weekly_Summary",
                                        method: "Jobs.UserWeeklySummary",
                                        timestamp: DateTime.UtcNow,
                                        rowcount: resultCount
                                    )
                            )
                            {
                                command.Connection = conn;
                                command.CommandTimeout = 0;
                                command.ExecuteNonQueryWithMeasurements("update_weekly_summary_etl");
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        Log.Error($@"UserListingJob: {ex}");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserWeeklySummary: {ex}");
            }
            finally
            {
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromDays(1));
            }

            Log.Info($@"UserWeeklySummary Complete. Next Run Time: {this.Config.NextRunTime.Value.ToString("yyyy-MM-dd HH:mm:ss")}");
        }

        private void WipeOldData()
        {
            Log.Info($"UserWeeklySummary: Wiping old weekly summary data from staging table");
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText =
                    $@"
                        truncate table {Constants.DatabaseSchema}microservice_staging_twitch_user_weekly_summary;
                    ";
                    command.ExecuteNonQueryWithMeasurements("user_weekly_summary_cleanup");
                }
            }
            Log.Info($"UserWeeklySummary: Completed wiping old weekly summary data from staging table");
        }

        private long WriteTwitchUserWeeklySummaryToS3(int channelID, string bucket, string keyfolderpath, Dictionary<long, List<TwitchUserWeeklySummaryQueryModel>> data)
        {
            long resultCount = 0;

            try
            {
                var serializedData = new Dictionary<long, string>();
                foreach (var item in data)
                {
                    serializedData.Add(item.Key, JsonConvert.SerializeObject(item.Value).GzipBase64Encode());
                }

                var partList = new List<Dictionary<long, string>>();
                var loop = 0;
                var itemList = new Dictionary<long, string>();
                foreach (var item in serializedData)
                {
                    itemList.Add(item.Key, item.Value);

                    if (loop == 10000)
                    {
                        partList.Add(itemList);
                        itemList.Clear();
                        loop = 0;
                    }
                    loop++;
                }

                for (var i = 0; i < partList.Count; i++)
                {
                    try
                    {
                        var keypath = $"{keyfolderpath}weekly-data-{i}.csv";
                        using (var stream = new MemoryStream())
                        {
                            using (TextWriter writer = new StreamWriter(stream, Encoding.UTF8, bufferSize: 2048, leaveOpen: true))
                            {
                                using (var csv = new CsvHelper.CsvWriter(writer, new CsvHelper.Configuration.Configuration()
                                {
                                    Delimiter = ",",
                                    Quote = '"',
                                    ShouldQuote = (field, context) => true,
                                    PrepareHeaderForMatch = (header, context) => header.ToLower(),
                                    Encoding = Encoding.UTF8,
                                },
                                leaveOpen: true))
                                {
                                    var records = partList[i].Select(x => new
                                    {
                                        ChannelID = x.Key,
                                        LastUpdateDate = DateTime.UtcNow.ToRedshiftDateFormat(),
                                        Data = x.Value
                                    }).ToArray();
                                    resultCount += records.Length;
                                    Log.Info($"UserWeeklySummary: Adding {records.Length} records to CSV");
                                    csv.WriteRecords(records);
                                }
                            }
                            stream.Position = 0;
                            Log.Info($@"UserWeeklySummary: Writing to S3 {bucket}/{keypath}");
                            S3Helper.UploadToS3(stream, bucket, keypath, Constants.AppConfig.Application.KmsArn);
                            Log.Info($@"UserWeeklySummary: Successfully wrote {bucket}/{keypath}");
                            stream.Close();
                            stream.Dispose();
                        }

                        if (S3Helper.ExistsInS3(bucket, keypath) == true)
                        {
                            Log.Info($"UserWeeklySummary: Uploading {bucket}/{keypath} to MySql for Channel Digit: {i}, Loop: {loop}");

                            try
                            {
                                using (var conn = DBManagerMysql.GetConnection(true))
                                {
                                    using (var command = conn.GetCommand())
                                    {
                                        var quotechar = "\"";
                                        command.CommandText =
                                        $@"
                                            load data from s3 's3://{bucket}/{keypath}'
                                            replace
                                            into table {Constants.DatabaseSchema}microservice_staging_twitch_user_weekly_summary
                                            character set 'utf8'
                                            fields terminated by ','
                                            enclosed by '{quotechar}'
                                            lines terminated by '\r\n'
                                            ignore 1 lines
                                            (TwitchUserID, LastUpdateDate, Data)
                                            ;
                                        ";
                                        command.ExecuteNonQueryWithMeasurements("user_weekly_summary_staging_load");
                                        Log.Info($"UserWeeklySummary: Completed uploading {bucket}/{keypath} to MySql for Channel Digit: {i}, Loop: {loop}");
                                    }
                                    Log.Info($"UserWeeklySummary: Updating {Constants.DatabaseSchema}microservice_twitch_user_weekly_summary");
                                }
                            }
                            catch (Exception ex)
                            {
                                Log.Error($@"UserWeeklySummary: {ex}");
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        Log.Error(ex);
                    }
                }

                try
                {
                    using (var conn = DBManagerMysql.GetConnection(true))
                    {
                        using (var command = conn.GetCommand())
                        {
                            command.CommandTimeout = 86400;
                            command.Parameters.AddWithValue("@channelDigit", channelID);
                            command.CommandText =
                            $@"
                                update 
                                    {Constants.DatabaseSchema}microservice_twitch_user_weekly_summary t
                                inner join {Constants.DatabaseSchema}microservice_staging_twitch_user_weekly_summary as s
                                    on t.TwitchUserID = s.TwitchUserID
                                set
                                    t.LastUpdateDate = s.LastUpdateDate,
                                    t.Data = s.Data
                                where
                                    right(s.TwitchUserID, 1) = @channelDigit
                                    and right(t.TwitchUserID, 1) = @channelDigit
                                ;

                                insert into {Constants.DatabaseSchema}microservice_twitch_user_weekly_summary(TwitchUserID, LastUpdateDate, Data)
                                select
                                    TwitchUserID,
                                    LastUpdateDate,
                                    Data
                                from {Constants.DatabaseSchema}microservice_staging_twitch_user_weekly_summary as a
                                where
                                    not exists
                                    (
                                        select 1
                                        from {Constants.DatabaseSchema}microservice_twitch_user_weekly_summary as b
                                        where a.TwitchUserID = b.TwitchUserID
                                    )
                                    and right(TwitchUserID, 1) = @channelDigit
                                ;
                            ";
                            command.ExecuteNonQueryWithMeasurements("user_weekly_summary_update");
                        }
                    }
                }
                catch(Exception ex)
                {
                    Log.Error(ex);
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserWeeklySummary: {ex}");
            }
            return resultCount;
        }

        private string GetUserWeeklySummarySql()
        {
            string sql =
            $@"
                /* BEGIN TWITCH USER WEEKLY SUMMARY QUERY */
                with apdcs as
                (
                    select cast(channel_id as bigint) as channel_id
                    from cubes.amp_application_population
                    group by cast(channel_id as bigint)
                ), apdcs_subs_weekly as
                (
                    select
                        channel_id as channel_id,
                        date_trunc('week', day) as week,
                        sum(subs_sold_total) as subs_total
                    from cubes.affiliates_partners_daily_channel_summary
                    where
                        day >= cast(@mindatestring as datetime)
                        and coalesce(channel_id, 0) > 0
                        and right(channel_id, 1) = @channelDigit
                    group by 
                        channel_id,
                        date_trunc('week', day)
                ), messages_raw as
                (
                    select
                        channel_id,
                        date_trunc('week', date) as week,
                        user_id,
                        count(*) as messages_sent
                    from tahoe.server_chat_message 
                    where 
                        date >= cast(@mindatestring as datetime)
                        and chatroom_type = 'twitch_chat'
                        and right(channel_id, 1) = @channelDigit
                    group by
                        channel_id,
                        date_trunc('week', date),
                        user_id
                ), unique_weeks as
                (
                    select week
                    from messages_raw
                    group by week
                ), unique_messages as
                (
                    select
                        channel_id,
                        week,
                        sum(messages_sent) as total_chat_messages
                    from messages_raw
                    where right(channel_id, 1) = @channelDigit
                    group by
                        channel_id,
                        week
                ), unique_users as
                (
                    select
                        channel_id,
                        week,
                        count(*) as unique_user_count
                    from messages_raw
                    where right(channel_id, 1) = @channelDigit
                    group by
                        channel_id,
                        week
                ), minutes_watched as
                (
                    select
                        coalesce(host_channel_id, channel_id) as channel_id,
                        date_trunc('week', day) as week,
                        sum(mw) as minutes_watched,
                        sum(case when host_channel_id is not null then mw else 0 end) as hosted_minutes_watched
                    from cubes.hours_watched_daily_by_device
                    where 
                        day >= cast(@mindatestring as datetime)
                        and right(coalesce(host_channel_id, channel_id), 1) = @channelDigit
                    group by
                        coalesce(host_channel_id, channel_id),
                        date_trunc('week', day)
                )
                select
                    apdcs_week.channel_id,
                    apdcs_week.week,
                    coalesce(unique_messages.total_chat_messages, 0) as total_chat_messages,
                    coalesce(unique_users.unique_user_count, 0) as unique_user_count,
                    coalesce(apdcs_subs_weekly.subs_total, 0) as subs_total,
                    coalesce(minutes_watched.minutes_watched, 0) as minutes_watched,
                    coalesce(minutes_watched.hosted_minutes_watched, 0) as hosted_minutes_watched
                from
                (
                    select channel_id, week
                    from apdcs
                    cross join unique_weeks
                ) as apdcs_week
                left join unique_messages
                    on apdcs_week.channel_id = unique_messages.channel_id
                    and apdcs_week.week = unique_messages.week
                left join unique_users
                    on apdcs_week.channel_id = unique_users.channel_id
                    and apdcs_week.week = unique_users.week
                left join apdcs_subs_weekly
                    on apdcs_week.channel_id = apdcs_subs_weekly.channel_id
                    and apdcs_week.week = apdcs_subs_weekly.week
                left join minutes_watched
                    on apdcs_week.channel_id = minutes_watched.channel_id
                    and apdcs_week.week = minutes_watched.week
                ;
                /* END TWITCH USER WEEKLY SUMMARY QUERY */
            ";
            return sql;
        }

        private void UpdateScorecardFromStaging(int channelDigit)
        {
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 86400;
                        command.CommandText =
                        $@"
                            update
                                {Constants.DatabaseSchema}microservice_twitch_user_scorecard t
                            inner join {Constants.DatabaseSchema}microservice_staging_twitch_user_auto_scorecard as s
                                on t.TwitchUserID = s.TwitchUserID
                            set
                                t.Login = s.Login,
                                t.DataSourced = s.DataSourced,
                                t.HasProfileImage = s.HasProfileImage,
                                t.HasUsedEvents = s.HasUsedEvents,
                                t.MeetsOrExceedsChannelPanelCount = s.MeetsOrExceedsChannelPanelCount,
                                t.HasUsedRules = s.HasUsedRules,
                                t.MeetsOrExceedsCollectionsCount = s.MeetsOrExceedsCollectionsCount,
                                t.AutomodEnabled = s.AutomodEnabled,
                                t.MeetsOrExceedsVodCount = s.MeetsOrExceedsVodCount,
                                t.MeetsOrExceedsHighlightCount = s.MeetsOrExceedsHighlightCount,
                                t.MeetsOrExceedsUploadCount = s.MeetsOrExceedsUploadCount,
                                t.MeetsOrExceedsEmoteCount = s.MeetsOrExceedsEmoteCount,
                                t.MeetsOrExceedsCheermoteCount = s.MeetsOrExceedsCheermoteCount,
                                t.MeetsOrExceedsClipCount = s.MeetsOrExceedsClipCount,
                                t.MeetsOrExceedsTenureBadgeCount = s.MeetsOrExceedsTenureBadgeCount,
                                t.MeetsOrExceedsExtensionInstallCount = s.MeetsOrExceedsExtensionInstallCount,
                                t.MeetsOrExceedsModeratorCount = s.MeetsOrExceedsModeratorCount
                            where
                                t.TwitchUserID = s.TwitchUserID
                                and right(t.TwitchUserID, 1) = @channelDigit
                                and right(s.TwitchUserID, 1) = @channelDigit
                                and t.TwitchUserID > 0
                            ;

                            insert into {Constants.DatabaseSchema}microservice_twitch_user_scorecard
                            (
                                TwitchUserID, Login, DataSourced, HasProfileImage, HasUsedEvents, MeetsOrExceedsChannelPanelCount, HasUsedRules,
                                MeetsOrExceedsCollectionsCount, AutomodEnabled, MeetsOrExceedsVodCount, MeetsOrExceedsHighlightCount, MeetsOrExceedsUploadCount,
                                MeetsOrExceedsEmoteCount, MeetsOrExceedsCheermoteCount, MeetsOrExceedsClipCount, MeetsOrExceedsTenureBadgeCount,
                                MeetsOrExceedsExtensionInstallCount
                            )
                            select
                                TwitchUserID, Login, DataSourced, HasProfileImage, HasUsedEvents, MeetsOrExceedsChannelPanelCount, HasUsedRules,
                                MeetsOrExceedsCollectionsCount, AutomodEnabled, MeetsOrExceedsVodCount, MeetsOrExceedsHighlightCount, MeetsOrExceedsUploadCount,
                                MeetsOrExceedsEmoteCount, MeetsOrExceedsCheermoteCount, MeetsOrExceedsClipCount, MeetsOrExceedsTenureBadgeCount,
                                MeetsOrExceedsExtensionInstallCount
                            from {Constants.DatabaseSchema}microservice_staging_twitch_user_auto_scorecard as a
                            where not exists
                            (
                                select 1
                                from {Constants.DatabaseSchema}microservice_twitch_user_scorecard as b
                                where a.TwitchUserID = b.TwitchUserID
                                and right(b.TwitchUserID, 1) = @channelDigit
                            )
                            and right(a.TwitchUserID, 1) = @channelDigit
                            and a.TwitchUserID > 0
                            ;                       
                        ";
                        command.Parameters.AddWithValue("@channelDigit", channelDigit);
                        command.ExecuteNonQueryWithMeasurements("user_weekly_summary_update");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error($@"UserScorecardJob: {ex}");
            }
        }
    }
}
