﻿using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using System;
using System.Collections.Generic;
using Resonance.Jobs.Atlas.ContractAutoExtender.Methods;
using Resonance.Jobs.Atlas.ContractAutoExtender.Query;
using Resonance.Jobs.Atlas.ContractAutoExtender.Interfaces;

namespace Resonance.Jobs.Atlas.Contracts
{
    public class ContractAutoExtenderJobResult
    {
        public int ContractsFound { get; set; }
        public int ContractsExtended { get; set; }
        public int ContractsUpdated { get; set; }
    }

    public class ContractAutoExtenderJob : JobBase
    {
        public static readonly string QueryingContractsMsg = "Querying for contracts to be extended";
        public static readonly string NoContractsFoundMsg = "No contracts found. Exiting";
        public static readonly string DoneMsg = "ContractAutoExtender: Done";                

        public ContractAutoExtenderJob(JobConfiguration config)
        {
            if (config == null)
            {
                throw new ArgumentNullException(nameof(config));
            }

            Config = config;            
        }

        public override void Run()
        {
            ExtenderMethods methods = new ExtenderMethods(new ExtenderQuery(), new ActivityLogger());
            RunJob(methods, message => Log.Info(message), error => Log.Error(error));
        }
        
        public ContractAutoExtenderJobResult RunJob(IContractExtender extender, Action<string> info, Action<Exception> error)
        {
            ContractAutoExtenderJobResult result = new ContractAutoExtenderJobResult();

            try
            {
                this.Config.IsRunning = true;
                this.Config.LastRunTime = DateTime.UtcNow;
                info?.Invoke(QueryingContractsMsg);
                IList<AtlasContract> contracts = extender.GetContractsForAutoExtension();

                result.ContractsFound = contracts.Count;

                if (contracts.Count == 0)
                { //no contracts to update
                    info?.Invoke(NoContractsFoundMsg);
                    return result;
                }
                
                info?.Invoke($"{contracts.Count} contracts found");                

                info?.Invoke("Extending contracts");
                var extendedContracts = extender.ExtendContracts(contracts);
                result.ContractsExtended = extendedContracts.Count;
                info?.Invoke(string.Format("{0} contracts have been extended", result.ContractsExtended));

                info?.Invoke("Updating database");                
                result.ContractsUpdated = extender.UpdateDatabase(extendedContracts);
                info?.Invoke(string.Format("{0} contract(s) updated in database", result.ContractsUpdated));
            }
            catch (Exception ex)
            {                
                error?.Invoke(ex);
            }
            finally
            {                
                info?.Invoke(DoneMsg);
                this.Config.IsRunning = false;
                Config.NextRunTime = Config.LastRunTime.Value.AddHours(24);
            }

            return result;
        }
    }
}
