﻿using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using Resonance.Jobs.Atlas.ContractAutoExtender.Interfaces;
using System;
using System.Collections.Generic;

namespace Resonance.Jobs.Atlas.ContractAutoExtender.Methods
{
    public class ExtenderMethods : IContractExtender
    {
        private IExtenderQuery _query;
        private IActivityLogger _activityLogger;

        public ExtenderMethods(IExtenderQuery query, IActivityLogger activityLogger)
        {
            if (query == null)
            {
                throw new ArgumentNullException(nameof(query));
            }            

            _query = query;
            _activityLogger = activityLogger;
        }
        
        public IList<AtlasContract> ExtendContracts(IList<AtlasContract> contracts)
        {
            IList<AtlasContract> extendedContracts = new List<AtlasContract>();
            
            if (contracts != null)
            {                   
                foreach (var contract in contracts)
                {
                    AtlasContract temp = ExtendContract(contract);
                    if (temp != null)
                    {
                        extendedContracts.Add(temp);
                    }                    
                }
            }

            return extendedContracts;
        }

        private AtlasContract ExtendContract(AtlasContract contract)
        {
            //This check is just an extra precaution and helps facilitate unit tests
            //The database query that pulls back contracts should only return 
            //contracts that need renewing
            if (contract.AutoRenew && contract.EndDate <= DateTime.UtcNow.Date)
            {
                var extendedContract = new AtlasContract
                {
                    ContractID = contract.ContractID,
                    StartDate = contract.StartDate,
                    EndDate = contract.EndDate
                };

                //store the original start / end dates in the original fields
                extendedContract.OriginalStartDate = contract.StartDate;
                extendedContract.OriginalEndDate = contract.EndDate;

                //extend the end date
                extendedContract.EndDate = contract.EndDate.AddYears(1);
                extendedContract.AutoRenew = false;

                return extendedContract;
            }

            return null;
        }

        public IList<AtlasContract> GetContractsForAutoExtension()
        {
            return _query.GetContractsForAutoExtension();
        }

        public int UpdateDatabase(IList<AtlasContract> contracts)
        {
            int totalUpdated = 0;
            foreach (var contract in contracts)
            {
                int updated = _query.UpdateContractDB(contract);
                totalUpdated += updated;

                string msg = updated > 0
                    ? GetSuccessUpdateMessage(contract.ContractID, contract.OriginalEndDate, contract.EndDate)
                    : GetFailedUpdateMessage(contract.ContractID);

                _activityLogger?.Info(msg);
            }

            return totalUpdated;
        }

        public static string GetSuccessUpdateMessage(int contractID, DateTime? originalEndDate, DateTime newEndDate)
        {
            string msg = string.Format("Contract {0} extended from {1} to {2}", contractID, originalEndDate.Value.ToString("MM/dd/yyyy"), newEndDate.ToString("MM/dd/yyyy"));
            return msg;
        }

        public static string GetFailedUpdateMessage(int contractID)
        {
            return string.Format("Unable to extend contract {0}", contractID);
        }
    }
}
