﻿using MySql.Data.MySqlClient;
using Resonance.Core;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using Resonance.Jobs.Atlas.ContractAutoExtender.Interfaces;
using System;
using System.Collections.Generic;
using System.Text;

namespace Resonance.Jobs.Atlas.ContractAutoExtender.Query
{
    public class ExtenderQuery : IExtenderQuery
    {
        public ExtenderQuery()
        {

        }

        public IList<AtlasContract> GetContractsForAutoExtension()
        {
            string q = $@"
                    SELECT contract_id, start_date, end_date, auto_renew
                    from {Constants.DatabaseSchema}microservice_twitch_atlas_contracts
                    where auto_renew = 1 and end_date <= @today                    
                    ;";
            List<AtlasContract> contracts = new List<AtlasContract>();

            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = q;
                    DateTime today = DateTime.Now.Date;
                    command.Parameters.AddWithValue("today", today);
                    using (var reader = new DataReaderWithMeasurements(command, null, "get_contract_auto_extend").MysqlReader)

                        while (reader.Read())
                    {
                        AtlasContract contract = new AtlasContract()
                        {
                            ContractID = (int)reader["contract_id"],                            
                            StartDate = (DateTime)reader["start_date"],
                            EndDate = (DateTime)reader["end_date"],         
                            AutoRenew = reader["auto_renew"] != DBNull.Value ? ((UInt64)reader["auto_renew"] == 1) : false
                        };

                        contracts.Add(contract);
                    }
                }
            }

            return contracts;
        }

        public int UpdateContractDB(AtlasContract contract)
        {
            int updated = 0;

            using (var conn = DBManagerMysql.GetConnection())
            {
                using (var command = GetUpdateCommand(contract, conn))
                {   
                    updated = command.ExecuteNonQueryWithMeasurements("update_contract_auto_extend");
                }
            }

            return updated;
        }

        private MySqlCommand GetUpdateCommand(AtlasContract contract, MySqlConnection conn)
        {
            string dateFmt = "yyyy-MM-dd";            
            MySqlCommand cmd = conn.GetCommand();
            cmd.CommandText = getUpdateSql(contract);            

            string endDate = contract.EndDate.ToString(dateFmt);
            string originalStartDate = contract.OriginalStartDate.Value.ToString(dateFmt);
            string originalEndDate = contract.OriginalEndDate.Value.ToString(dateFmt);

            cmd.Parameters.AddWithValue("contract_id", contract.ContractID);
            cmd.Parameters.AddWithValue("end_date", endDate);                        
            cmd.Parameters.AddWithValue("original_start_date", originalStartDate);
            cmd.Parameters.AddWithValue("original_end_date", originalEndDate);
            cmd.Parameters.AddWithValue("auto_renew", contract.AutoRenew ? 1 : 0);

            return cmd;
        }

        private string getUpdateSql(AtlasContract contract)
        {
            return $@"
                    update {Constants.DatabaseSchema}microservice_twitch_atlas_contracts
                    set
                        end_date = @end_date,
                        original_start_date = @original_start_date,
                        original_end_date = @original_end_date,
                        auto_renew = @auto_renew

                    where contract_id = @contract_id
                    ";
        }
    }
}
