﻿using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using System;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core;
using System.Collections.Generic;
using Resonance.Core.Models.MetaDataModels;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Models.ServiceModels.ActivityLoggerService;
using System.IO;
using System.Text;
using System.Linq;
using Resonance.Core.Helpers.StringHelpers;
using System.Text.RegularExpressions;
using System.Globalization;
using System.Diagnostics;
using Resonance.Core.Extensions;

namespace Resonance.Jobs.Atlas.DataPopulation
{
    public class ActivityLogImporter : JobBase, IJob<long>
    {
        private const string timePattern = "Atlas_(\\d+)_(\\d+)\\.";
        private static Regex timeRegex = new Regex(timePattern, RegexOptions.Singleline | RegexOptions.Compiled | RegexOptions.IgnoreCase);

        public ActivityLogImporter(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                Log.Info($@"ActivityLogImporter: configured. IsActive: {Config.IsActive}");
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }

        public override void Run()
        {
            try
            {
                this.Config.IsRunning = true;
                Log.Info("ActivityLogImporter: Starting");
                SaveS3FileList(GetS3FileList());
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                Log.Info("ActivityLogImporter: Done");
                this.Config.IsActive = false;
                this.Config.IsRunning = false;
            }
        }

        private List<ActivityLogDataRds> GetS3FileList()
        {
            var result = new List<ActivityLogDataRds>();

            var backfillDate = new DateTime(2020, 05, 01, 0, 0, 0, DateTimeKind.Utc);
            var loopDate = backfillDate;
            var endDate = DateTime.UtcNow;
            Log.Info($@"Start Date: {loopDate.ToString("yyyy-MM-dd HH:mm:ss")}, End Date: {endDate}");

            bool continueProcessing = true;
            var bucket = "resonance-activity-logs";

            while (loopDate <= endDate && continueProcessing)
            {
                try
                {
                    var searchPath = $@"{Constants.AppConfig.Application.Environment}/logs/Atlas/Atlas_{loopDate.ToString("yyyyMMdd")}_";
                    Log.Verbose($@"Searching {bucket}/{searchPath}");
                    var files = S3Helper.GetAllFilesAtPath(bucket, searchPath);
                    if(files != null && files.Length > 0)
                    {
                        Log.Verbose($@"Found {files.Length} files");
                        foreach (var fileGroup in files)
                        {
                            if(fileGroup.S3Objects.Count > 0)
                            {
                                var stopwatch = new Stopwatch();
                                foreach (var s3File in fileGroup.S3Objects)
                                {
                                    if (timeRegex.IsMatch(s3File.Key))
                                    {
                                        stopwatch.Start();
                                        var match = timeRegex.Match(s3File.Key);
                                        var dateTimeString = $@"{match.Groups[1].Captures[0].Value} {match.Groups[2].Captures[0].Value}";
                                        var date = DateTime.ParseExact(dateTimeString, "yyyyMMdd HHmmss", new CultureInfo("en-US"));
                                        if(date <= new DateTime(2020, 05, 01, 0, 0, 0, DateTimeKind.Utc))
                                        {
                                            continue;
                                        }
                                        var s3FileText = S3Helper.ReadTextFromS3(s3File.BucketName, s3File.Key, false);

                                        // Writing to a stream and reading it back is pretty hacky, TODO: See if we can return the s3 stream directly
                                        using (var stream = new MemoryStream())
                                        {
                                            using (var writer = new StreamWriter(stream, Encoding.UTF8, s3FileText.Length, true))
                                            {
                                                writer.Write(s3FileText);
                                            }
                                            stream.Position = 0;
                                            using (var reader = new StreamReader(stream, Encoding.UTF8))
                                            {
                                                using (var parser = new CsvHelper.CsvParser(reader, false))
                                                {
                                                    var linecount = 0;
                                                    while (true)
                                                    {
                                                        var row = parser.Read();
                                                        if (row == null)
                                                        {
                                                            break;
                                                        }
                                                        // Ignore the header
                                                        if (linecount > 0)
                                                        {
                                                            var item = new ActivityLogDataRds()
                                                            {
                                                                MessageTimestamp = DateTime.Parse(row[0]),
                                                                Message = row[1],
                                                                Controller = row[2],
                                                                Action = row[3],
                                                                User = row[4],
                                                                TrackingID = s3File.Key,
                                                                TrackingTag = s3File.ETag,
                                                                TrackingDate = date,
                                                                Hash = HashHelper.SimpleRandomLetterSequence.RandomHash()
                                                            };
                                                            result.Add(item);
                                                        }
                                                        linecount++;
                                                    }
                                                }
                                            }
                                        }
                                        stopwatch.Stop();
                                        Log.Verbose($@"{s3File.BucketName}/{s3File.Key} took {stopwatch.ElapsedMilliseconds}ms to process");
                                    }
                                    stopwatch.Reset();
                                }
                            }
                        }
                    }
                }
                catch(Exception ex)
                {
                    Log.Error(ex);
                    continueProcessing = false;
                    Log.Info($@"Import frozen, failure while processing {loopDate.ToString("yyyy-MM-dd")}");
                }
                loopDate = loopDate.AddDays(1);
            }
            Log.Info($@"Import complete");

            return result;
        }


        private void SaveS3FileList(List<ActivityLogDataRds> data)
        {
            if(data == null || data.Count == 0)
            {
                return;
            }

            try
            {
                Log.Info($@"Preparing to insert {data.Count} rows");
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var transaction = conn.BeginTransaction())
                    {
                        try
                        {
                            using (var command = conn.GetCommand())
                            {
                                command.Transaction = transaction;
                                command.CommandText =
                                $@"
                                    insert into {Constants.DatabaseSchema}microservice_twitch_atlas_activity_log
                                    (
                                        message_timestamp, message, controller, action, user, 
                                        tracking_id, tracking_tag, tracking_date, hash
                                    )
                                    select @message_timestamp, @message, @controller, @action, @user, 
                                    @tracking_id, @tracking_tag, @tracking_date, @hash
                                ;";
                                var counter = 0;
                                foreach (var row in data)
                                {
                                    command.Parameters.Clear();
                                    command.Parameters.AddWithValue("@message_timestamp", row.MessageTimestamp);
                                    command.Parameters.AddWithValue("@message", row.Message);
                                    command.Parameters.AddWithValue("@controller", row.Controller);
                                    command.Parameters.AddWithValue("@action", row.Action);
                                    command.Parameters.AddWithValue("@user", row.User);
                                    command.Parameters.AddWithValue("@tracking_id", row.TrackingID);
                                    command.Parameters.AddWithValue("@tracking_tag", row.TrackingTag);
                                    command.Parameters.AddWithValue("@tracking_date", row.TrackingDate);
                                    command.Parameters.AddWithValue("@hash", row.Hash);
                                    command.ExecuteNonQueryWithMeasurements("activity_log_import");
                                    counter++;
                                    if(counter % 1000 == 0)
                                    {
                                        Log.Verbose($@"Processed {counter} records");
                                    }
                                }
                            }
                            transaction.Commit();
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                            transaction.Rollback();
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }
    }
}