﻿using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using Resonance.Jobs.Amp;
using System;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using System.Collections.Generic;
using Resonance.Core.Helpers.DatabaseHelpers;
using System.Text;
using Resonance.Core.Models.ServiceModels.AtlasModels;
using System.Linq;
using Resonance.Core;
using Resonance.Core.Helpers.StringHelpers;

namespace Resonance.Jobs.Atlas.DataPopulation
{
    public class ChannelRevenueSummary : JobBase, IJob<long>
    {
        public ChannelRevenueSummary(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                Log.Info($@"ChannelRevenueSummary: configured. IsActive: {Config.IsActive}");
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }

        private Dictionary<int, List<PremiumContentCreatorChannelStatModel>> GetPccs()
        {
            Log.Info($@"ChannelRevenueSummary: Getting Products");
            var data = new Dictionary<int, List<PremiumContentCreatorChannelStatModel>>();
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText =
                        $@"
                            select distinct
	                            b.premium_content_creator_id,
	                            c.channel_id,
	                            b.start_time,
                                b.end_time
                            from {Constants.DatabaseSchema}microservice_twitch_atlas_event as b
                            inner join {Constants.DatabaseSchema}microservice_twitch_atlas_stream as c
	                            on b.event_id = c.event_id
                            where
	                            b.start_time <> cast('1900-01-01' as datetime)
	                            and b.end_time <> cast('9999-12-31' as datetime)
	                            and exists
	                            (
		                            select 1
		                            from {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_channel_map as d
		                            where b.premium_content_creator_id = d.premium_content_creator_id
		                            and c.channel_id = d.channel_id
	                            )
                            order by premium_content_creator_id, channel_id
                            ;
                        ";
                        using (var reader = new DataReaderWithMeasurements(command, null, "get_pccs").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    try
                                    {
                                        var pccID = reader.GetInt32(0);
                                        var channelID = reader.GetInt64(1);
                                        var startTime = reader.GetDateTime(2);
                                        var endTime = reader.GetDateTime(3);

                                        if (!data.ContainsKey(pccID) && pccID > 0)
                                        {
                                            data.Add(pccID, new List<PremiumContentCreatorChannelStatModel>());
                                        }
                                        data[pccID].Add(new PremiumContentCreatorChannelStatModel()
                                        {
                                            PremiumContentCreatorID = pccID,
                                            ChannelID = channelID,
                                            StartTime = startTime,
                                            EndTime = endTime
                                        });
                                    }
                                    catch (Exception ex)
                                    {
                                        Log.Error(ex);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return data;
        }

        private void WipeOldData()
        {
            Log.Info($@"ChannelRevenueSummary: Wiping Old Data");

            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText = $@"truncate table {Constants.DatabaseSchema}staging_microservice_twitch_atlas_channel_summary";
                        command.ExecuteNonQueryWithMeasurements("channel_summary_cleanup");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }

        public override void Run()
        {
            try
            {
                this.Config.IsRunning = true;
                Log.Info("ChannelRevenueSummary: Starting");

                WipeOldData();
                var pccs = GetPccs();
                if(pccs != null && pccs.Keys.Count > 0)
                {
                    foreach (var pcc in pccs)
                    {
                        try
                        {
                            var population = GetRequestPopulation(pcc.Key, pcc.Value);
                            if(population != null && population.Count > 0)
                            {
                                CreateChannelSummary(population);
                            }
                        }
                        catch (Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }

                    UpdateStatsSummary();
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                Log.Info("ChannelRevenueSummary: Done");
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromDays(1));
            }
        }

        private void UpdateStatsSummary()
        {
            Log.Info("ChannelRevenueSummary: Update Stats Summary");

            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText =
                        $@"
                            replace into {Constants.DatabaseSchema}microservice_twitch_atlas_aggregate_pcc_stats (premium_content_creator_id, measurement_date, total_revenue, hours_watched, hours_broadcast, last_update_time, hash)
                            select
	                            premium_content_creator_id,
	                            cast(concat(year(day), '-', month(day), '-', '01') as date) as measurement_date,
	                            sum(creator_revenue_total) as total_revenue,
	                            sum(daily_total_minutes_watched / 60.0) as hours_watched,
	                            sum(daily_total_minutes_broadcast / 60.0) as hours_broadcast,
	                            @now as last_update_time,
	                            max(hash) as hash
                            from {Constants.DatabaseSchema}staging_microservice_twitch_atlas_channel_summary
                            where cast(concat(year(day), '-', month(day), '-', '01') as date) >= date_add(day, interval -1 year)
                            group by
	                            premium_content_creator_id,
                                cast(concat(year(day), '-', month(day), '-', '01') as date)
                            ;
                        ";
                        command.Parameters.AddWithValue("@now", DateTime.UtcNow);
                        command.ExecuteNonQueryWithMeasurements("update_channel_revenue");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }

        private List<ChannelSummaryGraphDetailModel> GetRequestPopulation(int pccID, List<PremiumContentCreatorChannelStatModel> channelsRaw)
        {
            Log.Info($@"ChannelRevenueSummary: Getting Request Population for pccID {pccID}");

            var rawDetails = new List<ChannelSummaryGraphDetailModel>();

            try
            {
                var channels = new Dictionary<long, List<PremiumContentCreatorChannelStatModel>>();
                foreach(var channel in channelsRaw)
                {
                    if (!channels.ContainsKey(channel.ChannelID))
                    {
                        channels.Add(channel.ChannelID, new List<PremiumContentCreatorChannelStatModel>());
                    }
                    channels[channel.ChannelID].Add(channel);
                }
                var mindate = channels.Min(x => x.Value.Min(y => y.StartTime));
                var maxdate = channels.Max(x => x.Value.Max(y => y.EndTime));
                using (var conn = DBManagerRedshift.TahoeConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        var channelIDs = new List<long>();
                        foreach(var channel in channelsRaw.Select(x => x.ChannelID).Distinct().ToArray())
                        {
                            channelIDs.Add(channel);
                        }
                        command.CommandText =
                        $@"
                            select
                                channel_id::varchar as channel_id,
                                day,
                                sum(coalesce(ccu_minutes_watched_total, 0))::float as daily_total_minutes_watched,
                                sum(coalesce(ccu_minutes_broadcast_total, 0))::float as daily_total_minutes_broadcast,
                                sum(
                                    coalesce(creator_ad_revenue, 0.0)
                                    + coalesce(creator_sub_revenue_total, 0.0)
                                    + coalesce(creator_bits_revenue_total, 0.0)
                                    + coalesce(creator_fuel_revenue, 0.0)
                                    + coalesce(creator_bounty_board_revenue, 0.0)
                                )::float as creator_revenue_total,
                                sum(coalesce(creator_ad_revenue, 0.0))::float as ad_revenue_total,
                                sum(coalesce(creator_sub_revenue_total, 0.0))::float as sub_revenue_total,
                                sum(coalesce(creator_bits_revenue_total, 0.0))::float as bits_revenue_total,
                                sum(coalesce(creator_fuel_revenue, 0.0))::float as fuel_revenue_total,
                                sum(coalesce(creator_bounty_board_revenue, 0.0))::float as bounty_board_revenue_total
                            from cubes.creator_daily_channel_summary
                            where 
                                day >= cast('{mindate.ToString("yyyy-MM-dd")}' as datetime)
                                and day <= cast('{maxdate.ToString("yyyy-MM-dd")}' as datetime)
                                and channel_id is not null
                                and channel_id in ({string.Join(",", channelIDs.Select(x => $"'{x}'").ToArray()) /* This is only ok because of the values being of type 'long' */})
                            group by channel_id, day
                            order by channel_id, day
                            ;
                        ";
                        var debugSql = command.CommandText;
                        Log.Verbose(debugSql);
                        var unfilteredPairs = new Dictionary<long, List<ChannelSummaryGraphDetailModel>>();
                        using (var reader = new DataReaderWithMeasurements(command, null, "get_request_population").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    var item = new ChannelSummaryGraphDetailModel()
                                    {
                                        PremiumContentCreatorID = pccID,
                                        ChannelID = long.Parse(reader.GetString(0)),
                                        Day = reader.GetDateTime(1),
                                        DailyTotalMinutesWatched = (float)reader.GetDouble(2),
                                        DailyTotalMinutesBroadcast = (float)reader.GetDouble(3),
                                        CreatorRevenueTotal = (float)reader.GetDouble(4),
                                        AdRevenueTotal = (float)reader.GetDouble(5),
                                        SubRevenueTotal = (float)reader.GetDouble(6),
                                        BitsRevenueTotal = (float)reader.GetDouble(7),
                                        FuelRevenueTotal = (float)reader.GetDouble(8),
                                        BountyBoardRevenueTotal = (float)reader.GetDouble(9),
                                        Hash = HashHelper.SimpleRandomLetterSequence.RandomHash()
                                    };
                                    if (channels.ContainsKey(item.ChannelID))
                                    {
                                        if (!unfilteredPairs.ContainsKey(item.ChannelID))
                                        {
                                            unfilteredPairs.Add(item.ChannelID, new List<ChannelSummaryGraphDetailModel>());
                                        }
                                        unfilteredPairs[item.ChannelID].Add(item);
                                        
                                    }
                                }
                            }
                        }

                        foreach(var channel in channels)
                        {
                            foreach(var timepair in channel.Value)
                            {
                                if (!unfilteredPairs.ContainsKey(channel.Key))
                                {
                                    Log.Verbose($@"Missing data for {channel.Key}, {timepair.StartTime.ToString("yyyy-MM-dd")} to {timepair.EndTime.ToString("yyyy-MM-dd")}");
                                }
                                else
                                {
                                    var data = unfilteredPairs[channel.Key]?.Where(x => x.Day.Date >= timepair.StartTime.Date && x.Day.Date <= timepair.EndTime.Date).ToArray();
                                    if (data != null && data.Length > 0)
                                    {
                                        rawDetails.AddRange(data);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }

            return rawDetails;
        }

        private void CreateChannelSummary(List<ChannelSummaryGraphDetailModel> data)
        {
            Log.Info($@"ChannelRevenueSummary: Creating Summary");

            try
            {
                if(data == null || data.Count == 0)
                {
                    return;
                }
                Log.Info($@"Loading Data Staging Table");

                var sql = new StringBuilder();
                foreach(var item in data)
                {
                    sql.AppendLine($"replace into {Constants.DatabaseSchema}staging_microservice_twitch_atlas_channel_summary (premium_content_creator_id, channel_id, day, daily_total_minutes_watched, daily_total_minutes_broadcast, creator_revenue_total, ad_revenue_total, sub_revenue_total,  bits_revenue_total, fuel_revenue_total, bounty_board_revenue_total, hash) values ({item.PremiumContentCreatorID},{item.ChannelID},'{item.Day.ToString("yyyy-MM-dd HH:mm:ss")}', {item.DailyTotalMinutesWatched}, {item.DailyTotalMinutesBroadcast}, {item.CreatorRevenueTotal}, {item.AdRevenueTotal}, {item.SubRevenueTotal}, {item.BitsRevenueTotal}, {item.FuelRevenueTotal}, {item.BountyBoardRevenueTotal}, '{HashHelper.SimpleRandomLetterSequence.RandomHash()}');");
                }
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText = sql.ToString();
                        command.ExecuteNonQueryWithMeasurements(logname: "staging_channel_revenue");
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }

            Log.Info($@"Completed Loading Data Staging Table");
        }
    }
}
