﻿using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Resonance.Core.Helpers.StatsDHelpers;
using Resonance.Jobs.Atlas.Ldap;
using Resonance.Jobs.Atlas;
using Resonance.Jobs.Atlas.DataPopulation;
using Resonance.Jobs.Atlas.Contracts;
using Resonance.Core.Helpers.AwsHelpers;
using Amazon.CloudWatch;

namespace Resonance.Jobs.Amp
{
    public static class JobManager
    {
        private static bool monitorRunning = false;
        private static Dictionary<string, JobBase> jobs = new Dictionary<string, JobBase>();
        private static Dictionary<string, BackgroundWorker> jobQueue = new Dictionary<string, BackgroundWorker>();

        public static void Initialize()
        {
            PopulateJobs();

            var monitor = new BackgroundWorker();
            monitor.DoWork += MonitorJobs;
            monitor.RunWorkerAsync();
        }

        private static void PopulateJobs()
        {
            Log.Info($@"Populating Jobs");
            var ldapGroupSyncer = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "ldap-group-syncer");
            if (ldapGroupSyncer != null)
            {
                try
                {
                    Log.Info($@"Adding ldap-group-syncer job");
                    jobs.Add
                    (
                        "ldap-group-syncer",
                        new LdapGroupSyncer(new JobConfiguration()
                        {
                            IsActive = ldapGroupSyncer.IsActive,
                            IsRunning = false,
                            LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{ldapGroupSyncer.LogPath}",
                            JobName = ldapGroupSyncer.JobName,
                            StatsDName = ldapGroupSyncer.StatsDName,
                            NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                            S3ConfigurationBucket = $@"{ldapGroupSyncer.S3ConfigurationBucket}",
                            S3ConfigurationPath = $@"{ldapGroupSyncer.S3ConfigurationPath}ldap-group-syncer/{Constants.ApplicationVersion}.json.gz"
                        })
                    );
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }

            var channelRevenueSummary = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "channel-revenue-summary");
            if (channelRevenueSummary != null)
            {
                try
                {
                    Log.Info($@"Adding channel-revenue-summary job");
                    jobs.Add
                    (
                        "channel-revenue-summary",
                        new ChannelRevenueSummary(new JobConfiguration()
                        {
                            IsActive = channelRevenueSummary.IsActive,
                            IsRunning = false,
                            LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{channelRevenueSummary.LogPath}",
                            JobName = channelRevenueSummary.JobName,
                            StatsDName = channelRevenueSummary.StatsDName,
                            NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                            S3ConfigurationBucket = $@"{channelRevenueSummary.S3ConfigurationBucket}",
                            S3ConfigurationPath = $@"{channelRevenueSummary.S3ConfigurationPath}channel-revenue-summary/{Constants.ApplicationVersion}.json.gz"
                        })
                    );
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }

            var twitchGames = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "atlas-twitch-games");
            if (twitchGames != null)
            {
                try
                {
                    Log.Info($@"Adding atlas-twitch-games job");
                    jobs.Add
                    (
                        "atlas-twitch-games",
                        new GameSyncJob(new JobConfiguration()
                        {
                            IsActive = twitchGames.IsActive,
                            IsRunning = false,
                            LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{twitchGames.LogPath}",
                            JobName = twitchGames.JobName,
                            StatsDName = twitchGames.StatsDName,
                            NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                            S3ConfigurationBucket = $@"{twitchGames.S3ConfigurationBucket}",
                            S3ConfigurationPath = $@"{twitchGames.S3ConfigurationPath}atlas-twitch-games/{Constants.ApplicationVersion}.json.gz"
                        })
                    );
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }

            /*
            var backfill = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "atlas-backfill");
            if (backfill != null)
            {
                try
                {
                    Log.Info($@"Adding atlas-backfill job");
                    jobs.Add
                    (
                        "atlas-backfill",
                        new AtlasBackfill(new JobConfiguration()
                        {
                            IsActive = backfill.IsActive,
                            IsRunning = false,
                            LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{backfill.LogPath}",
                            JobName = backfill.JobName,
                            StatsDName = backfill.StatsDName,
                            NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                            S3ConfigurationBucket = $@"{backfill.S3ConfigurationBucket}",
                            S3ConfigurationPath = $@"{backfill.S3ConfigurationPath}atlas-backfill/{Constants.ApplicationVersion}.json.gz"
                        })
                    );
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }
            */

            var contractAutoExtenderJob = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == "contract-auto-extender");
            if (contractAutoExtenderJob != null)
            {
                try
                {
                    Log.Info($@"contract-auto-extender");
                    jobs.Add
                    (
                        "contract-auto-extender",
                        new ContractAutoExtenderJob(new JobConfiguration()
                        {
                            IsActive = contractAutoExtenderJob.IsActive,
                            IsRunning = false,
                            LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{contractAutoExtenderJob.LogPath}",
                            JobName = contractAutoExtenderJob.JobName,
                            StatsDName = contractAutoExtenderJob.StatsDName,
                            NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                            S3ConfigurationBucket = $@"{contractAutoExtenderJob.S3ConfigurationBucket}",
                            S3ConfigurationPath = $@"{contractAutoExtenderJob.S3ConfigurationPath}contract-auto-extender/{Constants.ApplicationVersion}.json.gz"
                        })
                    );
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }

            var activityLogJobname = "activity-log-import";
            var activityLogImport = AppConfig.Data.Jobs.FirstOrDefault(x => x.JobName == activityLogJobname);
            if (activityLogImport != null)
            {
                try
                {
                    Log.Info($@"Adding {activityLogJobname} job");
                    jobs.Add
                    (
                        activityLogJobname,
                        new ActivityLogImporter(new JobConfiguration()
                        {
                            IsActive = activityLogImport.IsActive,
                            IsRunning = false,
                            LogPath = $"{AppConfig.Data.Application.BaseCloudwatchLogPath}{activityLogImport.LogPath}",
                            JobName = activityLogImport.JobName,
                            StatsDName = activityLogImport.StatsDName,
                            NextRunTime = DateTime.UtcNow.AddMinutes(-1),
                            S3ConfigurationBucket = $@"{activityLogImport.S3ConfigurationBucket}",
                            S3ConfigurationPath = $@"{activityLogImport.S3ConfigurationPath}{activityLogJobname}/{Constants.ApplicationVersion}.json.gz"
                        })
                    );
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }
        }

        private static void MonitorJobs(object sender, DoWorkEventArgs e)
        {
            if (monitorRunning)
            {
                return;
            }
            monitorRunning = true;

            while (!CommandManager.quit)
            {
                Parallel.ForEach(jobs, new ParallelOptions() { MaxDegreeOfParallelism = 4 }, job =>
                {
                    try
                    {
                        if (job.Value.Config.IsActive && !job.Value.Config.IsRunning && (job.Value.Config.NextRunTime == null || (job.Value.Config.NextRunTime != null && job.Value.Config.NextRunTime <= DateTime.UtcNow)))
                        {
                            Log.Info($@"Running {job.Key}");
                            CloudwatchHelper.EnqueueMetricRequest($"{job.Value.Config.JobName}_start", 1, null, StandardUnit.Count);
                            StatsDHelper.Counter(measurement: $"jobstatus", measurementType: "start", val: 1, location: job.Key);
                            job.Value.Run();
                            CloudwatchHelper.EnqueueMetricRequest($"{job.Value.Config.JobName}_stop", 1, null, StandardUnit.Count);
                            StatsDHelper.Counter(measurement: $"jobstatus", measurementType: "stop", val: 1, location: job.Key);

                        }
                    }
                    catch (Exception ex)
                    {
                        Log.Error(ex);
                    }
                });
                Thread.Sleep(5000);
            }
            Log.Info($@"Quit command detected by the job manager. Waiting for all jobs to complete.");
            while (jobs.Any(x => x.Value.Config.IsRunning))
            {
                Thread.Sleep(5000);
            }
            Log.Info($@"All jobs complete. Exiting.");
            Program.GracefulQuitHandle.Set();
        }

        private static void ActivateJob(ref JobBase job)
        {
            if (!job.Config.IsActive)
            {
                Log.Info($@"{job.Config.JobName} is not active. Skipping activation process.");
                return;
            }
        }
    }
}