﻿using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models;
using System;
using Resonance.Core.Models.ConfigurationModels.Jobs;
using Resonance.Core.Services.LdapService;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Models.ApiModels.TwitchModels;
using System.Collections.Generic;
using Resonance.Core;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using System.Linq;
using Resonance.Core.Helpers.StringHelpers;

namespace Resonance.Jobs.Atlas.Ldap
{
    public class LdapGroupSyncer : JobBase, IJob<long>
    {
        private static LdapService ldapService = new LdapService();

        public LdapGroupSyncer(JobConfiguration _config)
        {
            try
            {
                Config = _config;
                Log.Info($@"LdapGroupSyncer: configured. IsActive: {Config.IsActive}");
            }
            catch (Exception)
            {

            }
        }

        public override void Run()
        {
            try
            {
                this.Config.IsRunning = true;

                Log.Info("LdapGroupSyncer: Starting");

                var admins = ldapService.GetUsersInGroup("atlas-admins", null);
                var accountManagers = ldapService.GetUsersInGroup("atlas-account-managers", null);

                Log.Info($@"Populating {accountManagers.Count()} Account Managers");
                PopulateAccountManagers(accountManagers, false);
                // Admins are just account managers in the system, the permissions are validated by the users groups through resonance auth
                Log.Info($@"Populating {admins.Count()} Admins");
                PopulateAccountManagers(admins, true);
                Log.Info($@"Completed LDAP population");
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                Log.Info("LdapGroupSyncer: Done");
                this.Config.IsRunning = false;
                this.Config.NextRunTime = DateTime.UtcNow.Add(TimeSpan.FromHours(4));
            }
        }

        private void PopulateAccountManagers(IEnumerable<UserAutocompleteEntry> users, bool isAdmin)
        {
            try
            {
                var ldapUsers = new List<AtlasAccountManager>();
                foreach (var user in users)
                {
                    string apUserName = "";
                    string managerName = user?.Manager;
                    if (!string.IsNullOrWhiteSpace(user?.Manager ?? ""))
                    {
                        var manager = ldapService.SearchUsers(user.Manager, context: null).FirstOrDefault();
                        apUserName = manager?.UserName;
                        managerName = !string.IsNullOrWhiteSpace(manager?.FullName)
                            ? manager?.FullName
                            : managerName;
                    }

                    string[] name = user == null
                       ? new string[2] { "", "" }
                       : user.FullName.Split(' ');
                    var item = new AtlasAccountManager()
                    {
                        AccountManagerLdapName = user?.UserName ?? "",
                        AccountManagerFirstName = name[0].Trim(),
                        AccountManagerLastName = name[1].Trim(),
                        AccountManagerEmail = user?.Mail ?? "",
                        ApprovingManagerAmazonID = apUserName ?? "",
                        ApprovingManagerName = managerName ?? "",
                        IsActive = true,
                        IsAdmin = isAdmin,
                        Hash = HashHelper.SimpleRandomLetterSequence.RandomHash()
                    };
                    ldapUsers.Add(item);
                }
                Log.Info($@"Found {ldapUsers.Count()} Account Manager items");

                if (ldapUsers.Count > 0)
                {
                    using (var conn = DBManagerMysql.GetConnection(true))
                    {
                        using (var command = conn.GetCommand())
                        {
                            command.CommandText =
                            $@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_account_manager
                                set 
                                    account_manager_first_name = @amFirstName, 
                                    account_manager_last_name = @amLastName, 
                                    account_manager_email = @amEmail, 
                                    approving_manager_amazon_id = @apID, 
                                    approving_manager_name = @apName, 
                                    is_active = @isActive, 
                                    created_time = @createdTime, 
                                    created_user = @createdUser, 
                                    hash = @hash,
                                    is_admin = @isAdmin
                                    where
                                        account_manager_ldap_name = @amLdapName
                                ;


                                insert into {Constants.DatabaseSchema}microservice_twitch_atlas_account_manager (account_manager_ldap_name, account_manager_first_name, account_manager_last_name, account_manager_email, approving_manager_amazon_id, approving_manager_name, is_active, created_time, created_user, hash, is_admin)
                                select @amLdapName, @amFirstName, @amLastName, @amEmail, @apID, @apName, @isActive, @createdTime, @createdUser,@hash, @isAdmin
                                where not exists
                                (
                                    select 1 
                                    from {Constants.DatabaseSchema}microservice_twitch_atlas_account_manager
                                    where
                                        account_manager_ldap_name = @amLdapName
                                )
                                ;
                            ";
                            foreach (var user in ldapUsers.ToList())
                            {
                                try
                                {
                                    command.Parameters.Clear();
                                    command.Parameters.AddWithValue("@amLdapName", user.AccountManagerLdapName);
                                    command.Parameters.AddWithValue("@amFirstName", user.AccountManagerFirstName);
                                    command.Parameters.AddWithValue("@amLastName", user.AccountManagerLastName);
                                    command.Parameters.AddWithValue("@amEmail", user.AccountManagerEmail);
                                    command.Parameters.AddWithValue("@apID", user.ApprovingManagerAmazonID);
                                    command.Parameters.AddWithValue("@apName", user.ApprovingManagerName);
                                    command.Parameters.AddWithValue("@isActive", true);
                                    command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                                    command.Parameters.AddWithValue("@createdUser", "BulkImport");
                                    command.Parameters.AddWithValue("@hash", user.Hash);
                                    command.Parameters.AddWithValue("@isAdmin", user.IsAdmin);
                                    command.ExecuteNonQueryWithMeasurements("atlas_am");
                                    command.Parameters.Clear();
                                }
                                catch (Exception ex)
                                {
                                    Log.Error(ex);
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }
    }
}
