﻿using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using Resonance.Core.Extensions;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.AtlasModels;
using Resonance.Core.Models.ApiModels.RequestModels;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using Resonance.Core.Models.DatabaseModels.RequestModels;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Models.SlaModels;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;

namespace Resonance.Metrics.SLA.Atlas
{
    /// <summary>
    /// To run all items in this Sla Measurement Tool you need:
    /// - Atlas
    /// - Amp
    /// - Twitch VPN
    /// - CrsData VPN
    /// </summary>
    public static class EventController
    {
        private static Random rand = new Random();
        private static int defaultIterationCount = 5;
        private static int defaultForceSleep = 2;
        private static float slabuffer = 0.10f;
        /// <summary>
        /// The content of this file should contain the ApiToken you have stored in explorer permissions-admin/edit endpoint.
        /// </summary>
        private static string DefaultAuthKey { get; set; } = File.ReadAllText(@"C:\Secrets\Resonance OAuth\SlaToken.dat");
        /// <summary>
        /// Valid Options: Development | Staging | Production
        /// </summary>
        private static string DefaultEnvironment { get; set; } = "Production";
        /// <summary>
        /// The base url used that hosts the Resonance service
        /// </summary>
        private static string baseUrl { get; set; } = DefaultEnvironment == "Development"
            ? "http://localhost:8087/"
            : DefaultEnvironment == "Staging"
                ? "http://172.29.7.169/"
                : DefaultEnvironment == "Production"
                    ? "https://service.atlas.xarth.tv/"
                    : "<UNKNOWN>";

        private static Dictionary<string, SlaRequestModel> endpoints = new Dictionary<string, SlaRequestModel>()
        {
            {
                "event/get-product-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        EventType = Core.Constants.AtlasInternalEventType.Product,
                        Columns = AtlasListingDefaults.ProductDefault(),
                        QueryFilters = new QueryFilter[0] { }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-product",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasProduct()
                    {
                        ProductName = $"Test Product {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                        ProductDescription = "Test Description",
                        Topic = "Test Topic",
                        SubTopic = "Test Sub Topic",
                        Format = "Test Format",
                        SubFormat = "Test sub topic",
                        ProductOwner = Environment.MachineName,
                        CreatedTime = DateTime.UtcNow,
                        CreatedUser = Environment.MachineName,
                        IsActive = true
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-season-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        EventType = Core.Constants.AtlasInternalEventType.Season,
                        Columns = AtlasListingDefaults.SeasonDefault(),
                        QueryFilters = new QueryFilter[0] { }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/user-auto-complete",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new RequestUserSearch()
                    {
                        LoginNameSearch = "ryu"
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-managed-channels",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasSeason()
                    {
                        ProductID = 1,
                        SeasonName = $"Test Season {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                        PrizePool = rand.Next(100, 100000),
                        PledgedHoursBroadcast = (float)rand.Next(0, 1000) / (float)rand.Next(1, 1000),
                        StartTime = DateTime.UtcNow,
                        EndTime = DateTime.UtcNow.Date.AddDays(1),
                        CreatedTime = DateTime.UtcNow,
                        CreatedUser = Environment.MachineName,
                        IsActive = true,
                        PreviousSeasonID = 42,
                        PreviousSeasonName = $"Test Previous Season {DateTime.UtcNow.ToString("yyyy-MM-dd")}"
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-season",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasSeason()
                    {
                        ProductID = 1,
                        SeasonName = $"Test Season {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                        PrizePool = rand.Next(100, 100000),
                        PledgedHoursBroadcast = (float)rand.Next(0, 1000) / (float)rand.Next(1, 1000),
                        StartTime = DateTime.UtcNow,
                        EndTime = DateTime.UtcNow.Date.AddDays(1),
                        CreatedTime = DateTime.UtcNow,
                        CreatedUser = Environment.MachineName,
                        IsActive = true,
                        PreviousSeasonID = 42,
                        PreviousSeasonName = $"Test Previous Season {DateTime.UtcNow.ToString("yyyy-MM-dd")}"
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-event-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "event_name",
                            Value = "a"
                        }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-event",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasEvent()
                    {
                        EstimatedAverageCCV = rand.Next(1000, 100000),
                        EventName = $"Test Event {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                        EventType = "Demo",
                        PledgedHoursBroadcast = (float)rand.Next(0, 1000) / (float)rand.Next(1, 1000),
                        ProductID = rand.Next(0, 1000),
                        SeasonID = rand.Next(0, 1000),
                        StartTime = DateTime.UtcNow,
                        EndTime = DateTime.UtcNow.Date.AddDays(1),
                        CreatedTime = DateTime.UtcNow,
                        CreatedUser = Environment.MachineName,
                        Topic = "test topic",
                        SubTopic = "test sub topic",
                        Format = "test format",
                        SubFormat = "test sub format",
                        IsActive = true,
                        PremiumContentCreatorID = rand.Next(100000, 500000),
                        CostreamingSettings = "Test costream settings",
                        Distribution = true,
                        GameName = "Example game",
                        TwitchInvolvement = "Very involved"
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-event-with-streams",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasEventWithStreamList()
                    {
                        Event = new AtlasEvent()
                        {
                            EstimatedAverageCCV = rand.Next(1000, 100000),
                            EventName = $"Test Event {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                            EventType = "Demo",
                            PledgedHoursBroadcast = (float)rand.Next(0, 1000) / (float)rand.Next(1, 1000),
                            ProductID = rand.Next(0, 1000),
                            SeasonID = rand.Next(0, 1000),
                            StartTime = DateTime.UtcNow,
                            EndTime = DateTime.UtcNow.Date.AddDays(1),
                            CreatedTime = DateTime.UtcNow,
                            CreatedUser = Environment.MachineName,
                            Topic = "test topic",
                            SubTopic = "test sub topic",
                            Format = "test format",
                            SubFormat = "test sub format",
                            IsActive = true,
                            PremiumContentCreatorID = rand.Next(100000, 500000),
                            CostreamingSettings = "Test costream settings",
                            Distribution = true,
                            GameName = "Example game",
                            TwitchInvolvement = "Very involved"
                        },
                        Streams = new List<EventStreamEntry>()
                        {
                            new EventStreamEntry()
                            {
                                ChannelID = 1,
                                Channel = $"Test Channel 1 {DateTime.UtcNow.ToString("yyyy-MM-dd")}",
                                Type = "Test Type"
                            },
                            new EventStreamEntry()
                            {
                                ChannelID = 2,
                                Channel = $"Test Channel 2 {DateTime.UtcNow.ToString("yyyy-MM-dd")}",
                                Type = "Test Type"
                            },
                            new EventStreamEntry()
                            {
                                ChannelID = 3,
                                Channel = $"Test Channel 3 {DateTime.UtcNow.ToString("yyyy-MM-dd")}",
                                Type = "Test Type"
                            }
                        }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-season-event-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "season",
                            Value = "2020"
                        }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-channel-event-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        Columns = new string[1] { "*" },
                        QueryFilters = new QueryFilter[0] {}
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-topic-event-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        Columns = new string[1] { "*" },
                        QueryFilters = new QueryFilter[0] { }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-format-event-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        Columns = new string[1] { "*" },
                        QueryFilters = new QueryFilter[0]{ }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-category-event-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        Columns = new string[1] { "*" },
                        QueryFilters = new QueryFilter[0] {}
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-stream-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        EventType = Core.Constants.AtlasInternalEventType.Stream,
                        Columns = AtlasListingDefaults.SeasonDefault(),
                        QueryFilters = new QueryFilter[1]
                        {
                            new QueryFilter()
                            {
                                FilterType = Core.Constants.FilterType.Contains,
                                Key = "stream_login",
                                Value = "no"
                            }
                        }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-stream",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasStream()
                    {
                        ChannelID = rand.Next(100000, 500000),
                        ChannelType = "partner",
                        EventID = rand.Next(0, 1000),
                        GameName = $"Test Game {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                        StreamLogin = "test login",
                        StartTime = DateTime.UtcNow,
                        EndTime = DateTime.UtcNow.Date.AddDays(1),
                        CreatedTime = DateTime.UtcNow,
                        CreatedUser = Environment.MachineName,
                        IsActive = true
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-event-types",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = null,
                    Iterations = defaultIterationCount,
                    MethodType = "GET"
                }
            },
            {
                "event/get-topics-formats",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = null,
                    Iterations = defaultIterationCount,
                    MethodType = "GET"
                }
            },
            {
                "event/get-premium-content-creator-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        EventType = Core.Constants.AtlasInternalEventType.PremiumContentCreator,
                        Columns = AtlasListingDefaults.PremiumContentCreatorDefault(),
                        QueryFilters = new QueryFilter[1]
                        {
                            new QueryFilter()
                            {
                                FilterType = Core.Constants.FilterType.Contains,
                                Key = "is_active",
                                Value = true
                            }
                        }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-premium-content-creator",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new PremiumContentCreator()
                    {
                        PremiumContentCreatorName = $"Test {DateTime.UtcNow.ToString("yyyy-MM-dd")}",
                        PremiumContentCreatorType = "Test",
                        IsActive = true
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-premium-content-creator-with-channels",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new CreatorWithStreamList()
                    {
                        AccountManagerIDs = new List<int>()
                        {
                            1, 2, 3
                        },
                        Channels = new List<CreatorChannel>()
                        {
                            new CreatorChannel() { ChannelID = 1, ChannelOwnership = "Ownership", ChannelVertical = "Vertical" }
                        },
                        Creator = new PremiumContentCreator()
                        {
                            AccountManagerAssignmentMapID = 1,
                            IsActive = true,
                            PremiumContentCreatorID = 1,
                            PremiumContentCreatorName = "Test Creator",
                            PremiumContentCreatorType = "Test"
                        }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-account-manager-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        EventType = Core.Constants.AtlasInternalEventType.AccountManager,
                        Columns = AtlasListingDefaults.AccountManagerDefault(),
                        QueryFilters = new QueryFilter[1]
                        {
                            new QueryFilter()
                            {
                                FilterType = Core.Constants.FilterType.Contains,
                                Key = "account_manager_ldap_name",
                                Value = "r"
                            }
                        }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-account-manager",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasAccountManager()
                    {
                        AccountManagerLdapName = $"test_{DateTime.UtcNow.ToString("yyyy-MM-dd_hh-mm-ss")}",
                        AccountManagerFirstName = "Test",
                        AccountManagerLastName = "User",
                        ApprovingManagerAmazonID = "nsinkule",
                        AccountManagerEmail = "nstewart@twitch.tv",
                        ApprovingManagerName = "Nic Sinkule",
                        IsActive = true
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-pcc-am-map-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        EventType = Core.Constants.AtlasInternalEventType.PccToAmMap,
                        Columns = AtlasListingDefaults.PccToAmMapDefault(),
                        QueryFilters = new QueryFilter[1]
                        {
                            new QueryFilter()
                            {
                                FilterType = Core.Constants.FilterType.Contains,
                                Key = "is_active",
                                Value = true
                            }
                        }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-pcc-am-map",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new PremiumContentCreatorToAccountManagerMap()
                    {
                        AccountManagerID = 1,
                        PremiumContentCreatorID = 1,
                        IsActive = true
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-pcc-channel-map-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        EventType = Core.Constants.AtlasInternalEventType.PccToChannelMap,
                        Columns = AtlasListingDefaults.PccToChannelMapDefault(),
                        QueryFilters = new QueryFilter[0] { }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-pcc-channel-map",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new PremiumContentCreatorToChannelMap()
                    {
                        ChannelID = 1,
                        PremiumContentCreatorID = 1,
                        IsActive = true
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-contract-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        EventType = Core.Constants.AtlasInternalEventType.Contract,
                        Columns = AtlasListingDefaults.ContractDefault(),
                        QueryFilters = new QueryFilter[0] { }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-contract-list-with-channels",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        EventType = Core.Constants.AtlasInternalEventType.Contract,
                        Columns = new string[1] { "*" },
                        QueryFilters = new QueryFilter[0] { }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-contract",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasContract()
                    {
                        ContractName = $"Test {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                        AdDensityPerHour = 0.15f,
                        AdPrerollsEnabled = true,
                        AdRevenueContractType = "cpm",
                        AdRevenueShareAmountTier1 = 75.0f,
                        AdRevenueShareAmountTier2 = 50.0f,
                        AdRevenueShareAmountTier3 = 25.0f,
                        AdRevenueShareTypeTier1 = "percent",
                        AdRevenueShareTypeTier2 = "percent",
                        AdRevenueShareTypeTier3 = "percent",
                        BitsRevenueSharePercent = 75.0f,
                        ContractType = "Test",
                        CreatedTime = DateTime.UtcNow,
                        CreatedUser = "Test",
                        EndDate = DateTime.UtcNow.AddDays(5).Date,
                        ExclusivityEnabled = true,
                        PremiumContentCreatorID = 1,
                        RevenueShareEnabled = true,
                        SponsorshipEnabled = true,
                        StartDate = DateTime.UtcNow.Date,
                        SubsRevenueSharePercent = 100.0f,
                        IsActive = true
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/edit-contract-with-channels",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasContractWithChannelList()
                    {
                        AccountManagerIDs = new List<int>()
                        {
                            1
                        },
                        Channels = new List<AtlasContractChannelEntry>()
                        {
                            new AtlasContractChannelEntry()
                            {
                                Channel = $"Test Channel {DateTime.UtcNow.ToString("yyyy-MM-dd")}",
                                ChannelID = 1,
                                CustomStartDate = DateTime.UtcNow.Date.AddDays(1),
                                CustomEndDate = DateTime.UtcNow.AddDays(2)
                            }
                        },
                        Contract = new AtlasContract()
                        {
                            AdDensityPerHour = 0.5f,
                            AdPrerollsEnabled = true,
                            AdRevenueContractType = "TEST",
                            AdRevenueShareAmountTier1 = 1.55f,
                            AdRevenueShareAmountTier2 = 1.0f,
                            AdRevenueShareAmountTier3 = 1.0f,
                            AdRevenueShareTypeTier1 = "cpm",
                            AdRevenueShareTypeTier2 = "percent",
                            AdRevenueShareTypeTier3 = "percent",
                            BitsRevenueSharePercent = 1.0f,
                            ContractName = $"Test Contract {DateTime.UtcNow.ToString("yyyy-MM-dd")}",
                            ContractType = "Test",
                            ExclusivityEnabled = true,
                            IsActive = true,
                            PremiumContentCreatorID = 1,
                            RevenueShareEnabled = true,
                            SponsorshipEnabled = true,
                            SubsRevenueSharePercent = 1.0f,
                            StartDate = DateTime.UtcNow.AddDays(-1),
                            EndDate = DateTime.UtcNow.AddDays(3),
                            ContractID = 1
                        }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            /* Commented Out 2020-04-17 by Nathan -- Nic states we do not use on front end. Remove after 2020-05-17 if still not needed
            {
                "event/get-contract-channel-list",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ListingFilter()
                    {
                        Limit = 10,
                        Page = 0,
                        EventType = Core.Constants.AtlasInternalEventType.ContractChannel,
                        Columns = new string[1] { "*" },
                        QueryFilters = new QueryFilter[0] { }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            */
            {
                "event/edit-contract-channel",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasContractChannel()
                    {
                        ChannelID = 1,
                        ContractID = 1,
                        CustomStartDate = DateTime.UtcNow.Date.AddDays(-1),
                        CustomEndDate = DateTime.UtcNow.Date.AddDays(1 + rand.Next(0, 3)),
                        IsActive = true
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-active-calendar-events",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = null,
                    Iterations = defaultIterationCount,
                    MethodType = "GET"
                }
            },
            {
                "event/edit-contract-account-manager-map",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AtlasContractAccountManagerMap()
                    {
                        AccountManagerID = 1,
                        ContractID = 1,
                        CreatedTime = DateTime.UtcNow,
                        CreatedUser = Environment.MachineName,
                        IsActive = true
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-aggregate-product-stats",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new AggregatePccStatsRequest()
                    {
                        PccIDs = new int[3] { 1,2,3 }
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            },
            {
                "event/get-channel-meta-data",
                new SlaRequestModel()
                {
                    AuthRequired = true,
                    AuthKey = DefaultAuthKey,
                    Data = new ChannelMetaDataRequest()
                    {
                        EventID = 1
                    },
                    Iterations = defaultIterationCount,
                    MethodType = "POST"
                }
            }
        };

        public static void Run()
        {
            if(DefaultEnvironment == "Production")
            {
                throw new ArgumentOutOfRangeException("Event controller measurement is disabled for production use.");
            }
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            try
            {
                if (baseUrl == "<UNKNOWN>")
                {
                    throw new NotImplementedException("Unknown environment");
                }

                if (DefaultEnvironment == "Development")
                {
                    Console.WriteLine("Napping for 60s while endpoints load");
                    Thread.Sleep(60000);
                }
                var expectedWebsiteMaxSla = 2.0f;
                Console.WriteLine($@"SLA Tool estimated to run for {(((expectedWebsiteMaxSla * endpoints.Sum(x => x.Value.Iterations > 1 ? (x.Value.Iterations * defaultForceSleep) : x.Value.Iterations))) / 60.0f).ToString("N2")} minutes");

                var elapsedMsList = new Dictionary<string, List<SlaResponseModel>>();

                // Process endpoints
                foreach (var endpoint in endpoints)
                {
                    Console.WriteLine($@"Executing {endpoint.Key}");
                    try
                    {
                        elapsedMsList.Add(endpoint.Key, new List<SlaResponseModel>());
                        for (var i = 0; i < endpoint.Value.Iterations; i++)
                        {
                            Console.WriteLine($@"Running iteration {i + 1}");
                            dynamic data = endpoint.Value.Data;
                            WebRequestResponse response = endpoint.Value.MethodType == "POST"
                                ? WebRequestHelper.PostData<dynamic>($@"{baseUrl}{endpoint.Key}", bearer: endpoint.Value.AuthKey, data: ref data, timeout: 30, isToken: true, needsAuth: true)
                                : WebRequestHelper.GetData($@"{baseUrl}{endpoint.Key}", bearer: endpoint.Value.AuthKey, timeout: 30, isToken: true);
                            if (response != null && response.IsSuccess)
                            {
                                try
                                {
                                    // We use empty model here because all we care about is top level metrics, it doesn't matter if it's an ApiResponse or ApiListResponse
                                    dynamic result;
                                    try
                                    {
                                        result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                                    }
                                    catch(Exception)
                                    {
                                        try
                                        {
                                            result = JsonConvert.DeserializeObject<ApiListResponse<EmptyModel>>(response.Data);
                                        }
                                        catch (Exception ex)
                                        {
                                            result = null;
                                        }
                                    }
                                    if (result.Metrics != null)
                                    {
                                        elapsedMsList[endpoint.Key].Add(new SlaResponseModel()
                                        {
                                            Iteration = i + 1,
                                            ElapsedMS = result.Metrics.ElapsedMS
                                        });
                                    }
                                }
                                catch (Exception ex)
                                {
                                    Console.WriteLine(ex.ToString());
                                    Debug.Print(ex.ToString());
                                }
                            }
                            if(endpoint.Value.Iterations  > 1)
                            {
                                Thread.Sleep(defaultForceSleep * 1000);
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine(ex.ToString());
                        Debug.Print(ex.ToString());
                    }
                }

                // Upload results to S3
                try
                {
                    Console.WriteLine($@"Uploading raw results to S3");
                    S3Helper.WriteStringToS3(JsonConvert.SerializeObject(elapsedMsList, Formatting.Indented), "resonance-metrics", $"{DefaultEnvironment}/raw-metrics/{DateTime.UtcNow.ToString("yyyy-MM-dd")}.json.gz", compress: true);
                }
                catch (Exception ex)
                {
                    Console.WriteLine(ex.ToString());
                    Debug.Print(ex.ToString());
                }

                var summaryList = new List<SlaSummaryModel>();
                // Process summary
                foreach (var result in elapsedMsList)
                {
                    try
                    {
                        summaryList.Add(new SlaSummaryModel()
                        {
                            Endpoint = result.Key,
                            Iterations = result.Value.Count > 0 ? (int?)result.Value.Count : null,
                            AverageElapsedMS = result.Value.Count > 0 ? (double?)result.Value.Average(x => (double)x.ElapsedMS) : null,
                            MinElapsedMS = result.Value.Count > 0 ? (long?)result.Value.Min(x => x.ElapsedMS) : null,
                            MaxElapsedMs = result.Value.Count > 0 ? (long?)result.Value.Max(x => x.ElapsedMS) : null,
                            MedianElapsedMs = result.Value.Count > 0 ? (long?)result.Value.Select(x => x.ElapsedMS).ToArray().Median() : null
                        });
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine(ex.ToString());
                        Debug.Print(ex.ToString());
                    }
                }

                // Upload summary to S3
                try
                {
                    Console.WriteLine($@"Uploading summary results to s3");
                    S3Helper.WriteStringToS3(JsonConvert.SerializeObject(summaryList.OrderBy(x => x.Endpoint).ToArray(), Formatting.Indented), "resonance-metrics", $"{DefaultEnvironment}/summary-metrics/{DateTime.UtcNow.ToString("yyyy-MM-dd")}.json.gz", compress: true);
                }
                catch (Exception ex)
                {
                    Console.WriteLine(ex.ToString());
                    Debug.Print(ex.ToString());
                }
            }
            finally
            {
                stopwatch.Stop();
                Console.WriteLine($@"SLA monitoring tool took {stopwatch.ElapsedMilliseconds}ms ({((float)stopwatch.ElapsedMilliseconds / 1000.0f / 60.0f).ToString("N2")} minutes) to execute {endpoints.Keys.Count} endpoints and {endpoints.Values.Sum(x => x.Iterations)} iterations.");
            }
        }
    }
}
