﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc.Infrastructure;
using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.AuthHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.AtlasModels;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Models.ServiceModels.ActivityLoggerService;
using Resonance.Core.Services.ActivityLoggerService;

using Resonance.Core.Services.ColumnFilterService;
using Resonance.Microservices.Methods;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;

namespace Resonance.Microservices.ListingHelpers
{
    public static class ListingHelper
    {
        private static AtlasEndpointAuthHelper atlasOwnershipHelper = new AtlasEndpointAuthHelper();

        private static AtlasMethods atlasMethod;

        static ListingHelper()
        {
            atlasMethod = new AtlasMethods();
            atlasMethod.Initialize();
        }

        public static ApiResponse<AtlasListing> DefaultResponse()
        {
            return new ApiResponse<AtlasListing>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
        }

        public static int PostProcessAtlasEventListing(ref AtlasListing listing, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            // Listing being null is an error
            if(listing == null)
            {
                return 400;
            }
            var statuscode = 200;
            // Listing being empty just means we have no items;
            if (listing.Items.Count() == 0)
            {
                return statuscode;
            }
            AtlasMethods atlasMethods = new AtlasMethods();
            Dictionary<int, List<dynamic>> eventChannels = new Dictionary<int, List<dynamic>>();
            Dictionary<long, List<dynamic>> channelLookup = new Dictionary<long, List<dynamic>>();
            Dictionary<int, dynamic> eventLookup = new Dictionary<int, dynamic>();
            foreach (dynamic eventEntry in listing?.Items)
            {
                int eventID = eventEntry.event_id;
                eventLookup[eventID] = eventEntry;
            }
            IEnumerable<int> eventIDs = eventLookup.Keys;
            ListingFilter existingEventChannelsFilter = new ListingFilter
            {
                Limit = 4000,
                Columns = new[] { "event_id", "channel_id", "channel_type" },
                EventType = Constants.AtlasInternalEventType.Stream,
                QueryFilters = new[]
            {
                new QueryFilter() { FilterType = Constants.FilterType.In, Key = "event_id", ValueArray = eventIDs.Cast<dynamic>().ToArray()},
            }
            };

            var existingChannelInfo = atlasMethods.GetListingByFilter(ref existingEventChannelsFilter, metrics, context);
            statuscode = existingChannelInfo.Item1;
            if (existingChannelInfo != null && existingChannelInfo.Item2.Items != null)
            {
                foreach (dynamic channelInfo in existingChannelInfo.Item2.Items)
                {
                    long channelID = channelInfo.channel_id;
                    int eventID = channelInfo.event_id;
                    if (!channelLookup.ContainsKey(channelID))
                    {
                        channelLookup.Add(channelID, new List<dynamic>());
                    }
                    channelLookup[channelID].Add(channelInfo);
                    if (!eventChannels.ContainsKey(eventID))
                    {
                        eventChannels.Add(eventID, new List<dynamic>());
                    }
                    eventChannels[eventID].Add(channelInfo);
                }
            }

            TwitchUsersServiceGetUsersResponse usersServiceResponse = TwitchUsersServiceHelpers.GetUsers(context, channelLookup.Keys);
            if (usersServiceResponse != null)
            {
                foreach (var channel in usersServiceResponse.Results)
                {
                    long channelID = channel.ID;
                    string channelName = channel.Login;
                    string profileImage = channel.ProfileImageUrl.FormatUsersServiceImageUrl();
                    foreach (var lookupChannel in channelLookup[channelID])
                    {
                        lookupChannel.channel_login = channelName;
                        lookupChannel.profile_image = profileImage;
                    }
                }

            }
            foreach (var contractChannel in eventChannels)
            {
                eventLookup[contractChannel.Key].channels = contractChannel.Value;
            }
            listing.Items = eventLookup.Values.ToArray();
            return statuscode;
        }

        public static int PostProcessAtlasContractListing(ref ListingFilter filter, ref ApiResponse<AtlasListing> response, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            if(response.ResponseData.Items == null)
            {
                return 400;
            }
            if (response.ResponseData.Items.Count() == 0)
            {
                return 200;
            }
            var statuscode = 200;
            List<int> pccIDS = new List<int>();
            foreach(var pccFilter in filter.QueryFilters.Where(x => x.Key == "premium_content_creator_id"))
            {
                pccIDS.Add(Convert.ToInt32(pccFilter.Value));
            }
            pccIDS = pccIDS.Distinct().ToList();
            if(pccIDS.Count() == 0)
            {
                return 400;
            }
            AtlasMethods atlasMethods = new AtlasMethods();
            Dictionary<int, List<dynamic>> contractChannels = new Dictionary<int, List<dynamic>>();
            Dictionary<long, List<dynamic>> channelLookup = new Dictionary<long, List<dynamic>>();
            Dictionary<int, List<dynamic>> amLookup = new Dictionary<int, List<dynamic>>();
            Dictionary<int, dynamic> contractLookup = new Dictionary<int, dynamic>();
            foreach(dynamic contract in response.ResponseData.Items)
            {
                int contractID = contract.contract_id;
                contractLookup[contractID] = contract;
            }
            IEnumerable<int> contractIDs = contractLookup.Keys;
            ListingFilter existingContractChannelsFilter = new ListingFilter();
            existingContractChannelsFilter.Limit = 4000;
            existingContractChannelsFilter.Columns = new[] { "contract_id", "channel_id","custom_start_date", "custom_end_date"  };
            existingContractChannelsFilter.EventType = Constants.AtlasInternalEventType.ContractChannel;
            existingContractChannelsFilter.QueryFilters = new[] 
            {
                new QueryFilter() { FilterType = Constants.FilterType.In, Key = "premium_content_creator_id", ValueArray = pccIDS.Cast<dynamic>().ToArray() },
                new QueryFilter() { FilterType = Constants.FilterType.In, Key = "contract_id", ValueArray = contractIDs.Cast<dynamic>().ToArray() }
            };
            var existingChannelInfo = atlasMethods.GetListingByFilter(ref existingContractChannelsFilter, metrics, context);
            statuscode = existingChannelInfo.Item1;
            if (existingChannelInfo != null && existingChannelInfo.Item2.Items != null)
            {
                foreach(dynamic channelInfo in existingChannelInfo.Item2.Items)
                {
                    long channelID = channelInfo.channel_id;
                    int contractID = channelInfo.contract_id;
                    if (!channelLookup.ContainsKey(channelID))
                    {
                        channelLookup.Add(channelID, new List<dynamic>());
                    }
                    channelLookup[channelID].Add(channelInfo);
                    if(!contractChannels.ContainsKey(contractID))
                    {
                        contractChannels.Add(contractID, new List<dynamic>());
                    }
                    contractChannels[contractID].Add(channelInfo);
                }
            }

            TwitchUsersServiceGetUsersResponse usersServiceResponse = TwitchUsersServiceHelpers.GetUsers(context, channelLookup.Keys);
            if(usersServiceResponse != null)
            {
                foreach(var channel in usersServiceResponse.Results)
                {
                    long channelID = channel.ID;
                    string channelName = channel.Login;
                    string profileImage = channel.ProfileImageUrl.FormatUsersServiceImageUrl();
                    
                    foreach (var lookupChannel in channelLookup[channelID])
                    {
                        lookupChannel.channel_login = channelName;
                        lookupChannel.profile_image = profileImage;
                    }
                }

            }
            foreach (var contractChannel in contractChannels)
            {
                contractLookup[contractChannel.Key].channels = contractChannel.Value;
            }

            ListingFilter existingContractAMsFilter = new ListingFilter();
            existingContractAMsFilter.Limit = 4000;
            existingContractAMsFilter.Columns = new[] { "contract_id", "account_manager_id", "account_manager_first_name", "account_manager_last_name" };
            existingContractAMsFilter.EventType = Constants.AtlasInternalEventType.ContractAccountManagerMap;
            existingContractAMsFilter.QueryFilters = new[] 
            {
                new QueryFilter() { FilterType = Constants.FilterType.Exact, Key = "premium_content_creator_id", ValueArray = pccIDS.Cast<dynamic>().ToArray() },
                new QueryFilter() { FilterType = Constants.FilterType.In, Key = "contract_id", ValueArray = contractIDs.Cast<dynamic>().ToArray() }
            };
            var existingAMInfo = atlasMethods.GetListingByFilter(ref existingContractAMsFilter, metrics, context, requireOwnershipVerification: true);
            statuscode = existingAMInfo.Item1;
            if (existingAMInfo != null && existingAMInfo.Item2.Items != null)
            {
                foreach (dynamic amMap in existingAMInfo.Item2.Items)
                {
                    int contractID = amMap.contract_id;
                    if (!amLookup.ContainsKey(contractID))
                    {
                        amLookup.Add(contractID, new List<dynamic>());
                    }
                    amLookup[contractID].Add(amMap);
                }
            }
            foreach (var pccAM in amLookup)
            {
                contractLookup[pccAM.Key].account_managers = pccAM.Value;
            }
            response.ResponseData.Items = contractLookup.Values.ToArray();
            return statuscode;
        }

        public static int PostProcessAtlasPCCListing(ref ApiResponse<AtlasListing> response, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            if(!response.ResponseData.Items.Any())
            {
                return 400;
            }
            var statuscode = 200;
            AtlasMethods atlasMethods = new AtlasMethods();
            Dictionary<int, List<dynamic>> pccChannels = new Dictionary<int, List<dynamic>>();
            Dictionary<long, List<dynamic>> channelLookup = new Dictionary<long, List<dynamic>>();
            Dictionary<int, List<int>> amLookup = new Dictionary<int, List<int>>();
            Dictionary<int, dynamic> pccLookup = new Dictionary<int, dynamic>();
            foreach (dynamic pcc in response.ResponseData.Items)
            {
                int pccID = pcc.premium_content_creator_id;
                pccLookup[pccID] = pcc;
            }
            IEnumerable<int> pccIDs = pccLookup.Keys;
            ListingFilter existingPCCChannelsFilter = new ListingFilter();
            existingPCCChannelsFilter.Limit = 4000;
            existingPCCChannelsFilter.Columns = new[] { "premium_content_creator_id", "channel_id", "channel_ownership", "channel_vertical" };
            existingPCCChannelsFilter.EventType = Constants.AtlasInternalEventType.PccToChannelMap;
            existingPCCChannelsFilter.QueryFilters = new[] 
            {
                new QueryFilter() { FilterType = Constants.FilterType.In, Key = "premium_content_creator_id", ValueArray = pccIDs.Cast<dynamic>().ToArray() }
            };
            var existingChannelInfo = atlasMethods.GetListingByFilter(ref existingPCCChannelsFilter, metrics, context);
            statuscode = existingChannelInfo.Item1;
            if (existingChannelInfo != null && existingChannelInfo.Item2.Items != null)
            {
                foreach (dynamic channelInfo in existingChannelInfo.Item2.Items)
                {
                    long channelID = channelInfo.channel_id;
                    int pccID = channelInfo.premium_content_creator_id;
                    if (!channelLookup.ContainsKey(channelID))
                    {
                        channelLookup.Add(channelID, new List<dynamic>());
                    }
                    channelLookup[channelID].Add(channelInfo);
                    if (!pccChannels.ContainsKey(pccID))
                    {
                        pccChannels.Add(pccID, new List<dynamic>());
                    }
                    pccChannels[pccID].Add(channelInfo);
                }
            }

            TwitchUsersServiceGetUsersResponse usersServiceResponse = TwitchUsersServiceHelpers.GetUsers(context, channelLookup.Keys);
            if (usersServiceResponse != null)
            {
                foreach (var channel in usersServiceResponse.Results)
                {
                    long channelID = channel.ID;
                    string channelName = channel.Login;
                    string profileImage = channel.ProfileImageUrl.FormatUsersServiceImageUrl();
                    foreach (var lookupChannel in channelLookup[channelID])
                    {
                        lookupChannel.channel_login = channelName;
                        lookupChannel.profile_image = profileImage;
                    }
                }

            }
            foreach (var pccChannel in pccChannels)
            {
                pccLookup[pccChannel.Key].channels = pccChannel.Value;
            }
            ListingFilter existingPCCAMsFilter = new ListingFilter();
            existingPCCAMsFilter.Limit = 4000;
            existingPCCAMsFilter.Columns = new[] { "premium_content_creator_id", "account_manager_id" };
            existingPCCAMsFilter.EventType = Constants.AtlasInternalEventType.PccToAmMap;
            existingPCCAMsFilter.QueryFilters = new[] 
            {
                new QueryFilter() { FilterType = Constants.FilterType.In, Key = "premium_content_creator_id", ValueArray = pccIDs.Cast<dynamic>().ToArray() }
            };
            var existingAMInfo = atlasMethods.GetListingByFilter(ref existingPCCAMsFilter, metrics, context);
            statuscode = existingAMInfo.Item1;

            if (existingAMInfo != null && existingAMInfo.Item2.Items != null)
            {
                foreach(dynamic amMap in existingAMInfo.Item2.Items)
                {
                    int pccID = amMap.premium_content_creator_id;
                    int amID = amMap.account_manager_id;
                    if(!amLookup.ContainsKey(pccID))
                    {
                        amLookup.Add(pccID, new List<int>());
                    }
                    amLookup[pccID].Add(amID);
                }
            }
            foreach (var pccAM in amLookup)
            {
                pccLookup[pccAM.Key].am_ids = pccAM.Value;
            }
            response.ResponseData.Items = pccLookup.Values.ToArray();
            return statuscode;
        }

        public static int WireUpAtlasListing(Constants.AtlasInternalEventType eventType, string[] defaultColumns, ref ApiResponse<AtlasListing> response, ref ListingFilter filter, HttpContext context, string actionName, string controllerName, IColumnFilterService _columnFilterService, IActivityLoggerService _activityLoggerService, IActionContextAccessor _actionContextAccessor, ConcurrentBag<ElapsedTimeModel> metrics, bool requireOwnershipVerification = false)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();
            var statuscode = 200;
            var partMeasure = new Stopwatch();
            try
            {
                partMeasure.Start();

                UserAuthData tokenData = (UserAuthData)context.Items[UserAuthDataContext.UserAuthDataKey];
                bool continueProcess = !requireOwnershipVerification; // If not requiring ownership, default to true, otherwise default to false and process ownership to set continue processing flag
                if (requireOwnershipVerification)
                {
                    var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, ref filter);
                    if(verifyOwnershipResult != null)
                    {
                        statuscode = verifyOwnershipResult.StatusCode;
                        if (verifyOwnershipResult.IsVerified)
                        {
                            continueProcess = true;
                        }
                        else
                        {
                            continueProcess = false;
                        }
                    }
                    else
                    {
                        continueProcess = false;
                    }
                }
                if (continueProcess)
                {
                    var validateListing = VerifyGetListing(eventType, defaultColumns, context, _columnFilterService, _activityLoggerService, _actionContextAccessor, ref tokenData, ref response, ref filter, ref actionName, ref controllerName);
                    if (!validateListing)
                    {
                        return 400;
                    }
                    partMeasure.Stop();
                    metrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "verify-listing",
                        ElapsedMS = partMeasure.ElapsedMilliseconds
                    });

                    partMeasure.Restart();
                    var result = atlasMethod.GetListingByFilter(ref filter, metrics, context, requireOwnershipVerification);
                    statuscode = result.Item1;
                    partMeasure.Stop();
                    metrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "get-listing-by-filter",
                        ElapsedMS = partMeasure.ElapsedMilliseconds
                    });
                    if (result != null && result.Item1 == 200)
                    {
                        response.Success = true;
                        response.ResponseData = result.Item2;
                        if (response.ResponseData.Items != null && response.ResponseData.Items.Any())
                        {
                            if (eventType == Constants.AtlasInternalEventType.Contract)
                            {
                                partMeasure.Restart();
                                statuscode = PostProcessAtlasContractListing(ref filter, ref response, metrics, context);
                                partMeasure.Stop();
                                metrics.Add(new ElapsedTimeModel()
                                {
                                    MetricName = "post-process-contract-listing",
                                    ElapsedMS = partMeasure.ElapsedMilliseconds
                                });
                            }
                            else if (eventType == Constants.AtlasInternalEventType.PremiumContentCreator)
                            {
                                partMeasure.Restart();
                                statuscode = PostProcessAtlasPCCListing(ref response, metrics, context);
                                partMeasure.Stop();
                                metrics.Add(new ElapsedTimeModel()
                                {
                                    MetricName = "post-process-pcc-listing",
                                    ElapsedMS = partMeasure.ElapsedMilliseconds
                                });
                            }
                            else if (eventType == Constants.AtlasInternalEventType.Event)
                            {
                                partMeasure.Restart();
                                AtlasListing responseData = response.ResponseData;
                                statuscode = PostProcessAtlasEventListing(ref responseData, metrics, context);
                                partMeasure.Stop();
                                metrics.Add(new ElapsedTimeModel()
                                {
                                    MetricName = "post-process-event-listing",
                                    ElapsedMS = partMeasure.ElapsedMilliseconds
                                });
                            }
                        }
                    }
                }
            }
            finally
            {
                stopwatch.Stop();
                response.Metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "listing_wireup",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statuscode;
        }

        public static bool VerifyGetListing(Constants.AtlasInternalEventType eventType, string[] defaultColumns, HttpContext context, IColumnFilterService _columnFilterService, IActivityLoggerService _activityLoggerService, IActionContextAccessor _actionContextAccessor, ref UserAuthData tokenData, ref ApiResponse<AtlasListing> response, ref ListingFilter filter, ref string actionName, ref string controllerName)
        {
            bool success = false;
            bool fail = false;
            try
            {
                if (filter == null)
                {
                    response.ErrorMessage = "Filter is null";
                    response.RequestEndTime = DateTime.UtcNow;
                    fail = true;
                }

                if (!fail)
                {
                    if (filter.Limit <= 0 || filter.Limit > 5000 || filter.Page < 0 || filter.EventType == Constants.AtlasInternalEventType.Unknown)
                    {
                        response.ErrorMessage = "Filter limit <= 0, > 100, Page < 0, or Unknown aggregation type";
                        response.RequestEndTime = DateTime.UtcNow;
                        fail = true;
                    }

                    filter.EventType = eventType;

                    if (filter.Columns != null && filter.Columns.Length == 1)
                    {
                        if (filter.Columns[0] == "*")
                        {
                            filter.Columns = defaultColumns;
                        }
                    }
                    filter.FilterOutRestrictedColumns(context.Request, _columnFilterService, _activityLoggerService, tokenData.UserID, _actionContextAccessor);

                    _activityLoggerService.LogActivity(new ActivityLogData() { Action = actionName, Controller = controllerName, Timestamp = DateTime.UtcNow, User = tokenData.UserID, Message = "Searched Data: " + filter.EventType.ToString() });
                    if (filter.QueryFilters != null)
                    {
                        foreach (var searchFilter in filter.QueryFilters)
                        {
                            _activityLoggerService.LogActivity(new ActivityLogData() { Action = actionName, Controller = controllerName, Timestamp = DateTime.UtcNow, User = tokenData.UserID, Message = "Used Filter: " + searchFilter.Key });
                        }
                    }
                }
                success = !fail;
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            return success;
        }
    }
}
