﻿using CsvHelper;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Novell.Directory.Ldap.Events.Edir.EventData;
using Resonance.Core;
using Resonance.Core.ConstantData.Amp;
using Resonance.Core.Extensions;
using Resonance.Core.Helpers;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.DateTimeHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.RequestModels;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Models.DatabaseModels.RequestModels;
using Resonance.Core.Models.DatabaseModels.TwitchUserListingModels;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading.Tasks;

namespace Resonance.Microservices.Methods
{
    public class AmpMethods
    {
        private static AmpQuery ampQuery = new AmpQuery();

        public void Initialize()
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            ampQuery.Initialize();

            stopwatch.Stop();
            Log.Verbose($@"AmpMethods Initialize took {stopwatch.ElapsedMilliseconds}ms total");
        }

        public static object ImportLock = new object();

        public UserListingOverviewModel GetUserListingOverview(ref ListingFilter filter)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var result = AmpQuery.GetUserListingOverview(ref filter);

            stopwatch.Stop();
            Log.Verbose($@"GetUserListingOverview took {stopwatch.ElapsedMilliseconds}ms total");

            return result;
        }

        public IEnumerable<RequestUserSearchUser> SearchUsers(string login)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var result = AmpQuery.SearchUsers(login);

            stopwatch.Stop();
            Log.Verbose($@"SearchUsers took {stopwatch.ElapsedMilliseconds}ms total");

            return result;
        }

        public RequestList GetRequests(string profileName, HttpContext httpContext)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            List<Request> requestsToApprove = new List<Request>();
            requestsToApprove.AddRange((PermissionHelper.HasPermission(httpContext, ConstantsPermissions.Amp.CanApproveRequests)) ? AmpQuery.GetRequests(RequestStatus.PendingApproval) : new Request[] { });
            IEnumerable<Request> revisedRequests = AmpQuery.GetRequests(RequestStatus.Revised);
            if (PermissionHelper.HasPermission(httpContext, ConstantsPermissions.Amp.CanApproveRequests))
            {
                requestsToApprove.AddRange(revisedRequests.Select(x => GetRequest(x.ID, true).Request));
            }
            RequestList requestList = new RequestList();
            requestList.InProgressRequests = AmpQuery.GetRequestsByProfile(profileName, RequestStatus.Created);
            requestList.NeedsApprovalRequests = requestsToApprove;
            requestList.SubmittedRequests = (PermissionHelper.HasPermission(httpContext, ConstantsPermissions.Amp.CanApproveRequests) || PermissionHelper.HasPermission(httpContext, ConstantsPermissions.Amp.CanSeeAllRequests)) ? AmpQuery.GetRequests(RequestStatus.Submitted) : AmpQuery.GetRequestsByProfile(profileName, RequestStatus.Submitted);
            requestList.FinalizedRequests = (PermissionHelper.HasPermission(httpContext, ConstantsPermissions.Amp.CanApproveRequests) || PermissionHelper.HasPermission(httpContext, ConstantsPermissions.Amp.CanSeeAllRequests)) ? AmpQuery.GetRequests(RequestStatus.Finalized) : AmpQuery.GetRequestsByProfile(profileName, RequestStatus.Finalized);
            requestList.PendingFinalizationRequests = (PermissionHelper.HasPermission(httpContext, ConstantsPermissions.Amp.CanApproveRequests) || PermissionHelper.HasPermission(httpContext, ConstantsPermissions.Amp.CanSeeAllRequests)) ? AmpQuery.GetRequests(RequestStatus.PendingFinalization) : AmpQuery.GetRequestsByProfile(profileName, RequestStatus.PendingFinalization);
            requestList.DraftRequests = AmpQuery.GetRequestsByProfile(profileName, RequestStatus.Draft);
            requestList.CompletedRequests = (PermissionHelper.HasPermission(httpContext, ConstantsPermissions.Amp.CanApproveRequests) || PermissionHelper.HasPermission(httpContext, ConstantsPermissions.Amp.CanSeeAllRequests)) ? AmpQuery.GetRequests(RequestStatus.Completed) : AmpQuery.GetRequestsByProfile(profileName, RequestStatus.Completed);
            stopwatch.Stop();
            Log.Verbose($@"GetRequests took {stopwatch.ElapsedMilliseconds}ms total");


            return requestList;
        }

        public void EndRequest(string requestID)
        {
            AmpQuery.UpdateRequestStatus(requestID, RequestStatus.Completed);
            SalesforceHelpers.UpdateRecordByExternalID(SalesforceNames.Amp_Request__C, SalesforceNames.Amp_ID__C, requestID, new { Status__c = RequestStatus.Completed });
        }

        public void DeleteRequest(string requestID, string profileName)
        {
            RequestDetails requestDetails = GetRequest(requestID, false, false);

            AmpQuery.RecordRequestStatusChange(requestID, requestDetails.Request.Status, RequestStatus.Deleted, profileName);
            AmpQuery.UpdateRequestStatus(requestID, RequestStatus.Deleted);
            SalesforceHelpers.UpdateRecordByExternalID(SalesforceNames.Amp_Request__C, SalesforceNames.Amp_ID__C, requestID, new { Status__c = RequestStatus.Deleted });
        }

        public List<RequestChannel> SortChannelListByStatus(List<RequestChannel> channels)
        {
            //list is reverse priority for sorting
            //a status not in list gets sorted to the end
            List<string> preferences = new List<string>() { RequestChannelStatus.Manual, RequestChannelStatus.Suggested, RequestChannelStatus.Deleted };

            return channels.OrderByDescending(x => preferences.IndexOf(x.Status)).ToList();
        }

        public List<RequestChannel> FilterOutDuplicateChannels(List<RequestChannel> channels)
        {
            List<RequestChannel> filteredList = new List<RequestChannel>();

            foreach (IGrouping<int, RequestChannel> channelGroup in channels.GroupBy(x => x.ChannelID))
            {
                if (channelGroup.Count() == 1)
                {
                    filteredList.Add(channelGroup.FirstOrDefault());
                }
                else
                {
                    filteredList.Add(SortChannelListByStatus(channelGroup.ToList())[0]);
                }
            }

            return filteredList;
        }

        public bool FinalizeRequest(HttpContext context, string requestID,AuthTokenData authTokenData, FinalizeRequest finalizeRequest)
        {
            finalizeRequest.Channels = FilterOutDuplicateChannels(finalizeRequest.Channels);

            //Grab a db copy to see who is new/existing in the db already
            RequestDetails originalRequest = GetRequest(requestID);

            //Set the priority in the db for any new channels manually prioritized
            foreach (var dataChannel in finalizeRequest.Channels.Where(x => x.Priority != null && x.Priority != 0 && x.Status != RequestChannelStatus.Deleted))
            {
                AmpQuery.UpdateRequestChannelPriority(requestID, dataChannel.ChannelLogin, dataChannel.Priority.Value);
            }
            
            //Mark any chanels as deleted that came through as deleted in the finalize request
            IEnumerable<string> deletedChannelLogins = finalizeRequest.Channels.Where(x => x.Status == RequestChannelStatus.Deleted).Select(x => x.ChannelLogin).ToArray();

            //Remove any deleted channels
            foreach (string deletedChannel in deletedChannelLogins)
            {
                AmpQuery.MarkRequestChannelDeleted(requestID, deletedChannel);
            }

            IEnumerable<string> existingChannelLogins = originalRequest.Channels.Where(x => x.Status != RequestChannelStatus.Deleted).Select(x => x.ChannelLogin).ToArray();
            IEnumerable<RequestChannel> dataChannels = finalizeRequest.Channels.Where(x => x.Status != RequestChannelStatus.Suggested && x.Status != RequestChannelStatus.Deleted);
            IEnumerable<RequestChannel> suggestedChannels = finalizeRequest.Channels.Except(dataChannels);
            HashSet<string> suggestedChannelIDs = new HashSet<string>(suggestedChannels.Select(x => x.ChannelLogin));

            dataChannels = finalizeRequest.Channels.Where(x => !suggestedChannelIDs.Contains(x.ChannelLogin) && x.Status != RequestChannelStatus.Deleted).ToArray();

            //Record any new channels added at the end
            foreach (var dataChannel in dataChannels.Where(x => !existingChannelLogins.Contains(x.ChannelLogin)))
            {
                AmpQuery.InsertNewRequestChannel(requestID, dataChannel);
            }

            //Set a priority on all of the other channels that need to be sent
            RecalcSortOrders(context, requestID, finalizeRequest.Channels);


            //Grab a fresh copy with manual and auto priority applied
            RequestDetails requestDetails = GetRequest(requestID);


            foreach (var dataChannel in requestDetails.Channels)
            {
                var channelExistsQuery = SalesforceHelpers.QueryData($"select Id from AMP_Request_Participant__c where AMP_Request__r.AMP_ID__c = '{requestID}' and Streamer__r.External_Account_ID__c = '{dataChannel.ChannelID}'");
                //participant exists in SF
                if (channelExistsQuery.Count > 0)
                {
                    string participantID = channelExistsQuery.ElementAt(0).Id;
                    object payload = null;
                    payload = new { Status__c = RequestChannelCreatorStatus.Pending, Sort_Order__c = (dataChannel.Priority ?? dataChannel.SortOrder ?? 0), Group__c = (dataChannel.Priority ?? dataChannel.SortOrder ?? 0) };
                    Log.Info("Salesforce Request Payload: " + JsonConvert.SerializeObject(payload));
                    var response = SalesforceHelpers.UpdateRecord(SalesforceNames.Amp_Request_Participant__C, participantID, payload);
                    if (!response)
                    {
                        throw new Exception("Error Saving Salesforce Participants");
                    }

                    Log.Info("Salesforce Request Payload Response: " + JsonConvert.SerializeObject(response));
                }
                else
                {
                    var batchPayload = new
                    {
                        records = new[] { dataChannel }.Select(channel =>
                        new
                        {
                            attributes = new { type = SalesforceNames.Amp_Request_Participant_Payload__C, referenceId = $"{requestID}_{channel.ChannelID}" },
                            AMP_ID__c = $"{requestID}_{channel.ChannelID}",
                            AMP_Request_ID__c = requestID,
                            Channel_ID__c = channel.ChannelID,
                            Channel_Name__c = channel.ChannelLogin,
                            Sort_Order__c = channel.Priority ?? channel.SortOrder ?? 0,
                            Group__c = channel.Priority ?? channel.SortOrder ?? 0
                        }
                )
                    };

                    Log.Info("Salesforce Request Payload: " + JsonConvert.SerializeObject(batchPayload));
                    var batchResponse = SalesforceHelpers.PostData($"/services/data/v45.0/composite/tree/{SalesforceNames.Amp_Request_Participant_Payload__C}", batchPayload);
                    if (batchResponse == null)
                    {
                        throw new Exception("Error Saving Salesforce Participants");
                    }
                    Log.Info("Salesforce Request Payload Response: " + JsonConvert.SerializeObject(batchResponse));
                }
            }
            


            AmpQuery.RecordRequestStatusChange(requestID, requestDetails.Request.Status, RequestStatus.Finalized, authTokenData.User);
            AmpQuery.UpdateRequestStatus(requestID, RequestStatus.Finalized);

            

            return true;
        }

        public bool SubmitRequest(string requestID, AuthTokenData authTokenData)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            RequestDetails requestDetails = GetRequest(requestID);


            string ownerUserID = SalesforceHelpers.GetOwnerID(requestDetails.Request.CreatedBy);

            string region = string.Join(';', requestDetails.Request.Region);
            if(requestDetails.Request.Region.Length == 1 && requestDetails.Request.Region[0] == "Any")
            {
                region = "";
            }

            string country = string.Join(';', requestDetails.Request.Country);
            if (requestDetails.Request.Country.Length == 1 && requestDetails.Request.Country[0] == "Any")
            {
                country = "";
            }

            var requestObject = new
            {
                attributes = new { type = SalesforceNames.Amp_Request__C, referenceId = requestID },
                name = requestDetails.Request.Name,
                Description__c = "<p>" + (requestDetails.Request.Description ?? "").Replace("\n", @"<br/>") + "</p>",
                Official_Statement__c = "<p>" + (requestDetails.Request.PublicStatement ?? "").Replace("\n", @"<br/>") + "</p>",
                AMP_ID__c = requestID,
                Target_Date__c = requestDetails.Request.DueDate?.ToString("yyyy-MM-dd"),
                Minimum_Engagement_Date__c = requestDetails.Request.EngagementDateMin?.ToString("yyyy-MM-dd"),
                Maximum_Engagement_Date__c = requestDetails.Request.EngagementDateMax?.ToString("yyyy-MM-dd"),
                Time_Commitment_Type__c = requestDetails.Request.TimeRequiredType,
                Time_Commitment_Amount__c = requestDetails.Request.TimeRequiredCount,
                Region__c = region,
                Language__c = requestDetails.Request.Language,
                Country__c = country,
                Requestor_Email__c = requestDetails.Request.CreatedBy + "@twitch.tv",
                Requestor_Name__c = requestDetails.Request.CreatedBy,
                Engagement_Type__c = requestDetails.Request.EngagementType,
                Opportunity_Source__c = requestDetails.Request.Type,
                Paid_Opportunity__c = requestDetails.Request.PaidOpportunity,
                OwnerId = ownerUserID,
                Number_Expected__c = requestDetails.Request.ParticipantCount,
                Status__c = RequestStatus.Submitted,
                Allow_Creator_Suggestions__c = requestDetails.Request.AllowAMSuggestions,
                Force_Flood_Sourcing__c = requestDetails.Request.Flood == true
            };
            var requestResponse = SalesforceHelpers.PostData($"/services/data/v45.0/composite/tree/{SalesforceNames.Amp_Request__C}", new { records = new[] { requestObject } });
            Log.Info("Salesforce Request Response: " + JsonConvert.SerializeObject(requestResponse));
            if (requestResponse == null)
            {
                throw new Exception("Salesforce Failed to Save");
            }
            if (!requestDetails.Request.AllowAMSuggestions)
            {
                AmpQuery.UpdateRequestStatus(requestID, RequestStatus.PendingFinalization);
                AmpQuery.RecordRequestStatusChange(requestID, requestDetails.Request.Status, RequestStatus.PendingFinalization, authTokenData.User);
                SalesforceHelpers.UpdateRecordByExternalID(SalesforceNames.Amp_Request__C, SalesforceNames.Amp_ID__C, requestID, new { Status__c = RequestStatus.PendingFinalization });
                return true;
            }
            AmpQuery.RecordRequestStatusChange(requestID, requestDetails.Request.Status, RequestStatus.Submitted, authTokenData.User);
            AmpQuery.UpdateRequestStatus(requestID, RequestStatus.Submitted);

            stopwatch.Stop();
            Log.Verbose($@"SubmitRequest took {stopwatch.ElapsedMilliseconds}ms total");

            return true;
        }

        public void RecalcSortOrders(HttpContext context, string requestID, IEnumerable<RequestChannel> existingChanels)
        {
            Dictionary<int, RequestChannel> existingChannelLookup = existingChanels.ToDictionary(x => x.ChannelID);
            var requestData = GetRequest(requestID);
            int sortOrder = 1;

            var channelBatches = requestData.Channels.Partition(50);
            Dictionary<long, float> channelLookup = new Dictionary<long, float>();
            foreach (var batch in channelBatches)
            {
                ListingFilter filter = new ListingFilter();
                filter.Limit = 50;
                filter.SortOrder = new SortFilter[] { new SortFilter() { Ascending = false, Column = "channelconcurrentsminuteswatchedtotal" } };
                filter.AggregateType = Constants.AggregationType.ThirtyDay;
                filter.Columns = new[] { "channelid", "channelconcurrentsminuteswatchedtotal" };
                filter.QueryFilters = new[] { new QueryFilter() { FilterType = Constants.FilterType.In, Key = "channelid", ValueArray = batch.Select(x => (dynamic)x.ChannelID).ToArray() } };
                var sortedChannelResponse = GetUserListingByFilter(context, ref filter);
                foreach(var channel in sortedChannelResponse)
                {
                    long channelID = channel.channelid;
                    float minutesWatched = channel.channelconcurrentsminuteswatchedtotal;
                    channelLookup[channelID] = minutesWatched;
                }
            }
            
            var priorityChannels = requestData.Channels.Where(x => x.Priority != null && x.Priority != 0).OrderBy(x => x.Priority).ToArray();            

            var nonPriorityChannels = requestData.Channels.Where(x => x.Priority == null || x.Priority == 0)
                .OrderBy(x => x.Status == RequestChannelStatus.Manual)
                .ThenBy(x => x.Status == RequestChannelStatus.Suggested)
                .ThenBy(x => x.Status == RequestChannelStatus.FilteredList)
                .OrderByDescending(x => channelLookup.ContainsKey(x.ChannelID) ? channelLookup[x.ChannelID] : 0).ToArray();

            var sortedChannels = priorityChannels.Union(nonPriorityChannels);

            foreach (var channel in sortedChannels)
            {
                bool newPriority = channel.Priority == null || channel.Priority == 0;
                channel.SortOrder = sortOrder++;
                channel.Priority = channel.SortOrder;
                //If we have in memory channel, then make sure this priority is set
                if (existingChannelLookup.ContainsKey(channel.ChannelID))
                {
                    existingChannelLookup[channel.ChannelID].Priority = channel.SortOrder;
                    existingChannelLookup[channel.ChannelID].SortOrder = channel.SortOrder;
                }
                if (newPriority)
                {
                    AmpQuery.UpdateRequestChannelPriority(requestID, channel.ChannelLogin, channel.SortOrder.Value);
                }
            }
        }

        

        public RequestDetails GetRequest(string requestID, bool isEdit = false, bool getChannels = true)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var result = AmpQuery.GetRequest(requestID, isEdit, getChannels);

            stopwatch.Stop();
            Log.Verbose($@"GetRequest took {stopwatch.ElapsedMilliseconds}ms total");

            return result;
        }

        public bool RateRequest(RateRequest rateRequest)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            bool dbResponse = AmpQuery.RateRequest(rateRequest);
            SalesforceHelpers.UpdateRecordByExternalID(SalesforceNames.Amp_Request_Participant__C, SalesforceNames.Amp_ID__C, $"{rateRequest.RequestID}_{rateRequest.ChannelID}", new { AMP_Rating__c = rateRequest.Rating, AMP_Comments__c = rateRequest.Comment });

            stopwatch.Stop();
            Log.Verbose($@"RateRequest took {stopwatch.ElapsedMilliseconds}ms total");

            return dbResponse;
        }

        public Request EditRequest(RequestDetails editRequest, string requestID, string profileName, string newStatus = RequestStatus.PendingApproval)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            string originalStatus = GetRequest(requestID)?.Request?.Status ?? "";
            editRequest.Channels = FilterOutDuplicateChannels(editRequest.Channels);
            string editedRequestID = AmpQuery.EditRequest(editRequest, requestID, null, newStatus, profileName);
            var result = new Request() { ID = editedRequestID };

            if (newStatus != originalStatus)
            {
                AmpQuery.RecordRequestStatusChange(requestID, originalStatus, newStatus, profileName);
            }

            stopwatch.Stop();
            Log.Verbose($@"EditRequest took {stopwatch.ElapsedMilliseconds}ms total");

            return result;
        }

        public Request ReviseRequest(RequestDetails editRequest, string requestID, string profileName)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            string editedRequestID = AmpQuery.ReviseRequest(editRequest, requestID, profileName);
            var result = new Request() { ID = editedRequestID };

            stopwatch.Stop();
            Log.Verbose($@"ReviseRequest took {stopwatch.ElapsedMilliseconds}ms total");

            return result;
        }

        public async Task<ChannelDetailsModel> GetChannelDetails(string twitchLogin, HttpContext context = null, ElapsedTimeModel metrics = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();
            // Get base data
            var details = AmpQuery.GetChannelDetails(twitchLogin);
            if (details == null)
            {
                return null;
            }

            // Get additional content

            var tasks = new ConcurrentBag<Task>
            {
                /* CLIPS */
                TwitchUserDetailsHelper.GetClips(details.ChannelID, metrics)
                .ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            details.Clips = data.Result;
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }),
                /* SCORECARD */
                TwitchUserDetailsHelper.GetScorecard(details.ChannelID, metrics).ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            details.Scorecard = data.Result;
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }),
                /* CALENDAR */
                TwitchUserDetailsHelper.GetCalendar(details.ChannelID, metrics).ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            details.Calendar = data.Result;
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }),
                /* VIEWBOT */
                TwitchUserDetailsHelper.GetViewbot(details.ChannelID, metrics).ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            details.Viewbot = data.Result;
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }),
                /* WEEKLY SUMMARY */
                /* Disabled due to non-use, re-enable on need (Disabled 2019-10-28)
                TwitchUserDetailsHelper.GetWeeklySummary(details.ChannelID, metrics).ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            details.WeeklySummary = data.Result;
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }),
                */
                /* STREAMS */
                TwitchUserDetailsHelper.GetStreams(details.ChannelID, details.Login, metrics).ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            details.Streams = data.Result;
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }),
                /* Top Countries */
                TwitchUserDetailsHelper.GetTopCountries(details.ChannelID, metrics).ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            details.TopCountries = data.Result;
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }),
                /* Emotes */
                TwitchUserDetailsHelper.GetEmotes(details.ChannelID, context, metrics).ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            details.Emotes = data.Result;
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }),
                /* Sub Tenure Badges */
                // https://prod.badges.twitch.a2z.com/v1/badges/channels/13405587/display
                /* Disabled until peering is complete. (Disabled 2019-10-28)*/
                TwitchUserDetailsHelper.GetTenureBadges(details.ChannelID, context, metrics).ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            details.TenureBadges = data.Result?.TenureBadges?.SubscriberBadges?.Badges;
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }), 
                
                /* Bit Cheermotes */
                // https://main.us-west-2.beta.payday.twitch.a2z.com/api/actions?channel_id=116076154
                // https://prod.payday.twitch.a2z.com/
                TwitchUserDetailsHelper.GetCheermotes(details.ChannelID, context, metrics).ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            var cheermotes = new Dictionary<string, Dictionary<string, CheermoteDisplayModel>>();
                            if(data.Result.Actions != null)
                            {
                                foreach(var item in data.Result.Actions)
                                {
                                    if (item.Type == "channel_custom")
                                    {
                                        if (!cheermotes.ContainsKey(item.Prefix))
                                        {
                                            cheermotes.Add(item.Prefix, new Dictionary<string, CheermoteDisplayModel>());
                                    
                                            foreach(var tier in item.Tiers)
                                            {
                                                if (!cheermotes[item.Prefix].ContainsKey(tier.ID))
                                                {
                                                    cheermotes[item.Prefix].Add(tier.ID, new CheermoteDisplayModel()
                                                    {
                                                        CanCheer = tier.CanCheer,
                                                        MinBits = tier.MinBits,
                                                        ShowInBitsCard = tier.ShowInBitsCard,
                                                        Type = item.Type,
                                                        Small = tier?.Images?.Light?.Animated?.FirstOrDefault(x => x.Key == "1").Value,
                                                        Medium = tier?.Images?.Light?.Animated?.FirstOrDefault(x => x.Key == "2").Value,
                                                        Large = tier?.Images?.Light?.Animated?.FirstOrDefault(x => x.Key == "4").Value
                                                    });
                                                }
                                            }
                                        }
                                    }
                                    details.Cheermotes = cheermotes;
                                }
                            }
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }),

                /* SALESFORCE AUDIT */
                /* Disabled due to no front end implementation (Disabled 2019-10-28)
                 * 
                TwitchUserDetailsHelper.GetPartnershipAuditData(details.ChannelID, metrics).ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        details.PartnerApplications = data.Result;
                    }
                })*/

                TwitchUserDetailsHelper.GetChannelListingData(twitchLogin)
                .ContinueWith((data) =>
                {
                    if (data.IsCompletedSuccessfully)
                    {
                        try
                        {
                            details.ValueScoreOverallWeighted = data.Result.ValueScoreOverallWeighted ?? 0;
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }),

            };

            try
            {
                await Task.WhenAny(Task.WhenAll(tasks), Task.Delay(5000));
            }
            catch (OperationCanceledException)
            {
                Log.Warn($@"One or more items had their operation canceled");
            }
            catch (TimeoutException)
            {
                Log.Warn($@"One or more items timed out");
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                // Patches in cheermotes to scorecard
                if (details.Cheermotes != null && details.Cheermotes.Count > 0 && details.Scorecard != null && details.Scorecard.ScorecardTask != null)
                {
                    foreach (var task in details.Scorecard.ScorecardTask)
                    {
                        if (task.Name == "meets_or_exceeds_cheermote_count")
                        {
                            task.IsComplete = true;
                        }
                    }
                }

                // Patches in tenure badges to scorecard
                if (details.TenureBadges != null && details.TenureBadges.Count > 0 && details.Scorecard != null && details.Scorecard.ScorecardTask != null)
                {
                    foreach (var task in details.Scorecard.ScorecardTask)
                    {
                        if (task.Name == "meets_or_exceeds_tenure_badge_count")
                        {
                            task.IsComplete = true;
                        }
                    }
                }

                stopwatch.Stop();
                Log.Info($@"Get User Details took {stopwatch.ElapsedMilliseconds}ms total.");
            }

            return details;
        }

        public RequestWithRevisionDetails GetRequestWithRevisions(string requestID)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            string revisionRequestID = AmpQuery.GetRevisionRequestID(requestID);
            RequestWithRevisionDetails request = new RequestWithRevisionDetails();
            request.Original = GetRequest(requestID);
            request.Revision = GetRequest(revisionRequestID);

            stopwatch.Stop();
            Log.Verbose($@"GetRequestWithRevisions took {stopwatch.ElapsedMilliseconds}ms total");

            return request;
        }

        public Request ProcessRequestRevisions(bool approve, string requestID, AuthTokenData authTokenData)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            AmpQuery.ProcessRequestRevisions(requestID, approve, authTokenData.User);
            SubmitRequest(requestID, authTokenData);

            var result = new Request() { ID = requestID };

            stopwatch.Stop();
            Log.Verbose($@"ProcessRequestRevisions took {stopwatch.ElapsedMilliseconds}ms total");

            return result;
        }

        public string GetCSVExport(HttpContext context, TwitchDataExportRequestModel model)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            try
            {
                ListingFilter filter = new ListingFilter();
                filter.Columns = model.Columns.Concat(new string[] { "mindate", "maxdate" }).ToArray();
                filter.QueryFilters = model.QueryFilters ?? new QueryFilter[] { };
                filter.Limit = 500;
                filter.Page = 0;
                filter.AggregateType = model.AggregationType;
                filter.SortOrder = new[] { new SortFilter() { Ascending = false, Column = TwitchUserListingColumnDefinition.ChannelConcurrentsMinutesWatchedTotal, Ordinal = 0 } };

                var tableData = GetUserListingByFilter(context, ref filter);

                using (MemoryStream stream = new MemoryStream())
                {
                    using (TextWriter textWriter = new StreamWriter(stream))
                    {
                        using (CsvWriter writer = new CsvWriter(textWriter))
                        {
                            foreach (string column in filter.Columns)
                            {
                                writer.WriteField(column);
                            }
                            writer.NextRecord();
                            foreach (var entry in tableData)
                            {
                                foreach (string column in filter.Columns)
                                {
                                    writer.WriteField((entry as IDictionary<string, Object>)[column]);
                                }
                                writer.NextRecord();
                            }

                            textWriter.Flush();
                        }
                        return Convert.ToBase64String(stream.ToArray());
                    }
                }
            }
            finally
            {
                stopwatch.Stop();
                Log.Verbose($@"GetCSVExport took {stopwatch.ElapsedMilliseconds}ms total");
            }
        }

        public Request CreateRequest(ref ListingFilter filter, string profileName)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            string requestID = Guid.NewGuid().ToString();

            AmpQuery.CreateRequest(ref filter, profileName, requestID);
            var result = new Request() { ID = requestID };

            stopwatch.Stop();
            Log.Verbose($@"CreateRequest took {stopwatch.ElapsedMilliseconds}ms total");

            return result;
        }

        public Request CreateEmptyRequest(string profileName)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            string requestID = Guid.NewGuid().ToString();

            AmpQuery.CreateEmptyRequest(profileName, requestID);
            var result = new Request() { ID = requestID };

            stopwatch.Stop();
            Log.Verbose($@"CreateEmptyRequest took {stopwatch.ElapsedMilliseconds}ms total");

            return result;
        }

        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public IEnumerable<dynamic> GetUserListingByFilter(HttpContext context, ref ListingFilter filter)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            if (filter == null)
            {
                throw new ArgumentNullException("filter");
            }

            if (filter.Limit <= 0 || filter.Page < 0 || filter.AggregateType == Constants.AggregationType.Unknown)
            {
                throw new ArgumentOutOfRangeException("filter");
            }

            IEnumerable<dynamic> result = null;

            try
            {
                result = AmpQuery.GetUserListing(context, filter: ref filter);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }

            if (result == null)
            {
                result = new EmptyModel[0];
            }

            stopwatch.Stop();
            Log.Verbose($@"GetUserListingByFilter took {stopwatch.ElapsedMilliseconds}ms total");

            return result;
        }

        public void CancelRequest(string requestID, string profileName)
        {            
            RequestDetails requestDetails = GetRequest(requestID, false, false);

            AmpQuery.RecordRequestStatusChange(requestID, requestDetails.Request.Status, RequestStatus.Canceled, profileName);
            AmpQuery.UpdateRequestStatus(requestID, RequestStatus.Canceled);
            SalesforceHelpers.UpdateRecordByExternalID(SalesforceNames.Amp_Request__C, SalesforceNames.Amp_ID__C, requestID, new { Status__c = RequestStatus.Canceled });
        }

        public void RejectRequest(string requestID, string profileName)
        {
            RequestDetails requestDetails = GetRequest(requestID, false, false);

            AmpQuery.RecordRequestStatusChange(requestID, requestDetails.Request.Status, RequestStatus.Draft, profileName);
            AmpQuery.UpdateRequestStatus(requestID, RequestStatus.Draft);
        }

        public Tuple<int, bool> RequestHasRevision(string requestID)
        {
            int status = 500;
            bool value = false;

            if (requestID != null)
            {
                bool? result = AmpQuery.RequestHasRevision(requestID);

                if (result != null)
                {
                    status = 200;
                    value = result.Value;
                }
            }

            return new Tuple<int, bool>(status, value);
        }

        public Tuple<int, bool> RequestIsRevision(string requestID)
        {
            int status = 500;
            bool value = false;

            if (requestID != null)
            {
                bool? result = AmpQuery.RequestIsRevision(requestID);

                if (result != null)
                {
                    status = 200;
                    value = result.Value;
                }
            }

            return new Tuple<int, bool>(status, value);
        }

        public bool ChannelsContainDuplicatePriorityValues(IList<RequestChannel> channels)
        {
            if (channels == null)
            {
                return false;
            }

            return channels
                .Where(x => x.Priority != null && x.Status != RequestChannelStatus.Deleted && x.Priority != 0)
                .GroupBy(x => x.Priority)
                .Where(x => x.Count() > 1)
                .Any();
        }
    }
}
