﻿using CsvHelper;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.AuthHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.AtlasModels;
using Resonance.Core.Models.ApiModels.RequestModels;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using Resonance.Core.Models.DatabaseModels.RequestModels;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Models.ServiceModels.AtlasModels;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using TimeZoneConverter;

namespace Resonance.Microservices.Methods
{
    /// <summary>
    /// TODO: NOTE FROM TOURNYMASTERBOT: We need to lift AtlasMethods so that Initialize takes in context / metrics / auth user tokens
    /// </summary>
    public class AtlasMethods
    {
        private static AtlasEndpointAuthHelper atlasOwnershipHelper = new AtlasEndpointAuthHelper();
        private static AtlasQuery atlasQuery = new AtlasQuery();

        public void Initialize()
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            atlasQuery.Initialize();

            stopwatch.Stop();
            Log.Verbose($@"AtlasMethods Initialize took {stopwatch.ElapsedMilliseconds}ms total");
        }

        public Tuple<int, AtlasListing> GetListingByFilter(ref ListingFilter filter, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context, bool requireOwnershipVerification = false)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var statuscode = 200;
            AtlasListing result = null;

            try
            {
                try
                {
                    if (requireOwnershipVerification)
                    {
                        var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, ref filter);
                        if (verifyOwnershipResult != null)
                        {
                            statuscode = verifyOwnershipResult.StatusCode;
                            if (verifyOwnershipResult.IsVerified)
                            {
                                result = atlasQuery.GetListing(context: context, filter: ref filter, metrics: metrics);
                            }
                        }
                    }
                    else
                    {
                        result = atlasQuery.GetListing(context: context, filter: ref filter, metrics: metrics);
                    }
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }

                if (result == null)
                {
                    result = new AtlasListing()
                    {
                        TotalPages = 0,
                        Items = null
                    };
                }
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "get_listing_by_filter",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }

            Log.Verbose($@"GetListingByFilter took {stopwatch.ElapsedMilliseconds}ms total");

            return new Tuple<int, AtlasListing>(statuscode, result);
        }

        public Tuple<int, IEnumerable<TopicListEntry>> GetTopicsFormats(ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var statuscode = 200;
            IEnumerable<TopicListEntry> result = null;

            try
            {
                result = atlasQuery.GetTopicsAndFormats(context, metrics);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }

            if (result == null)
            {
                result = new TopicListEntry[0];
            }

            stopwatch.Stop();
            Log.Verbose($@"GetTopicsFormats took {stopwatch.ElapsedMilliseconds}ms total");

            return new Tuple<int, IEnumerable<TopicListEntry>>(statuscode, result);
        }


        public Tuple<int, int> EditProduct(UserAuthData tokenData, ProductEditModal data, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var stopwatch = new Stopwatch();
            // Status Code, Product ID
            Tuple<int, int> statusCodeAndProductID = new Tuple<int, int>(500, 0);
            try
            {
                stopwatch.Start();
                var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, data.PremiumContentCreatorID);
                if(verifyOwnershipResult != null)
                {
                    if (verifyOwnershipResult.IsVerified)
                    {
                        statusCodeAndProductID = atlasQuery.EditProduct(context, tokenData, data, metrics);
                    }
                    else
                    {
                        statusCodeAndProductID = new Tuple<int, int>(verifyOwnershipResult.StatusCode, 0);
                    }

                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_product",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCodeAndProductID;
        }

        /// <summary>
        /// Returns a list of owned creators and another list of all creators
        /// </summary>
        public Tuple<int, GetManagedCreatorsResult> GetOwnerCreators(HttpContext context, string ldap, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statuscode = 200;

            List<PremiumContentCreator> ownedCreators = atlasQuery.GetOwnedCreators(context, ldap, metrics);
            List<PremiumContentCreator> viewableCreators = atlasQuery.GetOwnedCreators(context, "*", metrics);

            return new Tuple<int, GetManagedCreatorsResult>(statuscode, new GetManagedCreatorsResult
            {
                OwnedCreators = ownedCreators,
                ViewableCreators = viewableCreators
            });
        }

        /// <summary>
        /// Todo
        /// </summary>
        public Tuple<int, GroupedOwnedChannelList> GetOwnedChannels(HttpContext context, int pccID, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statuscode = 200;
            return new Tuple<int, GroupedOwnedChannelList>(statuscode, atlasQuery.GetOwnedChannels(context, pccID, metrics));
        }

        public int EditSeason(UserAuthData tokenData, AtlasSeason data, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var statusCode = 500;
            try
            {
                var premiumContentCreatorQuery = $@"select premium_content_creator_id from {Constants.DatabaseSchema}microservice_twitch_atlas_product where product_id = @product_id limit 1;";
                var pccIDs = atlasQuery.GetPremiumContentCreatorIDsFromQuery(context, premiumContentCreatorQuery, new Dictionary<string, dynamic>()
                {
                    { "product_id", data.ProductID }
                });

                var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, pccIDs);
                if(verifyOwnershipResult != null)
                {
                    statusCode = verifyOwnershipResult.StatusCode;
                    if (verifyOwnershipResult.IsVerified)
                    {
                        statusCode = atlasQuery.EditSeason(context, tokenData, data, metrics);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return statusCode;
        }

        /// <summary>
        /// Todo: Change auto complete query to return statuscode
        /// </summary>
        public Tuple<int, IEnumerable<string>> SearchGames(HttpContext context, string gameSearch, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                return new Tuple<int, IEnumerable<string>>(statuscode, atlasQuery.AutocompleteGames(context, gameSearch));
            }
            catch (Exception)
            {
                statuscode = 500;
            }
            return new Tuple<int, IEnumerable<string>>(statuscode, new string[] { });
        }

        /// <summary>
        /// Todo: Statuscode
        /// </summary>
        /// <param name="usernameSearch"></param>
        /// <param name="metrics"></param>
        /// <returns></returns>
        public Tuple<int, IEnumerable<RequestUserSearchUser>> SearchUsers(HttpContext context, string usernameSearch, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statuscode = 200;

            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                //Commented out as we wanted to use twitch but the quality is bad
                /*
                var userServiceSearchResults = TwitchUsersServiceHelpers.GetUsers(usernameSearch);
                List<RequestUserSearchUser> users = new List<RequestUserSearchUser>();
                foreach (var user in userServiceSearchResults.Results)
                {
                    users.Add(new RequestUserSearchUser() { ChannelID = user.ID, Login = user.Login, ProfileImage = user.ProfileImages });
                }
                return users;
                */
                var payload = new RequestUserSearch() { LoginNameSearch = usernameSearch };
                var ampResults = WebRequestHelper.PostData($"{Constants.AppConfig.Endpoints.Amp}worker/amp/request-user-auto-complete", WebRequestHelper.AppToken, ref payload, needsAuth: true, isToken: true);
                if(ampResults != null && ampResults.Data != null)
                {
                    var deserialize = JsonConvert.DeserializeObject<ApiListResponse<RequestUserSearchUser>>(ampResults.Data);
                    List<RequestUserSearchUser> users = deserialize?.ResponseData?.ToList();
                    if(users != null && users.Count > 0)
                    {
                        foreach (var user in users)
                        {
                            user.ProfileImage = "https://static-cdn.jtvnw.net/jtv_user_pictures/" + user.ProfileImage;
                        }
                    }
                    //Send a single lookup for exact results to users service to always get an exact match
                    if (!users.Any(x => x.Login == usernameSearch))
                    {
                        var userServiceSearchResults = TwitchUsersServiceHelpers.GetUserByLogin(context, usernameSearch);
                        if (userServiceSearchResults != null && userServiceSearchResults.Results != null)
                        {
                            foreach (var user in userServiceSearchResults.Results.Where(u => u != null))
                            {
                                users.Add(new RequestUserSearchUser() { ChannelID = user.ID, Login = user.Login, ProfileImage = user.ProfileImageUrl.FormatUsersServiceImageUrl() });
                            }
                        }
                        
                        if (userServiceSearchResults == null)
                        {
                            Log.Error("userServiceSearchResults is null");
                        } else if (userServiceSearchResults.Results == null)
                        {
                            Log.Error("userServiceSearchResults.Results is empty");
                        }
                    }
                    return new Tuple<int, IEnumerable<RequestUserSearchUser>>(statuscode, users);
                }
                else
                {
                    return new Tuple<int, IEnumerable<RequestUserSearchUser>>(200, null);
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
                return new Tuple<int, IEnumerable<RequestUserSearchUser>>(400, null);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "search_users_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
        }
        public int EditEventWithStreams(UserAuthData tokenData, AtlasEventWithStreamList data, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var statusCode = 500;
            try
            {
                TimeZoneInfo providedTimeZone = TZConvert.GetTimeZoneInfo(data.TimeZone);
                data.Event.StartTime = TimeZoneInfo.ConvertTimeToUtc(data.Event.StartTime.Value, providedTimeZone);
                data.Event.EndTime = TimeZoneInfo.ConvertTimeToUtc(data.Event.EndTime.Value, providedTimeZone);

                var verifyOwnershipResultEvent = atlasOwnershipHelper.VerifyUser(context, metrics, data.Event.PremiumContentCreatorID);
                if(verifyOwnershipResultEvent != null)
                {
                    statusCode = verifyOwnershipResultEvent.StatusCode;
                    if (verifyOwnershipResultEvent.IsVerified)
                    {
                        Tuple<int, int> statusCodeAndEventID = atlasQuery.EditEvent(context, tokenData, data.Event, metrics);
                        statusCode = statusCodeAndEventID.Item1;
                        int eventID = statusCodeAndEventID.Item2;
                        List<AtlasStream> atlasStreams = new List<AtlasStream>();
                        Dictionary<long, int> channelToStreamIDLookup = new Dictionary<long, int>();
                        ListingFilter existingEventStreamsFilter = new ListingFilter();
                        existingEventStreamsFilter.Limit = 4000;
                        existingEventStreamsFilter.Columns = new[] { "stream_id", "channel_id" };
                        existingEventStreamsFilter.EventType = Constants.AtlasInternalEventType.Stream;
                        existingEventStreamsFilter.QueryFilters = new[] { new QueryFilter() { FilterType = Constants.FilterType.Exact, Key = "event_id", Value = eventID } };
                        var existingStreamInfo = GetListingByFilter(ref existingEventStreamsFilter, metrics, context);
                        statusCode = existingStreamInfo.Item1;
                        if (existingStreamInfo.Item2?.Items != null)
                        {
                            foreach (var existingStream in existingStreamInfo.Item2.Items)
                            {
                                long channelID = existingStream.channel_id;
                                int streamID = existingStream.stream_id;
                                channelToStreamIDLookup[channelID] = streamID;
                            }
                        }
                        foreach (var stream in data.Streams)
                        {
                            AtlasStream atlasStream = new AtlasStream() { StreamID = channelToStreamIDLookup.ContainsKey(stream.ChannelID) ? channelToStreamIDLookup[stream.ChannelID] : 0, ChannelID = stream.ChannelID, ChannelType = stream.Type, EventID = eventID, IsActive = true, StreamLogin = stream.Channel, GameName = data.Event.GameName, StartTime = data.Event.StartTime, EndTime = data.Event.EndTime };
                            int code = EditStream(context, tokenData, atlasStream, metrics);
                            atlasStreams.Add(atlasStream);
                        }
                        IEnumerable<long> selectedChannelIDs = data.Streams.Select(x => x.ChannelID);
                        IEnumerable<int> streamIDsToRemove = channelToStreamIDLookup.Where(x => !selectedChannelIDs.Contains(x.Key)).Select(x => x.Value);
                        atlasQuery.DeleteStreams(context, streamIDsToRemove);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return statusCode;
        }

        public int EditEvent(UserAuthData tokenData, AtlasEvent data, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, data.PremiumContentCreatorID);
                if(verifyOwnershipResult != null)
                {
                    CloudwatchHelper.EnqueueMetricRequest("verify_ownership_edit_event_nn", 1, context);
                    statusCode = verifyOwnershipResult.StatusCode;
                    if (verifyOwnershipResult.IsVerified)
                    {
                        CloudwatchHelper.EnqueueMetricRequest("verify_ownership_edit_event_iv", 1, context);
                        statusCode = atlasQuery.EditEvent(context, tokenData, data, metrics).Item1;
                    }
                }
                else
                {
                    CloudwatchHelper.EnqueueMetricRequest("verify_ownership_edit_event_in", 1, context);
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_event_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        public int EditStream(HttpContext context, UserAuthData tokenData, AtlasStream data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                statusCode = atlasQuery.EditStream(context, tokenData, data, metrics);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_stream_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        public Tuple<int, int?> EditPremiumContentCreatorWithChannels(UserAuthData tokenData, CreatorWithStreamList data, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var statusCode = 500;
            int? pccID = null;
            var pccExists = atlasQuery.DoesPccExist(context, data.Creator.PremiumContentCreatorID, data.Creator.PremiumContentCreatorName);
            if (pccExists == null)
            {
                return new Tuple<int, int?>(statusCode, pccID);
            }
            if (pccExists == false)
            {
                var result = atlasQuery.EditPremiumContentCreator(context, tokenData, data.Creator, metrics);
                statusCode = result.Item1;
                pccID = result.Item2;
            }
            else if (pccExists == true)
            {
                // We found a match by name, look up the associated ID
                if(data.Creator.PremiumContentCreatorID == 0)
                {
                    data.Creator.PremiumContentCreatorID = atlasQuery.GetPremiumContentCreatorIDByName(context, data.Creator.PremiumContentCreatorName);
                }
                var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, data.Creator.PremiumContentCreatorID);
                if (verifyOwnershipResult != null)
                {
                    statusCode = verifyOwnershipResult.StatusCode;
                    if (verifyOwnershipResult.IsVerified)
                    {
                        var pccResponse = atlasQuery.EditPremiumContentCreator(context, tokenData, data.Creator, metrics);
                        statusCode = pccResponse.Item1;
                        pccID = pccResponse.Item2;

                        List<PremiumContentCreatorToChannelMap> atlasContractChannels = new List<PremiumContentCreatorToChannelMap>();
                        Dictionary<long, int> channelToPCCChannelLookup = new Dictionary<long, int>();
                        ListingFilter existingPCCChannelsFilter = new ListingFilter();
                        existingPCCChannelsFilter.Limit = 4000;
                        existingPCCChannelsFilter.Columns = new[] { "pcc_channel_map_id", "channel_id" };
                        existingPCCChannelsFilter.EventType = Constants.AtlasInternalEventType.PccToChannelMap;
                        existingPCCChannelsFilter.QueryFilters = new[] { new QueryFilter() { FilterType = Constants.FilterType.Exact, Key = "premium_content_creator_id", Value = pccID } };
                        var existingPCCChannelInfo = GetListingByFilter(ref existingPCCChannelsFilter, metrics, context);
                        statusCode = existingPCCChannelInfo.Item1;
                        if (existingPCCChannelInfo.Item2?.Items != null)
                        {
                            foreach (var existingChannel in existingPCCChannelInfo.Item2.Items)
                            {
                                long channelID = existingChannel.channel_id;
                                int pccChannelID = existingChannel.pcc_channel_map_id;
                                channelToPCCChannelLookup[channelID] = pccChannelID;
                            }
                        }
                        if(pccID != null)
                        {
                            foreach (var channel in data.Channels)
                            {
                                PremiumContentCreatorToChannelMap atlasContractChannel = new PremiumContentCreatorToChannelMap() { PccChannelMapId = channelToPCCChannelLookup.ContainsKey(channel.ChannelID) ? channelToPCCChannelLookup[channel.ChannelID] : 0, ChannelID = channel.ChannelID, IsActive = true, PremiumContentCreatorID = pccID.Value, ChannelOwnership = channel.ChannelOwnership, ChannelVertical = channel.ChannelVertical };
                                int code = EditPremiumContentCreatorChannels(tokenData, atlasContractChannel, metrics, context);
                                atlasContractChannels.Add(atlasContractChannel);
                            }
                            IEnumerable<long> selectedChannelIDs = data.Channels.Select(x => x.ChannelID);
                            IEnumerable<int> pccChannelIDsToRemove = channelToPCCChannelLookup.Where(x => !selectedChannelIDs.Contains(x.Key)).Select(x => x.Value);
                            atlasQuery.DeletePCCChannels(context, pccChannelIDsToRemove);

                            List<PremiumContentCreatorToAccountManagerMap> amMaps = new List<PremiumContentCreatorToAccountManagerMap>();
                            Dictionary<int, int> amMapLookup = new Dictionary<int, int>();
                            ListingFilter existingAMMapsFilter = new ListingFilter();
                            existingAMMapsFilter.Limit = 4000;
                            existingAMMapsFilter.Columns = new[] { "account_manager_id", "pcc_am_map_id" };
                            existingAMMapsFilter.EventType = Constants.AtlasInternalEventType.PccToAmMap;
                            existingAMMapsFilter.QueryFilters = new[] { new QueryFilter() { FilterType = Constants.FilterType.Exact, Key = "premium_content_creator_id", Value = pccID } };
                            var existingAMMaps = GetListingByFilter(ref existingAMMapsFilter, metrics, context);
                            statusCode = existingAMMaps.Item1;
                            if (existingAMMaps.Item2?.Items != null)
                            {
                                foreach (var existingStream in existingAMMaps.Item2.Items)
                                {
                                    int amID = existingStream.account_manager_id;
                                    int amMapID = existingStream.pcc_am_map_id;
                                    amMapLookup[amID] = amMapID;
                                }
                            }
                            foreach (var am in data.AccountManagerIDs)
                            {
                                PremiumContentCreatorToAccountManagerMap atlasPCCAMMap = new PremiumContentCreatorToAccountManagerMap() { PccToAmMapID = amMapLookup.ContainsKey(am) ? amMapLookup[am] : 0, AccountManagerID = am, IsActive = true, PremiumContentCreatorID = pccID.Value };
                                int code = EditPremiumContentCreatorAccountManager(tokenData, atlasPCCAMMap, metrics, context);
                                amMaps.Add(atlasPCCAMMap);
                            }
                            IEnumerable<int> amMapIDsToRemove = amMapLookup.Where(x => !data.AccountManagerIDs.Contains(x.Key)).Select(x => x.Value);
                            atlasQuery.DeletePCCAms(context, amMapIDsToRemove);
                        }
                    }
                }
            }
            return new Tuple<int, int?>(statusCode, pccID);
        }

        public Tuple<int, int?> EditPremiumContentCreator(UserAuthData tokenData, PremiumContentCreator data, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            int? pccID = null;
            try
            {
                stopwatch.Start();
                var pccExists = atlasQuery.DoesPccExist(context, data.PremiumContentCreatorID, data.PremiumContentCreatorName);
                if(pccExists == null)
                {
                    return new Tuple<int, int?>(statusCode, pccID);
                }
                if (pccExists == false)
                {
                    var result = atlasQuery.EditPremiumContentCreator(context, tokenData, data, metrics);
                    statusCode = result.Item1;
                    pccID = result.Item2;
                }
                else if(pccExists == true)
                {
                    var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, data.PremiumContentCreatorID);
                    if (verifyOwnershipResult != null)
                    {
                        statusCode = verifyOwnershipResult.StatusCode;
                        if (verifyOwnershipResult.IsVerified)
                        {
                            var result = atlasQuery.EditPremiumContentCreator(context, tokenData, data, metrics);
                            statusCode = result.Item1;
                            pccID = result.Item2;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_pcc_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return new Tuple<int, int?>(statusCode, pccID);
        }

        public int EditPremiumContentCreatorChannels(UserAuthData tokenData, PremiumContentCreatorToChannelMap data, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var pccExists = atlasQuery.DoesPccExist(context, data.PremiumContentCreatorID, null);
                if (pccExists == null)
                {
                    return statusCode;
                }
                else if (pccExists == true)
                {
                    var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, data.PremiumContentCreatorID);
                    if (verifyOwnershipResult != null)
                    {
                        statusCode = verifyOwnershipResult.StatusCode;
                        if (verifyOwnershipResult.IsVerified)
                        {
                            statusCode = atlasQuery.EditPremiumContentCreatorChannels(context, tokenData, data, metrics);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_pcc_channels_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        public int EditPremiumContentCreatorAccountManager(UserAuthData tokenData, PremiumContentCreatorToAccountManagerMap data, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, data.PremiumContentCreatorID);
                if(verifyOwnershipResult != null)
                {
                    statusCode = verifyOwnershipResult.StatusCode;
                    if (verifyOwnershipResult.IsVerified)
                    {
                        statusCode = atlasQuery.EditPremiumContentCreatorAccountManager(context, tokenData, data, metrics);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_pcc_am_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        public int EditAccountManager(HttpContext context, UserAuthData tokenData, AtlasAccountManager data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                // Permissions check expected to limit to admin
                statusCode = atlasQuery.EditAccountManager(context, tokenData, data, metrics);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_am_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        public int EditContract(UserAuthData tokenData, AtlasContract data, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, data.PremiumContentCreatorID);
                if(verifyOwnershipResult != null)
                {
                    statusCode = verifyOwnershipResult.StatusCode;
                    if (verifyOwnershipResult.IsVerified)
                    {
                        statusCode = atlasQuery.EditContract(context, tokenData, data, metrics).Item1;
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_contract_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        public Tuple<int, int> EditContractWithChannels(UserAuthData tokenData, AtlasContractWithChannelList data, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            Tuple<int, int> statusCodeAndContractID = null;
            try
            {
                var verifyOwnershipResult = atlasOwnershipHelper.VerifyUser(context, metrics, data.Contract.PremiumContentCreatorID);
                if(verifyOwnershipResult != null)
                {
                    statusCodeAndContractID = new Tuple<int, int>(verifyOwnershipResult.StatusCode, 0);
                    if (verifyOwnershipResult.IsVerified)
                    {
                        statusCodeAndContractID = atlasQuery.EditContract(context, tokenData, data.Contract, metrics);
                        int statusCode = statusCodeAndContractID.Item1;
                        int contractID = statusCodeAndContractID.Item2;

                        List<AtlasContractChannel> atlasContractChannels = new List<AtlasContractChannel>();
                        Dictionary<long, int> channelToContractChannelLookup = new Dictionary<long, int>();
                        ListingFilter existingContractChannelsFilter = new ListingFilter();
                        existingContractChannelsFilter.Limit = 4000;
                        existingContractChannelsFilter.Columns = new[] { "contract_channel_id", "channel_id" };
                        existingContractChannelsFilter.EventType = Constants.AtlasInternalEventType.ContractChannel;
                        existingContractChannelsFilter.QueryFilters = new[] { new QueryFilter() { FilterType = Constants.FilterType.Exact, Key = "contract_id", Value = contractID } };
                        var existingStreamInfo = GetListingByFilter(ref existingContractChannelsFilter, metrics, context);
                        statusCode = existingStreamInfo.Item1;
                        if (existingStreamInfo.Item2?.Items != null)
                        {
                            foreach (var existingStream in existingStreamInfo.Item2.Items)
                            {
                                long channelID = existingStream.channel_id;
                                int contractChannelID = existingStream.contract_channel_id;
                                channelToContractChannelLookup[channelID] = contractChannelID;
                            }
                        }
                        foreach (var channel in data.Channels)
                        {
                            AtlasContractChannel atlasContractChannel = new AtlasContractChannel() { ChannelContractID = channelToContractChannelLookup.ContainsKey(channel.ChannelID) ? channelToContractChannelLookup[channel.ChannelID] : 0, ChannelID = channel.ChannelID, IsActive = true, ContractID = contractID, CustomStartDate = channel.CustomStartDate, CustomEndDate = channel.CustomEndDate };
                            int code = EditContractChannels(context, tokenData, atlasContractChannel, metrics);
                            atlasContractChannels.Add(atlasContractChannel);
                        }
                        IEnumerable<long> selectedChannelIDs = data.Channels.Select(x => x.ChannelID);
                        IEnumerable<int> contractChannelIDsToRemove = channelToContractChannelLookup.Where(x => !selectedChannelIDs.Contains(x.Key)).Select(x => x.Value);
                        atlasQuery.DeleteContractChannels(context, contractChannelIDsToRemove);

                        List<AtlasContractAccountManagerMap> amMaps = new List<AtlasContractAccountManagerMap>();
                        Dictionary<int, int> amMapLookup = new Dictionary<int, int>();
                        ListingFilter existingAMMapsFilter = new ListingFilter();
                        existingAMMapsFilter.Limit = 4000;
                        existingAMMapsFilter.Columns = new[] { "account_manager_id", "contract_account_manager_map_id" };
                        existingAMMapsFilter.EventType = Constants.AtlasInternalEventType.ContractAccountManagerMap;
                        existingAMMapsFilter.QueryFilters = new[] { new QueryFilter() { FilterType = Constants.FilterType.Exact, Key = "contract_id", Value = contractID } };
                        var existingAMMaps = GetListingByFilter(ref existingAMMapsFilter, metrics, context);
                        statusCode = existingAMMaps.Item1;
                        if (existingAMMaps.Item2?.Items != null)
                        {
                            foreach (var existingStream in existingAMMaps.Item2.Items)
                            {
                                int amID = existingStream.account_manager_id;
                                int amMapID = existingStream.contract_account_manager_map_id;
                                amMapLookup[amID] = amMapID;
                            }
                        }
                        foreach (var am in data.AccountManagerIDs)
                        {
                            AtlasContractAccountManagerMap atlasContractAMMap = new AtlasContractAccountManagerMap() { ContractAccountManagerMapID = amMapLookup.ContainsKey(am) ? amMapLookup[am] : 0, AccountManagerID = am, IsActive = true, ContractID = contractID };
                            int code = EditContractAccountManagerMap(context, tokenData, atlasContractAMMap, metrics);
                            amMaps.Add(atlasContractAMMap);
                        }
                        IEnumerable<int> amMapIDsToRemove = amMapLookup.Where(x => !data.AccountManagerIDs.Contains(x.Key)).Select(x => x.Value);
                        atlasQuery.DeleteContractChannels(context, contractChannelIDsToRemove);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return statusCodeAndContractID;
        }

        public int EditContractChannels(HttpContext context, UserAuthData tokenData, AtlasContractChannel data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                // This methods permissions are gated on the controller to Administrator for the direct access method, or account manager (gated by contract)
                statusCode = atlasQuery.EditContractChannel(context, tokenData, data, metrics);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_contract_channels_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        public int EditContractAccountManagerMap(HttpContext context, UserAuthData tokenData, AtlasContractAccountManagerMap data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                statusCode = atlasQuery.EditContractAccountManagerMap(context, tokenData, data, metrics);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_contract_account_manager_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        public Tuple<int, SeasonEventList> GetSeasonEventList(ref ListingFilter filter, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            SeasonEventList eventList = new SeasonEventList();
            try
            {
                filter.EventType = Constants.AtlasInternalEventType.Season;
                filter.Columns = AtlasListingDefaults.SeasonDefault();
                var seasonResult = GetListingByFilter(ref filter, metrics, context);
                statuscode = seasonResult.Item1;
                AtlasListing seasons = seasonResult.Item2;
                List<int> seasonIDs = new List<int>();
                List<int> productIDs = new List<int>();
                Dictionary<int, dynamic> seasonLookup = new Dictionary<int, dynamic>();
                foreach (dynamic season in seasons.Items)
                {
                    int seasonID = season.season_id;
                    int productID = season.product_id;
                    seasonIDs.Add(seasonID);
                    productIDs.Add(productID);
                    seasonLookup.Add(seasonID, season);
                }
                Dictionary<int, string> productNames = atlasQuery.GetProductNames(context, productIDs, metrics);
                Tuple<int, Dictionary<int, List<dynamic>>> eventData = atlasQuery.GetSeasonEventList(seasonIDs, filter, metrics, context);
                statuscode = eventData.Item1;
                eventList.Groupings = new List<SeasonGroupedEvent>();
                foreach (var season in seasonLookup)
                {
                    if (eventData.Item2.ContainsKey(season.Key))
                    {
                        int productID = season.Value.product_id;
                        eventList.Groupings.Add(new SeasonGroupedEvent() { Season = $"{(productNames.ContainsKey(productID) ? productNames[productID] : "")} - {season.Value.season_name}", Events = eventData.Item2[season.Key] });
                    }
                }
                eventList.TotalPages = seasons.TotalPages;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "get_season_event_list_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }

            return new Tuple<int, SeasonEventList>(statuscode, eventList);
        }

        public Tuple<int, CategoryGroupedEventList> GetCategoryEventList(ref ListingFilter filter, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                Dictionary<string, CategoryGroupedEvent> groupLookup = new Dictionary<string, CategoryGroupedEvent>();
                CategoryGroupedEventList groupedEventList = new CategoryGroupedEventList() { Groupings = new List<CategoryGroupedEvent>() };
                filter.EventType = Constants.AtlasInternalEventType.Event;
                filter.Columns = new[] { "game_name" };
                List<QueryFilter> filters = filter.QueryFilters.ToList();
                filters.Add(new QueryFilter() { FilterType = Constants.FilterType.NotEqual, Key = "game_name", Value = "" });
                filter.QueryFilters = filters;
                List<string> gameNames = new List<string>();
                var games = atlasQuery.GetListing(context, ref filter, metrics, groupBy: "game_name", overrideCountColumn: "count(distinct(game_name))");
                foreach (dynamic gameRow in games.Items)
                {
                    string gameName = gameRow.game_name;
                    gameNames.Add(gameName);
                }
                var streamResults = atlasQuery.GetEventListByGameNames(gameNames, metrics, filter, context);
                statuscode = streamResults.Item1;
                foreach (dynamic streamEntry in streamResults.Item2)
                {
                    string gameName = streamEntry.game_name;
                    if (!groupLookup.ContainsKey(gameName))
                    {
                        groupLookup.Add(gameName, new CategoryGroupedEvent() { Category = gameName, Events = new List<dynamic>() });
                    }
                    groupLookup[gameName].Events.Add(streamEntry);
                }
                foreach (var group in groupLookup)
                {
                    CategoryGroupedEvent categoryGroup = group.Value;
                    string gameName = group.Key;
                    var gameViewerData = TwitchLivelineAPIHelper.GetStreamSummaryByGameName(gameName);
                    if (gameViewerData != null && gameViewerData.Summaries != null && gameViewerData.Summaries.Any())
                    {
                        int viewers = gameViewerData.Summaries[0]["viewcount"];
                        categoryGroup.Viewers = viewers;
                    }
                    groupedEventList.Groupings.Add(categoryGroup);
                }
                groupedEventList.TotalPages = games.TotalPages;

                return new Tuple<int, CategoryGroupedEventList>(statuscode, groupedEventList);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "group_event_category_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
        }

        /// <summary>
        /// Todo
        /// </summary>
        public Tuple<int, GroupedEventList> GetTopicEventList(HttpContext context, ref ListingFilter filter, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                Dictionary<string, GroupedEvent> groupLookup = new Dictionary<string, GroupedEvent>();
                GroupedEventList groupedEventList = new GroupedEventList() { Groupings = new List<GroupedEvent>() };
                filter.EventType = Constants.AtlasInternalEventType.Event;
                filter.Columns = AtlasListingDefaults.EventDefault();
                List<string> topics = new List<string>();
                string baseQuery = $"select topic from {Constants.DatabaseSchema}microservice_twitch_atlas_event where is_active=1 and coalesce(topic, '') != '' group by coalesce(topic, '')";
                string countQuery = $"select count(*) from {Constants.DatabaseSchema}microservice_twitch_atlas_event where is_active=1 and coalesce(topic, '') != '' group by coalesce(topic, '')";
                var topicData = atlasQuery.GetListing(context, ref filter, metrics, baseQuery, countQuery);
                foreach (dynamic topicRow in topicData.Items)
                {
                    string topic = topicRow.topic;
                    topics.Add(topic);
                }
                var streamResults = atlasQuery.GetEventListByTopics(context, topics, metrics);
                foreach (dynamic streamEntry in streamResults)
                {
                    string topic = streamEntry.topic;
                    if (!groupLookup.ContainsKey(topic))
                    {
                        groupLookup.Add(topic, new GroupedEvent() { Grouping = topic, Events = new List<dynamic>() });
                    }
                    groupLookup[topic].Events.Add(streamEntry);
                }
                foreach (var group in groupLookup)
                {
                    GroupedEvent categoryGroup = group.Value;
                    groupedEventList.Groupings.Add(categoryGroup);
                }
                groupedEventList.TotalPages = topicData.TotalPages;

                return new Tuple<int, GroupedEventList>(statuscode, groupedEventList);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "topic_event_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
        }

        /// <summary>
        /// Todo
        /// </summary>
        public Tuple<int, GroupedEventList> GetFormatEventList(HttpContext context, ref ListingFilter filter, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                Dictionary<string, GroupedEvent> groupLookup = new Dictionary<string, GroupedEvent>();
                GroupedEventList groupedEventList = new GroupedEventList() { Groupings = new List<GroupedEvent>() };
                filter.EventType = Constants.AtlasInternalEventType.Event;
                filter.Columns = AtlasListingDefaults.EventDefault();
                List<string> topics = new List<string>();
                string baseQuery = $"select format from {Constants.DatabaseSchema}microservice_twitch_atlas_event where is_active=1 and format != '' group by format";
                string countQuery = $"select count(*) from {Constants.DatabaseSchema}microservice_twitch_atlas_event where is_active=1 and format != '' group by format";
                var topicData = atlasQuery.GetListing(context, ref filter, metrics, baseQuery, countQuery);
                foreach (dynamic topicRow in topicData.Items)
                {
                    string format = topicRow.format;
                    topics.Add(format);
                }
                var streamResults = atlasQuery.GetEventListByFormats(context, topics, metrics);
                foreach (dynamic streamEntry in streamResults)
                {
                    string format = streamEntry.format;
                    if (!groupLookup.ContainsKey(format))
                    {
                        groupLookup.Add(format, new GroupedEvent() { Grouping = format, Events = new List<dynamic>() });
                    }
                    groupLookup[format].Events.Add(streamEntry);
                }
                foreach (var group in groupLookup)
                {
                    GroupedEvent categoryGroup = group.Value;
                    groupedEventList.Groupings.Add(categoryGroup);
                }
                groupedEventList.TotalPages = topicData.TotalPages;

                return new Tuple<int, GroupedEventList>(statuscode, groupedEventList);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "format_event_list_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
        }

        public Tuple<int, ChannelGroupedEventList> GetChannelEventList(ref ListingFilter filter, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;

            try
            {
                stopwatch.Start();
                //Gather streams from atlas
                filter.EventType = Constants.AtlasInternalEventType.Stream;
                filter.Columns = AtlasListingDefaults.StreamDefault();
                var streams = GetListingByFilter(ref filter, metrics, context);
                HashSet<long> channelIDs = new HashSet<long>();
                Dictionary<long, ChannelGroupedEvent> channelLookup = new Dictionary<long, ChannelGroupedEvent>();
                statuscode = streams.Item1;
                foreach (dynamic stream in streams.Item2.Items)
                {
                    long channelID = stream.channel_id;
                    channelIDs.Add(channelID);
                    channelLookup[channelID] = new ChannelGroupedEvent() { ChannelID = channelID, ChannelStatus = "Offline" };
                }
                //Lookup those channels on liveline
                LivelineGetStreamsResponse livelineChannelsResponse = TwitchLivelineAPIHelper.GetChannelResponseByChannelIDs(channelIDs.Select(x => x.ToString()));
                if (livelineChannelsResponse != null)
                {
                    foreach (dynamic stream in livelineChannelsResponse.Streams)
                    {
                        int channelID = stream.channel_id;
                        if (channelLookup.ContainsKey(channelID))
                        {
                            int viewers = stream.viewcount_data.viewcount;
                            channelLookup[channelID].ChannelStatus = "Online";
                            channelLookup[channelID].Viewers = viewers;
                        }
                    }
                }
                TwitchUsersServiceGetUsersResponse usersServiceResponse = TwitchUsersServiceHelpers.GetUsers(context, channelIDs);
                if (usersServiceResponse != null)
                {
                    foreach (var channel in usersServiceResponse.Results)
                    {
                        long channelID = channel.ID;
                        string channelName = channel.Login;
                        string profileImage = channel.ProfileImageUrl.FormatUsersServiceImageUrl();
                        channelLookup[channelID].ChannelLogin = channelName;
                        channelLookup[channelID].ChannelProfileImage = profileImage;
                    }
                }
                

                Dictionary<long, List<dynamic>> eventData = atlasQuery.GetChannelStreamList(context, channelIDs, metrics);
                ChannelGroupedEventList eventList = new ChannelGroupedEventList();
                eventList.Groupings = new List<ChannelGroupedEvent>();
                foreach (var channel in channelLookup)
                {
                    if (eventData.ContainsKey(channel.Key))
                    {
                        List<int> eventIDs = new List<int>();
                        foreach (dynamic eventStream in eventData[channel.Key])
                        {
                            int eventID = eventStream.event_id;
                            eventIDs.Add(eventID);
                        }
                        var result = atlasQuery.GetEventListByIDs(eventIDs, metrics, context);
                        statuscode = result.Item1;
                        IEnumerable<dynamic> eventResults = result.Item2;
                        ChannelGroupedEvent grouping = channelLookup[channel.Key];
                        grouping.Events = eventResults.ToList();
                        eventList.Groupings.Add(grouping);
                    }
                }
                eventList.TotalPages = streams.Item2.TotalPages;

                return new Tuple<int, ChannelGroupedEventList>(statuscode, eventList);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "get_channel_event_list_method",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
        }

        /// <summary>
        /// todo
        /// </summary>
        public Tuple<int, CalendarEventListByDate> GetActiveEventsForCalendar(HttpContext context, int pccID)
        {
            var statuscode = 200;
            return new Tuple<int, CalendarEventListByDate>(statuscode, atlasQuery.GetActiveEventsForCalendar(context, pccID));
        }

        public Tuple<int, IEnumerable<PccStats>> AggregateProductStats(HttpContext context, int[] pccIds)
        {
            var statuscode = 200;
            return new Tuple<int, IEnumerable<PccStats>>(statuscode, atlasQuery.GetAggregateProductStats(context, pccIds));
        }
        
        public Tuple<int, IList<ChannelMetaData>> GetChannelMetaDataForEvent(HttpContext context, int eventID)
        {
            var statuscode = 200;
            IList<long> channelIDs = atlasQuery.GetChannelIDsFromEvent(context, eventID);
            if (channelIDs.Count() == 0)
            {
                return new Tuple<int, IList<ChannelMetaData>>(400, new List<ChannelMetaData>());
            }            

            Dictionary<long, ChannelMetaData> dict = new Dictionary<long, ChannelMetaData>();
            dict = channelIDs.ToDictionary(x => x, y => new ChannelMetaData { ChannelID = y, ChannelName = "", ProfileImage = "" });            
            int batchsize = 5000;
            int totalpages = (int)Math.Ceiling((decimal)channelIDs.Count() / batchsize);
            int page = 1;

            while (page <= totalpages)
            {                
                long[] channelBatch = channelIDs.Skip((page - 1) * batchsize).Take(batchsize).ToArray();

                if (channelBatch.Length == 0)
                {
                    continue;
                }
                TwitchUsersServiceGetUsersResponse usersServiceResponse = TwitchUsersServiceHelpers.GetUsers(context, channelBatch);
                if (usersServiceResponse != null)
                {
                    foreach (var channel in usersServiceResponse.Results)
                    {
                        long channelID = channel.ID;
                        string channelName = channel.Login;
                        string profileImage = channel.ProfileImageUrl.FormatUsersServiceImageUrl();
                        dict[channelID].ChannelName = channelName;
                        dict[channelID].ProfileImage = profileImage;
                    }

                }

                page++;
            }

            return new Tuple<int, IList<ChannelMetaData>>(statuscode, dict.Values.ToList());
        }

        public Tuple<int, IList<PccChannelData>> GetPccChannelData(HttpContext context, int pccID)
        {
            var statuscode = 200;
            IList<long> channels =  atlasQuery.GetChannelsByPcc(context, pccID);            

            if (channels.Any())
            {
                Dictionary<long, PccChannelData> dict = channels.ToDictionary(k => k, v => new PccChannelData { ChannelID = v });
                var usersServiceResponse = TwitchUsersServiceHelpers.GetUsers(context, channels);

                if (usersServiceResponse != null && usersServiceResponse.Results != null)
                {
                    foreach (var data in usersServiceResponse.Results)
                    {                        
                        dict[data.ID].Login = data.Login;
                        dict[data.ID].ProfileImage = data.ProfileImageUrl.FormatUsersServiceImageUrl();
                    }

                    return new Tuple<int, IList<PccChannelData>>(statuscode, dict.Values.ToList());
                }
            }

            return new Tuple<int, IList<PccChannelData>>(statuscode, new List<PccChannelData>());
        }

        public IList<ContractsByPCC> GetContractsByPremiumContentCreator(HttpContext context)
        {
            return atlasQuery.GetContractsByPremiumContentCreator(context);
        }

        public IList<EventCsvRecord> GetEventByPcc(int pccID)
        {   
            Dictionary<int, List<EventChannelMap>> channelsForEvent = atlasQuery
                .GetChannelInfoForEventByPCC(pccID)
                .GroupBy(x => x.EventID)
                .ToDictionary(x => x.FirstOrDefault().EventID, y => y.ToList());

            IList<EventCsvRecord> eventsByPcc = atlasQuery.GetEventsByPcc(pccID)
                .GroupBy(x => x.EventID)
                .Select(x => x.FirstOrDefault())
                .OrderBy(x => x.EventID)
                .ToList();

            foreach (EventCsvRecord e in eventsByPcc.Where(x => channelsForEvent.ContainsKey(x.EventID)))
            {
                e.ChannelIDs = string.Join('|', channelsForEvent[e.EventID].Select(x => x.ChannelID));
                e.ChannelNames = string.Join('|', channelsForEvent[e.EventID].Select(x => x.ChannelName));
            }

            return eventsByPcc;
        }

        public string GetEventCsvByPcc(int pccID)
        {
            IList<EventCsvRecord> events = GetEventByPcc(pccID);

            using (MemoryStream stream = new MemoryStream())
            {
                using (TextWriter textWriter = new StreamWriter(stream))
                {
                    using (CsvWriter writer = new CsvWriter(textWriter))
                    {
                        writer.WriteHeader<EventCsvRecord>();
                        writer.NextRecord();
                        foreach (EventCsvRecord record in events)
                        {
                            writer.WriteRecord(record);
                            writer.NextRecord();
                        }

                        textWriter.Flush();
                    }
                }

                return Convert.ToBase64String(stream.ToArray());
            }
        }
    }
}
