﻿using Resonance.Core.Extensions;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Text;
using static Resonance.Core.Constants;

namespace Resonance.Microservices.Methods
{
    /// <summary>
    /// PUBLIC methods to access internal query methods
    /// </summary>
    public class TwitchUserMethods
    {
        private TwitchUserQuery twitchUserQuery = new TwitchUserQuery();

        public void Initialize()
        {
            twitchUserQuery.Initialize();
        }

        #region GET
        public TwitchUserModel GetTwitchUser(long? twitchUserID = null, string twitchLogin = null)
        {
            TwitchUserModel result = null;
            try
            {
                result = twitchUserQuery.GetModel(twitchUserID: twitchUserID, twitchLogin: twitchLogin);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        public IEnumerable<TwitchUserModel> GetTwitchUsers(long[] twitchUserIDs = null, string[] twitchLogins = null)
        {
            if 
            (
                (twitchUserIDs == null || twitchUserIDs.Length <= 0) 
                && (twitchLogins == null || twitchLogins.Length <= 0)
            )
            {
                throw new ArgumentOutOfRangeException("twitchUserIDs,twitchLogins");
            }
            IEnumerable<TwitchUserModel> result = null;
            try
            {
                result = twitchUserQuery.GetListModel(twitchUserIDs: twitchUserIDs, twitchLogins: twitchLogins);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }
        #endregion

        #region POST
        

        public long? InsertUpdateBulkTwitchUser(string workerIdentifier, string batchID, DateTime batchStart, string s3Bucket, string s3BasePath, string schemaprefix, TwitchUserModel[] data)
        {
            if (data == null || data.Length == 0)
            {
                return null;
            }

            long? result = 0;
            var now = DateTime.UtcNow;
            var filepath = $"{s3BasePath}{now.Year}/{now.Month}/{now.Day}/twitch-user-lookup/{workerIdentifier}_{batchID}.csv.gz";

            try
            {
                Log.Info($"Writing S3 File - {filepath}");
                using (var stream = new MemoryStream())
                {
                    using (var gzipStream = new GZipStream(stream, CompressionMode.Compress, true))
                    {
                        using (TextWriter writer = new StreamWriter(gzipStream, Encoding.UTF8))
                        {
                            using (var csv = new CsvHelper.CsvWriter(writer, new CsvHelper.Configuration.Configuration()
                            {
                                Delimiter = ",",
                                Quote = '"',
                                ShouldQuote = (field, context) => true,
                                Encoding = Encoding.UTF8
                            }))
                            {
                                csv.WriteRecords(data.Select(x => new
                                {
                                    TwitchUserID = x.TwitchUserID,
                                    TwitchLogin = x.TwitchLogin,
                                    IsActive = x.IsActive,
                                    InactiveReason = (int)x.InactiveReason,
                                    WorkerIdentifier = workerIdentifier,
                                    BatchID = batchID,
                                    BatchStart = batchStart.ToRedshiftDateFormat()
                                }));
                            }
                        }

                        S3Helper.UploadToS3(stream, s3Bucket, filepath);
                        Log.Info($"Upload Complete: {filepath}");
                    }
                    result += twitchUserQuery.InsertUpdateBulkTwitchUsers(ref s3Bucket, ref filepath, ref schemaprefix, ref workerIdentifier, ref batchID, ref data) ?? 0;
                }
            }
            catch (Exception ex)
            {
                result = null;
                Log.Error(ex);
            }
            finally
            {
                Log.Info($"Deleting Filepath: {filepath}");
                S3Helper.DeleteFromS3(s3Bucket, filepath);
                Log.Info($"Deleted Filepath: {filepath}");
                result = data.Length;
            }
            return result;
        }

        public bool? InsertUpdateTwitchUser(string schemaprefix, ref TwitchUserModel user)
        {
            if (user == null || user.TwitchUserID <= 0 || string.IsNullOrWhiteSpace(user.TwitchLogin))
            {
                return null;
            }

            return twitchUserQuery.InsertUpdateTwitchUser(ref schemaprefix, ref user);
        }

        public long? UpdateInactiveUsers(string redshiftSchema, TwitchBannedOrDeletedUserModel[] data)
        {
            long? processedRecords = null;
            try
            {
                if(data == null || data.Length == 0)
                {
                    return processedRecords;
                }

                var batch = new List<TwitchBannedOrDeletedUserModel>();
                foreach(var record in data)
                {
                    if(record.Reason != TwitchInactiveStatus.Active && record.Reason != TwitchInactiveStatus.Unknown && record.UserID > 0)
                    {
                        batch.Add(record);
                    }

                    if (batch.Count == 100)
                    {
                        long? resultCount = twitchUserQuery.UpdateInactiveUsers(ref redshiftSchema, ref batch);
                        batch.Clear();
                        if (resultCount != null)
                        {
                            processedRecords += resultCount.Value;
                        }
                    }
                }
                if(batch.Count > 0)
                {
                    long? resultCount = twitchUserQuery.UpdateInactiveUsers(ref redshiftSchema, ref batch);
                    batch.Clear();
                    if (resultCount != null)
                    {
                        processedRecords += resultCount.Value;
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            return processedRecords;
        }

        public bool? UpdateInactiveUser(string schemaprefix, TwitchBannedOrDeletedUserModel data)
        {
            bool? success = null;

            if (data == null || data.UserID <= 0)
            {
                throw new ArgumentOutOfRangeException("data");
            }

            try
            {
                success = twitchUserQuery.UpdateInactiveUser(ref schemaprefix, ref data);
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            return success;
        }
        #endregion
    }
}
