﻿using Resonance.Core.Extensions;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using Resonance.Microservices.Queries;
using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Text;

namespace Resonance.Microservices.Methods
{
    /// <summary>
    /// PUBLIC methods to access internal query methods
    /// </summary>
    public class TwitchVerifiedLinkMethods
    {
        private static TwitchVerifiedLinkQuery query { get; set; } = new TwitchVerifiedLinkQuery();

        public void Initialize()
        {
            query.Initialize();
        }

        #region GET
        public TwitchVerifiedLinkModel GetTwitchVerifiedLinkByTwitchUserID(long twitchUserID)
        {
            if (twitchUserID <= 0)
            {
                throw new ArgumentOutOfRangeException("twitchUserID");
            }
            TwitchVerifiedLinkModel result = null;
            try
            {
                result = query.GetModel(twitchUserID: twitchUserID);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        public IEnumerable<TwitchVerifiedLinkModel> GetTwitchVerifiedLinkGetAll()
        {
            IEnumerable<TwitchVerifiedLinkModel> result = null;
            try
            {
                result = query.GetListModel(getAllRecords: true);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        public IEnumerable<TwitchVerifiedLinkModel> GetTwitchVerifiedLinkByTwitchUserIDs(long[] twitchUserIDs)
        {
            if (twitchUserIDs == null || twitchUserIDs.Length <= 0)
            {
                throw new ArgumentOutOfRangeException("twitchUserIDs");
            }
            IEnumerable<TwitchVerifiedLinkModel> result = null;
            try
            {
                result = query.GetListModel(twitchUserIDs: twitchUserIDs);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        public IEnumerable<TwitchVerifiedLinkModel> GetTwitchVerifiedLinkByYoutubeTags(string[] tags)
        {
            if (tags == null || tags.Length <= 0)
            {
                throw new ArgumentOutOfRangeException("tags");
            }
            IEnumerable<TwitchVerifiedLinkModel> result = null;
            try
            {
                result = query.GetListModel(youtubeTags: tags);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        public bool DeleteTwitchVerifiedLinkByTwitchUserIDs(TwitchDeletedUserListModel twitchUserIDs)
        {
            if (twitchUserIDs == null || twitchUserIDs.DeletedUsers.Length <= 0)
            {
                throw new ArgumentOutOfRangeException("twitchUserIDs");
            }
            try
            {
                return query.DeleteByTwitchUserIDs(deleteUsers: twitchUserIDs);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return false;
        }

        public bool DeleteTwitchVerifiedLinkByYoutubeTags(TwitchDeletedUserListModel youtubeUsers)
        {
            if (youtubeUsers == null || youtubeUsers.DeletedUsers.Length <= 0)
            {
                throw new ArgumentOutOfRangeException("twitchUserIDs");
            }
            try
            {
                return query.DeleteByYoutubeTags(deleteUsers: youtubeUsers);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return false;
        }


        public IEnumerable<TwitchVerifiedLinkModel> GetTwitchVerifiedLinkByTwitchLogins(string[] twitchLogins)
        {
            if (twitchLogins == null || twitchLogins.Length <= 0)
            {
                throw new ArgumentOutOfRangeException("twitchLogins");
            }
            IEnumerable<TwitchVerifiedLinkModel> result = null;
            try
            {
                result = query.GetListModel(twitchLogins: twitchLogins);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        public IEnumerable<TwitchVerifiedLinkModel> GetTwitchVerifiedLinkByYoutubeTag(string[] youtubeTags)
        {
            if (youtubeTags == null || youtubeTags.Length <= 0)
            {
                throw new ArgumentOutOfRangeException("youtubeTags");
            }
            IEnumerable<TwitchVerifiedLinkModel> result = null;
            try
            {
                result = query.GetListModel(youtubeTags: youtubeTags);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        public TwitchVerifiedLinkModel GetTwitchVerifiedLinkByTwitchLogin(string twitchLogin)
        {
            if (string.IsNullOrWhiteSpace(twitchLogin))
            {
                throw new ArgumentOutOfRangeException("twitchLogin");
            }
            TwitchVerifiedLinkModel result = null;
            try
            {
                result = query.GetModel(twitchLogin: twitchLogin);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        public TwitchVerifiedLinkModel GetTwitchVerifiedLinkByYoutubeTag(string youtubeTag)
        {
            if (string.IsNullOrWhiteSpace(youtubeTag))
            {
                throw new ArgumentOutOfRangeException("youtubeTag");
            }
            TwitchVerifiedLinkModel result = null;
            try
            {
                result = query.GetModel(youtubeTag: youtubeTag);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }
        #endregion

        #region POST
        public long? InsertUpdateBulkTwitchVerifiedLink(string workerIdentifier, string batchID, DateTime batchStart, string s3Bucket, string s3BasePath, string schemaprefix, TwitchVerifiedLinkModel[] data)
        {
            if (data == null || data.Length == 0)
            {
                return null;
            }

            long? result = 0;
            var now = DateTime.UtcNow;
            var filepath = $"{s3BasePath}{now.Year}/{now.Month}/{now.Day}/twitch-verified-link/{workerIdentifier}_{batchID}.csv.gz";
            try
            {
                Log.Info($"Writing S3 File - {filepath}");
                using (var stream = new MemoryStream())
                {
                    using (var gzipStream = new GZipStream(stream, CompressionMode.Compress, true))
                    {
                        using (TextWriter writer = new StreamWriter(gzipStream, Encoding.UTF8))
                        {
                            using (var csv = new CsvHelper.CsvWriter(writer, new CsvHelper.Configuration.Configuration()
                            {
                                Delimiter = ",",
                                Quote = '"',
                                ShouldQuote = (field, context) => true,
                                Encoding = Encoding.UTF8
                            }))
                            {
                                csv.WriteRecords(data.Select(x => new
                                {
                                    TwitchUserID = x.TwitchUserID,
                                    YoutubeTag = x.YoutubeTag,
                                    TwitchLogin = x.TwitchLogin,
                                    AddedBy = x.AddedBy,
                                    LastUpdated = x.LastUpdated.ToRedshiftDateFormat(),
                                    WorkerIdentifier = workerIdentifier,
                                    BatchID = batchID,
                                    BatchStart = batchStart.ToRedshiftDateFormat()
                                }));
                            }
                        }
                        S3Helper.UploadToS3(stream, s3Bucket, filepath);
                        Log.Info($"Upload Complete: {filepath}");
                        if (query.InsertUpdateBulkTwitchVerifiedLink(s3Bucket, filepath, schemaprefix, workerIdentifier, batchID))
                        {
                            result = data.Length;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                result = null;
                Log.Error(ex);
            }
            finally
            {
                try
                {
                    Log.Info($"Deleting Filepath: {filepath}");
                    S3Helper.DeleteFromS3(s3Bucket, filepath);
                    Log.Info($"Deleted Filepath: {filepath}");
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }
            return result;
        }

        public TwitchVerifiedLinkModel InsertUpdateTwitchVerifiedLink(string schemaprefix, ref TwitchVerifiedLinkModel data)
        {
            TwitchVerifiedLinkModel result = null;
            try
            {
                if (data == null || !data.YoutubeTag.StartsWith("UC") || data.TwitchUserID <= 0)
                {
                    return null;
                }

                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {

                        command.CommandTimeout = 600;
                        command.CommandText =
                        $@"
                            update {schemaprefix}microservice_twitch_verified_link
                                set TwitchUserID = @twitchUserID,
                                TwitchLogin = @twitchLogin,
                                AddedBy = @addedBy,
                                LastUpdated = @lastUpdated
                            from {schemaprefix}microservice_twitch_verified_link as a
                            where 
                                YoutubeTag = @youtubeTag
                                and 
                                (
                                        isnull(a.TwitchUserID, 0) <> isnull(@twitchUserID, 0) 
                                        and isnull(a.TwitchLogin, '') <> isnull(@twitchLogin, '')
                                )
                            ;

                            insert into {schemaprefix}microservice_twitch_verified_link
                            select @twitchUserID, @youtubeTag, @twitchLogin, @addedBy, @lastUpdated
                            where not exists
                            (
                                select 1
                                from {schemaprefix}microservice_twitch_verified_link as b
                                where b.YoutubeTag = @youtubeTag
                            );
                                
                            select TwitchUserID, YoutubeTag, TwitchLogin, AddedBy, LastUpdated from {schemaprefix}microservice_twitch_verified_link where YoutubeTag = @youtubeTag;
                        ";
                        command.Parameters.AddWithValue("@twitchUserID", data.TwitchUserID);
                        command.Parameters.AddWithValue("@youtubeTag", data.YoutubeTag);
                        command.Parameters.AddWithValue("@twitchLogin", data.TwitchLogin);
                        command.Parameters.AddWithValue("@addedBy", data.AddedBy);
                        command.Parameters.AddWithValue("@lastUpdated", data.LastUpdated);
                        using (var wrappedreader = new DataReaderWithMeasurements(command, null, "InsertUpdateTwitchVerifiedLink"))
                        {
                            var reader = wrappedreader.NpgReader;
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    result = new TwitchVerifiedLinkModel()
                                    {
                                        TwitchUserID = reader.GetValue(0) as long?,
                                        YoutubeTag = reader.GetString(1),
                                        TwitchLogin = reader.GetString(2),
                                        AddedBy = reader.GetString(3),
                                        LastUpdated = reader.GetDateTime(4)
                                    };
                                }
                            }
                        }
                        command.Parameters.Clear();
                    }
                }
            }
            catch (Exception ex)
            {
                result = null;
                Log.Error(ex);
            }
            return result;
        }

        public long? RenameBulkTwitchVerifiedLink(string redshiftSchema, ref TwitchRenamedUserListModel data)
        {
            long? result = 0;
            try
            {
                if (data == null || data.Data == null || data.Data.Length == 0)
                {
                    return null;
                }

                var batch = new Dictionary<long, Tuple<string, string>>();
                var currentcount = 0;
                foreach (var user in data.Data)
                {
                    if (!batch.ContainsKey(user.TwitchUserID))
                    {
                        batch.Add(user.TwitchUserID, new Tuple<string, string>($@"@twitchLogin_{currentcount}", user.NewTwitchLogin));
                        currentcount++;
                        if (currentcount == 100)
                        {
                            result += ExecuteRenameBatch(ref redshiftSchema, ref batch) ?? 0;
                            currentcount = 0;
                            batch.Clear();
                        }
                    }
                }
                if (batch.Count > 0)
                {
                    result += ExecuteRenameBatch(ref redshiftSchema, ref batch) ?? 0;
                    currentcount = 0;
                    batch.Clear();
                }
            }
            catch (Exception ex)
            {
                result = null;
                Log.Error(ex);
            }
            return result;
        }

        private long? ExecuteRenameBatch(ref string schemaprefix, ref Dictionary<long, Tuple<string, string>> users)
        {
            try
            {
                if (users == null || users.Count == 0)
                {
                    return null;
                }

                var query = new StringBuilder();
                foreach (var user in users)
                {
                    query.AppendLine($"update {schemaprefix}microservice_twitch_verified_link set TwitchLogin = {user.Value.Item1}, LastUpdated = getdate() where TwitchUserID = {user.Key} and TwitchLogin <> {user.Value.Item1};");
                }

                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 600;
                        command.CommandText = query.ToString();
                        foreach (var user in users)
                        {
                            command.Parameters.AddWithValue(user.Value.Item1, user.Value.Item2);
                        }
                        command.ExecuteNonQueryWithMeasurements(logname: "ExecuteRenameBatch");
                    }
                }

                return users.Count;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                return null;
            }
        }
        #endregion
    }
}
