﻿using Microsoft.AspNetCore.Http;
using MySql.Data.MySqlClient;
using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.ConstantData.Amp;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.FilterHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Models.DatabaseModels.RequestModels;
using Resonance.Core.Models.DatabaseModels.TwitchUserListingModels;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Models.ServiceModels.ExplorerModels;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using static Resonance.Core.Constants;

namespace Resonance.Microservices.Queries
{
    /// <summary>
    /// This class should be internal.
    /// Methods in this should _not_ be static
    /// Accessing this from an external project should be done though 'AmpMethods'
    /// </summary>
    public partial class AmpQuery
    {
        private const int ChannelLimit = 50;
        internal void Initialize()
        {
        }
        
        internal TwitchVerifiedLinkModel GetModel(long twitchUserID)
        {
            throw new NotImplementedException();
        }
        
        public static string ReviseRequest(RequestDetails editRequest, string requestID, string profileName)
        {
            string revisedRequestID = GetRevisionRequestID(requestID);
            var existingRequest = GetRequest(requestID, false, false);
            if (revisedRequestID == null)
            {
                revisedRequestID = Guid.NewGuid().ToString();
                CreateEmptyRequest(profileName, revisedRequestID);
            }
            EditRequest(editRequest, revisedRequestID, requestID, RequestStatus.Revised, profileName);
            UpdateRequestStatus(requestID, RequestStatus.Revised);
            RecordRequestStatusChange(requestID, existingRequest.Request.Status, RequestStatus.Revised, profileName);
            return requestID;
        }

        public static ChannelDetailsModel GetChannelDetails(string twitchLogin)
        {
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    ChannelDetailsModel channelDetailsModel = new ChannelDetailsModel()
                    {
                        Tims = new UserTimsModel()
                    };
                    command.CommandText = $"select * from {Constants.DatabaseSchema}microservice_twitch_user_details where login = @login";
                    command.Parameters.AddWithValue("login", twitchLogin);
                    long channelID = 0;
                    using (var reader = new DataReaderWithMeasurements(command, null, "get_channel_details").MysqlReader)
                    {
                        if (reader.Read())
                        {
                            if(reader["ChannelID"] == System.DBNull.Value)
                            {
                                return null;
                            }
                            channelID = (long)reader["ChannelID"];
                            channelDetailsModel.ChannelID = channelID;
                            channelDetailsModel.Login = (string)reader["Login"];
                            channelDetailsModel.CCUTier = (string)reader["CCUTier"];
                            channelDetailsModel.ProfileImage = (string)reader["ProfileImage"];
                            channelDetailsModel.FollowerCountTotal = (long)reader["FollowerCountTotal"];
                            channelDetailsModel.SubsTotal = (long)reader["SubsTotal"];
                            channelDetailsModel.UserType = (string)reader["UserType"];
                            channelDetailsModel.MaxDate = (DateTime)reader["MaxDate"];
                            channelDetailsModel.LastBroadcastDate = (DateTime)reader["LastBroadcastDate"];
                            channelDetailsModel.AccountManagerUserName = reader["AccountManagerUserName"] == System.DBNull.Value ? null : (string)reader["AccountManagerUserName"];
                            channelDetailsModel.AccountManagerFullName = reader["AccountManagerFullName"] == System.DBNull.Value ? null : (string)reader["AccountManagerFullName"];
                            channelDetailsModel.MinutesWatchedData = reader["MinutesWatchedData"] == System.DBNull.Value ? null : (string)reader["MinutesWatchedData"];
                            channelDetailsModel.AverageCCUData = reader["AverageCCUData"] == System.DBNull.Value ? null : (string)reader["AverageCCUData"];
                            channelDetailsModel.RevenueData = reader["RevenueData"] == System.DBNull.Value ? null : (string)reader["RevenueData"];
                            channelDetailsModel.MonthlySubsData = reader["MonthlySubsData"] == System.DBNull.Value ? null : (string)reader["MonthlySubsData"];
                            channelDetailsModel.Tims.TwitchUserID = channelID;
                            channelDetailsModel.Tims.FormType = reader["TimsFormType"] == System.DBNull.Value ? "N/A" : (string)reader["TimsFormType"];
                            channelDetailsModel.Tims.IsUsPerson = reader["TimsIsUsPerson"] == System.DBNull.Value ? false : (bool)reader["TimsIsUsPerson"];
                            channelDetailsModel.Tims.IsUserActionRequired = reader["TimsIsUserActionRequired"] == System.DBNull.Value ? false : (bool)reader["TimsIsUserActionRequired"];
                            channelDetailsModel.Tims.Status = reader["TimsStatus"] == System.DBNull.Value ? "N/A" : (string)reader["TimsStatus"];
                            channelDetailsModel.Tims.WithholdingRate = reader["TimsWithholdingRate"] == System.DBNull.Value ? 0.0f : (float)reader["TimsWithholdingRate"];
                            channelDetailsModel.Tims.ServiceWithholdingRate = reader["TimsServiceWithholdingRate"] == System.DBNull.Value ? 0.0f : (float)reader["TimsServiceWithholdingRate"];
                            channelDetailsModel.Tims.RoyaltyWithholdingRate = reader["TimsRoyaltyWithholdingRate"] == System.DBNull.Value ? 0.0f : (float)reader["TimsRoyaltyWithholdingRate"];
                            channelDetailsModel.PathToAffiliateDate = reader["PathToAffiliateDate"] == System.DBNull.Value ? null : (DateTime?)(DateTime)reader["PathToAffiliateDate"];
                            channelDetailsModel.PathToPartnerDate = reader["PathToPartnerDate"] == System.DBNull.Value ? null : (DateTime?)(DateTime)reader["PathToPartnerDate"];
                        }
                        else
                        {
                            return null;
                        }
                    }
                    command.Parameters.Clear();
                    if(channelID > 0)
                    {
                        command.CommandText = $@"select BeforeAvg, MergedAvg, BestGuessAvg, Data from {Constants.DatabaseSchema}microservice_twitch_viewbot where ChannelID = @twitchuserid;";
                        command.Parameters.AddWithValue("@twitchuserid", channelID);
                        using (var reader = new DataReaderWithMeasurements(command, null, "get_channel_details_viewbot").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    channelDetailsModel.Viewbot = new TwitchViewbotModel();
                                    channelDetailsModel.Viewbot.BeforeAvg = reader["BeforeAvg"] == System.DBNull.Value ? null : (float?)((float)reader["BeforeAvg"]);
                                    channelDetailsModel.Viewbot.MergedAvg = reader["MergedAvg"] == System.DBNull.Value ? null : (float?)((float)reader["MergedAvg"]);
                                    channelDetailsModel.Viewbot.BestGuessAvg = reader["BestGuessAvg"] == System.DBNull.Value ? null : (float?)((float)reader["BestGuessAvg"]);
                                    channelDetailsModel.Viewbot.Data = reader["Data"] == System.DBNull.Value ? null : (string)reader["Data"];
                                }
                            }
                        }
                    }
                    return channelDetailsModel;
                }
            }
        }

        public static void UpdateRequestStatus(string requestID, string status)
        {
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request set status = @status where request_id = @request_id";
                    command.Parameters.AddWithValue("status", status);
                    command.Parameters.AddWithValue("request_id", requestID);
                    command.ExecuteNonQueryWithMeasurements("amp_update_request");
                }
            }
        }

        public static void ProcessRequestRevisions(string requestID, bool approve, string revisingUser)
        {
            string editRequestID = GetRevisionRequestID(requestID);
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    string status;
                    if (approve)
                    {
                        status = RequestStatus.Approved;
                        string[] fieldsToUpdate = new[] { "name", "description", "type", "privacy", "region", "country", "participant_count", "due_date", "engagement_date_min", "engagement_date_max", "engagement_type", "time_requirement_type", "time_requirement_count", "paid_opportunity", "public_statement", "contact_method", "allow_am_suggestions" };
                        command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request main_request inner join {Constants.DatabaseSchema}microservice_twitch_request revision_request on main_request.request_id = revision_request.edit_request_id set " +
                             string.Join(",", fieldsToUpdate.Select(field => $"main_request.{field} = COALESCE(revision_request.{field}, main_request.{field})").ToArray());
                        command.CommandText += ", main_request.revised_by = @revised_by";
                        command.CommandText += " where revision_request.request_id = @edit_request_id";
                        command.Parameters.AddWithValue("edit_request_id", editRequestID);
                        command.Parameters.AddWithValue("revised_by", revisingUser);

                        command.ExecuteNonQueryWithMeasurements("process_request_revision");
                        command.CommandText = $"delete from {Constants.DatabaseSchema}microservice_twitch_request where request_id = @edit_request_id";

                        command.ExecuteNonQueryWithMeasurements("process_request_revision_delete");


                        command.Parameters.Clear();
                        command.Parameters.AddWithValue("request_id", requestID);
                        command.Parameters.AddWithValue("edit_request_id", editRequestID);
                        command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request_channel c1 left join {Constants.DatabaseSchema}microservice_twitch_request_channel c2 on c2.channel_id = c1.channel_id and c2.request_id = @edit_request_id set c1.status = '{RequestChannelStatus.Deleted}' where c1.request_id = @request_id and c2.request_id is null";
                        command.ExecuteNonQueryWithMeasurements("process_request_revision_status1");

                        command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request_channel inner join {Constants.DatabaseSchema}microservice_twitch_request_channel c2 on c2.channel_id = {Constants.DatabaseSchema}microservice_twitch_request_channel.channel_id  set microservice_twitch_request_channel.status = c2.status where {Constants.DatabaseSchema}microservice_twitch_request_channel.request_id = @request_id and c2.request_id = @edit_request_id";
                        command.ExecuteNonQueryWithMeasurements("process_request_revision_status2");

                        command.CommandText = $"insert into {Constants.DatabaseSchema}microservice_twitch_request_channel (request_id, channel_id, channel_login, status, profile_image) select @request_id, channel_id, channel_login, status,profile_image from {Constants.DatabaseSchema}microservice_twitch_request_channel where request_id = @edit_request_id and channel_id not in (select channel_id from {Constants.DatabaseSchema}microservice_twitch_request_channel where request_id = @request_id)";
                        command.ExecuteNonQueryWithMeasurements("process_request_revision_insert");

                    }
                    else
                    {
                        status = RequestStatus.Edited;

                    }

                    command.Parameters.Clear();
                    command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request set status = @status where request_id = @request_id";
                    command.Parameters.AddWithValue("status", status);
                    command.Parameters.AddWithValue("request_id", requestID);

                    command.ExecuteNonQueryWithMeasurements("process_request_revision_status3");
                }
            }
        }

        public static string GetRevisionRequestID(string requestID)
        {
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandText = $"select request_id from {Constants.DatabaseSchema}microservice_twitch_request where edit_request_id = @edit_request_id";
                        command.Parameters.AddWithValue("edit_request_id", requestID);

                        string revisionRequestID = (string)command.ExecuteScalar();
                        return revisionRequestID;
                    }
                }
            }
            catch(Exception)
            {
                return null;
            }
        }

        public static void UpdateRequestChannelPriority(string requestID, string channelName, int priority)
        {
            using (var conn = DBManagerMysql.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request_channel set Priority = @priority, sort_order = @priority where request_id = @request_id and channel_login = @login";
                    command.Parameters.AddWithValue("@priority", priority);
                    command.Parameters.AddWithValue("@request_id", requestID);
                    command.Parameters.AddWithValue("@login", channelName);
                    command.ExecuteNonQueryWithMeasurements("update_request_channel_priority");
                }
            }
        }

        public static void MarkRequestChannelDeleted(string requestID, string channelLogin)
        {
            using (var conn = DBManagerMysql.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request_channel set status = '{RequestChannelStatus.Deleted}' where request_id = @request_id and channel_login = @channel_login";
                    command.Parameters.AddWithValue("request_id", requestID);
                    command.Parameters.AddWithValue("channel_login", channelLogin);
                    command.ExecuteNonQueryWithMeasurements("mark_request_channel_deleted");
                }
            }
        }

        public static void InsertNewRequestChannel(string requestID, RequestChannel channel)
        {
            using (var conn = DBManagerMysql.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = $"insert into {Constants.DatabaseSchema}microservice_twitch_request_channel (request_id, channel_id, channel_login, status, sort_order, profile_image, Priority) values(@request_id, @channel_id, @channel_login, @status, @priority, @profile_image, @priority)";
                    command.Parameters.AddWithValue("request_id", requestID);
                    command.Parameters.AddWithValue("channel_id", channel.ChannelID);
                    command.Parameters.AddWithValue("channel_login", channel.ChannelLogin);
                    command.Parameters.AddWithValue("status", channel.Status);
                    command.Parameters.AddWithValue("profile_image", channel.ProfileImage);
                    command.Parameters.AddWithValue("priority", channel.Priority ?? channel.SortOrder ?? 0);
                    command.ExecuteNonQueryWithMeasurements("insert_new_request_channel");
                }
            }
        }

        public static string EditRequest(RequestDetails editRequest, string requestID, string editRequestID = null, string status = RequestStatus.PendingApproval, string profileName = "")
        {
            RequestDetails originalRequest;
            RequestDetails dbRequest;
            string currentStatus = "";
            if(editRequestID != null)
            {
                originalRequest = GetRequest(editRequestID);
                dbRequest = originalRequest;
            }
            else
            {
                originalRequest = editRequest;
                dbRequest = GetRequest(requestID, false, false);
                currentStatus = dbRequest.Request.Status;
            }
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandTimeout = 10000;
                    command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request set type = @type,flood = @flood, due_date = @due_date, name = @name, description = @description, privacy = @privacy, participant_count = @expected_creators, contact_method = @contact_method, time_requirement_type = @time_requirement_type, time_requirement_count = @time_requirement_count, public_statement = @public_statement, region = @region, country = @country, engagement_type = @engagement_type, engagement_date_min = @engagement_date_min,engagement_date_max = @engagement_date_max ,paid_opportunity = @paid_opportunity, edit_request_id = @edit_request_id, status = @status, language = @language, allow_am_suggestions = @allow_am_suggestions where request_id = @request_id";
                    command.Parameters.AddWithValue("type", editRequest.Request.Type ?? originalRequest.Request.Type);
                    command.Parameters.AddWithValue("edit_request_id", editRequestID);
                    command.Parameters.AddWithValue("allow_am_suggestions", editRequest.Request.AllowAMSuggestions);
                    command.Parameters.AddWithValue("due_date", editRequest.Request.DueDate ?? originalRequest.Request.DueDate);
                    command.Parameters.AddWithValue("name", (editRequest.Request.Name == null || editRequest.Request.Name.Length == 0) ? originalRequest.Request.Name : editRequest.Request.Name);
                    command.Parameters.AddWithValue("description", (editRequest.Request.Description == null || editRequest.Request.Description.Length == 0) ? originalRequest.Request.Description : editRequest.Request.Description);
                    command.Parameters.AddWithValue("privacy", (editRequest.Request.RequestPrivacy == null || editRequest.Request.RequestPrivacy.Length == 0) ? originalRequest.Request.RequestPrivacy : editRequest.Request.RequestPrivacy);
                    command.Parameters.AddWithValue("expected_creators", editRequest.Request.ParticipantCount ?? originalRequest.Request.ParticipantCount);
                    command.Parameters.AddWithValue("time_requirement_type", editRequest.Request.TimeRequiredType ?? originalRequest.Request.TimeRequiredType);
                    command.Parameters.AddWithValue("time_requirement_count", editRequest.Request.TimeRequiredCount ?? originalRequest.Request.TimeRequiredCount);
                    command.Parameters.AddWithValue("public_statement", (editRequest.Request.PublicStatement == null || editRequest.Request.PublicStatement.Length == 0) ? originalRequest.Request.PublicStatement : editRequest.Request.PublicStatement);
                    command.Parameters.AddWithValue("request_id", requestID);
                    command.Parameters.AddWithValue("status", status);
                    command.Parameters.AddWithValue("contact_method", editRequest.Request.ContactMethod ?? originalRequest.Request.ContactMethod);
                    command.Parameters.AddWithValue("language", editRequest.Request.Language ?? originalRequest.Request.Language);
                    string finalRegion = "";
                    if(editRequest.Request.Region != null && editRequest.Request.Region.Length > 0)
                    {
                        finalRegion = string.Join(',',editRequest.Request.Region);
                    }
                    else if (originalRequest.Request.Region != null)
                    {
                        finalRegion = string.Join(',', originalRequest.Request.Region);
                    }
                    command.Parameters.AddWithValue("region", finalRegion);

                    string finalCountry = "";
                    if (editRequest.Request.Country != null && editRequest.Request.Country.Length > 0)
                    {
                        finalCountry = string.Join(',', editRequest.Request.Country);
                    }
                    else if (originalRequest.Request.Country != null)
                    {
                        finalCountry = string.Join(',', originalRequest.Request.Country);
                    }
                    command.Parameters.AddWithValue("country", finalCountry);                    
                    command.Parameters.AddWithValue("engagement_type", editRequest.Request.EngagementType ?? originalRequest.Request.EngagementType);
                    command.Parameters.AddWithValue("engagement_date_min", editRequest.Request.EngagementDateMin ?? originalRequest.Request.EngagementDateMin);
                    command.Parameters.AddWithValue("engagement_date_max", editRequest.Request.EngagementDateMax ?? originalRequest.Request.EngagementDateMax);
                    command.Parameters.AddWithValue("paid_opportunity", editRequest.Request.PaidOpportunity ?? originalRequest.Request.PaidOpportunity);
                    command.Parameters.AddWithValue("flood", editRequest.Request.Flood ?? originalRequest.Request.Flood);

                    command.ExecuteNonQueryWithMeasurements("EditRequest1");

                    command.Parameters.Clear();

                    Dictionary<int, RequestChannel> filteredChannels = new Dictionary<int, RequestChannel>();
                    foreach(var channel in editRequest.Channels)
                    {
                        if(!filteredChannels.ContainsKey(channel.ChannelID))
                        {
                            filteredChannels[channel.ChannelID] = channel;
                        }
                    }

                    command.CommandText = "drop table if exists microservice_twitch_request_channel_temp;create temporary table microservice_twitch_request_channel_temp(channel_id int,channel_login varchar(50), status varchar(16), profile_image varchar(100), priority int, sort_order int);";
                    int counter = 0;
                    HashSet<string> seenStatuses = new HashSet<string>();
                    int index = 1;
                    foreach(var channel in filteredChannels.Values)
                    {
                        channel.SortOrder = index++;
                    }
                    foreach (var channel in filteredChannels.Values.OrderByDescending(x => x.Status == RequestChannelStatus.Manual).ThenBy(x => x.SortOrder))
                    {
                        command.CommandText += "insert into microservice_twitch_request_channel_temp values(@channel_id_" + counter + ",@channel_login_" + counter + ", @status_" + channel.Status + ",@profile_image_" + counter + ", @priority_" + counter + ", @sort_order_" + counter + ");";
                        command.Parameters.AddWithValue("channel_id_" + counter, channel.ChannelID);
                        command.Parameters.AddWithValue("channel_login_" + counter, channel.ChannelLogin);
                        command.Parameters.AddWithValue("profile_image_" + counter, channel.ProfileImage);
                        command.Parameters.AddWithValue("priority_" + counter, channel.Priority);
                        command.Parameters.AddWithValue("sort_order_" + counter, counter);
                        if (!seenStatuses.Contains(channel.Status))
                        {
                            seenStatuses.Add(channel.Status);
                            command.Parameters.AddWithValue("status_" + channel.Status, channel.Status);
                        }
                        counter += 1;
                    }

                    command.ExecuteNonQueryWithMeasurements("EditRequest2");

                    command.Parameters.Clear();
                    command.Parameters.AddWithValue("request_id", requestID);
                    command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request_channel set microservice_twitch_request_channel.status = '{RequestChannelStatus.Deleted}' where request_id = @request_id and channel_id not in (select channel_id from microservice_twitch_request_channel_temp)";
                    command.ExecuteNonQueryWithMeasurements("EditRequest3");

                    command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request_channel inner join microservice_twitch_request_channel_temp on microservice_twitch_request_channel_temp.channel_id = {Constants.DatabaseSchema}microservice_twitch_request_channel.channel_id  set microservice_twitch_request_channel.status = microservice_twitch_request_channel_temp.status,microservice_twitch_request_channel.sort_order = microservice_twitch_request_channel_temp.sort_order  where {Constants.DatabaseSchema}microservice_twitch_request_channel.request_id = @request_id";
                    command.ExecuteNonQueryWithMeasurements("EditRequest4");

                    command.CommandText = $"insert into {Constants.DatabaseSchema}microservice_twitch_request_channel (request_id, channel_id, channel_login, status, profile_image, Priority, sort_order) select @request_id, channel_id, channel_login, status,profile_image, priority, sort_order from microservice_twitch_request_channel_temp where channel_id not in (select channel_id from {Constants.DatabaseSchema}microservice_twitch_request_channel where request_id = @request_id)";
                    command.ExecuteNonQueryWithMeasurements("EditRequest5");

                    List<RequestUser> authorizedUsersToAdd = new List<RequestUser>();
                    List<string> authorizedUsersToRemove = new List<string>();
                    HashSet<string> seenUserIDs = new HashSet<string>(dbRequest.AuthorizedUsers.Select(x => x.UserID));
                    authorizedUsersToAdd.AddRange(editRequest.AuthorizedUsers.Where(x => !seenUserIDs.Contains(x.UserID)));
                    authorizedUsersToRemove.AddRange(dbRequest.AuthorizedUsers.Select(x => x.UserID).Except(editRequest.AuthorizedUsers.Select(x => x.UserID)));

                    foreach (RequestUser userToAdd in authorizedUsersToAdd)
                    {
                        command.CommandText = $"insert into {Constants.DatabaseSchema}microservice_twitch_request_authorized_users values (@request_id,@user_id,@user_name)";
                        command.Parameters.Clear();
                        command.Parameters.AddWithValue("request_id", requestID);
                        command.Parameters.AddWithValue("user_name", userToAdd.UserName);
                        command.Parameters.AddWithValue("user_id", userToAdd.UserID);
                        command.ExecuteNonQueryWithMeasurements("insert_request_authorized_users");
                    }
                    foreach (string userToRemove in authorizedUsersToRemove)
                    {
                        command.CommandText = $"delete from {Constants.DatabaseSchema}microservice_twitch_request_authorized_users  where request_id = @request_id and user_id = @user_id";
                        command.Parameters.Clear();
                        command.Parameters.AddWithValue("request_id", requestID);
                        command.Parameters.AddWithValue("user_id", userToRemove);
                        command.ExecuteNonQueryWithMeasurements("delete_request_authorized_users");
                    }

                    command.CommandText = "drop table microservice_twitch_request_channel_temp";
                    command.ExecuteNonQueryWithMeasurements("EditRequest6");
                    if (editRequestID != null)
                    {
                        RecordRequestStatusChange(requestID, currentStatus, status, profileName);
                    }
                }
            }
            
            return requestID;
        }

        public static void CreateEmptyRequest(string profileName, string requestID)
        {
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = $"insert into {Constants.DatabaseSchema}microservice_twitch_request (request_id, created_by, created_date, status, participant_count, data_key) values(@request_id, @created_by, CURRENT_DATE, @status, @participant_count, @data_key)";
                    command.Parameters.AddWithValue("created_by", profileName);
                    command.Parameters.AddWithValue("request_id", requestID);
                    command.Parameters.AddWithValue("status", RequestStatus.Draft);
                    command.Parameters.AddWithValue("participant_count", 0);
                    command.Parameters.AddWithValue("data_key", AwsEncryption.GenerateKey(Constants.AppConfig.Application.KmsArn));

                    command.ExecuteNonQueryWithMeasurements("CreateEmptyRequest");

                }
            }
        }

        public static void CreateRequest(ref ListingFilter filter, string profileName, string requestID)
        {
            var overviewData = GetUserListingOverview(ref filter);
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = $"insert into {Constants.DatabaseSchema}microservice_twitch_request (request_id, created_by, created_date, status, participant_count, listing_filters,data_key) values(@request_id, @created_by, CURRENT_DATE, @status, @participant_count, @listing_filters,@data_key)";
                    command.Parameters.AddWithValue("created_by", profileName);
                    command.Parameters.AddWithValue("request_id", requestID);
                    command.Parameters.AddWithValue("status", RequestStatus.Draft);
                    command.Parameters.AddWithValue("participant_count", overviewData.TotalChannels);
                    command.Parameters.AddWithValue("listing_filters", JsonConvert.SerializeObject(filter));
                    command.Parameters.AddWithValue("data_key", AwsEncryption.GenerateKey(Constants.AppConfig.Application.KmsArn));
                    command.ExecuteNonQueryWithMeasurements("CreateRequest");

                    command.Parameters.Clear();
                    command.CommandText = $"insert into {Constants.DatabaseSchema}microservice_twitch_request_channel (request_id, channel_id, channel_login,status,profile_image) ";
                    command.Parameters.AddWithValue("request_id", requestID);
                    command.Parameters.AddWithValue("channel_type", RequestChannelStatus.FilteredList);

                    var queryData = GetUserListingFilter(ref filter, "@request_id, channelid,login,@channel_type, profileimage");
                    command.CommandText += $@"{queryData.Item1} limit {ChannelLimit}" ;
                    if(queryData.Item2 != null)
                    {
                        foreach(var param in queryData.Item2)
                        {
                            command.Parameters.AddWithValue(param.Key, param.Value);
                        }
                    }

                    command.ExecuteNonQueryWithMeasurements("CreateRequest");
                }
            }
            RecordRequestStatusChange(requestID, "", RequestStatus.Draft, profileName);
        }

        private static RequestUserSearchUser ReadUser(MySqlDataReader reader)
        {
            string channelLogin = (string)reader["Login"];
            string profileImage = (string)reader["ProfileImage"];
            long channelID = (long)reader["ChannelID"];
            string accountManager = "";
            string accountManagerFirstName = reader["AccountManagerFirstName"] != DBNull.Value ? (string)reader["AccountManagerFirstName"] : "";
            string accountManagerLastName = reader["AccountManagerLastName"] != DBNull.Value ? (string)reader["AccountManagerLastName"] : "";
            if (accountManagerFirstName != "" || accountManagerLastName != "")
            {
                accountManager = accountManagerFirstName + " " + accountManagerLastName;
            }

            return new RequestUserSearchUser() { Login = channelLogin, ProfileImage = profileImage, ChannelID = channelID, AccountManager = accountManager };
        }

        public static IEnumerable<RequestUserSearchUser> SearchUsers(string login)
        {
            List<RequestUserSearchUser> users = new List<RequestUserSearchUser>();

            //if user exists by exact name add to beginning of list
            if (!string.IsNullOrEmpty(login))
            {
                RequestUserSearchUser userExactMatch = SearchUsersExactMatch(login);
                if (userExactMatch != null)
                {
                    users.Add(userExactMatch);
                }
            }


            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.Parameters.AddWithValue("login", "%" + login + "%");
                    command.CommandText = $"select Login, ProfileImage, ChannelID, AccountManagerFirstName, AccountManagerLastName from {Constants.DatabaseSchema}microservice_twitch_user_listing_past_360_days where Login like @login order by ChannelConcurrentsMinutesWatchedTotal desc limit 0,10";
                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "AmpQuerySearchUsers"))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            var user = ReadUser(reader);
                            if (user.Login != login)
                            { //Don't add the exact match because if it exists it will already be in the list
                                users.Add(user);
                            }
                        }
                    }
                }
            }

            
            return users;
        }

        private static RequestUserSearchUser SearchUsersExactMatch(string login)
        {
            List<RequestUserSearchUser> users = new List<RequestUserSearchUser>();
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.Parameters.AddWithValue("login", login);
                    command.CommandText = $"select Login, ProfileImage, ChannelID, AccountManagerFirstName, AccountManagerLastName from {Constants.DatabaseSchema}microservice_twitch_user_listing_past_360_days where Login = @login order by ChannelConcurrentsMinutesWatchedTotal desc limit 0,1";
                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "AmpQuerySearchUsers"))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            users.Add(ReadUser(reader));
                        }
                    }
                }
            }

            return users.FirstOrDefault();
        }

        public static bool RateRequest(RateRequest request)
        {
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandText = $"update {Constants.DatabaseSchema}microservice_twitch_request_channel set rating = @rating, comment = @comment where request_id = @request_id and channel_id = @channel_id";
                        command.Parameters.AddWithValue("rating", request.Rating);
                        command.Parameters.AddWithValue("comment", request.Comment);
                        command.Parameters.AddWithValue("request_id", request.RequestID);
                        command.Parameters.AddWithValue("channel_id", request.ChannelID);
                        command.ExecuteNonQueryWithMeasurements("update_rate_request");

                    }
                }
            }
            catch(Exception)
            {
                return false;
            }
            return true;
        }

        public static void RecordRequestStatusChange(string requestID, string oldStatus, string newStatus, string editedBy)
        {
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = $"insert into {Constants.DatabaseSchema}microservice_twitch_request_timeline values(@request_id, @old_status, @new_status, NOW(), @edited_by)";
                    command.Parameters.AddWithValue("request_id", requestID);
                    command.Parameters.AddWithValue("old_status", oldStatus);
                    command.Parameters.AddWithValue("new_status", newStatus);
                    command.Parameters.AddWithValue("edited_by", editedBy);
                    command.ExecuteNonQueryWithMeasurements("record_request_status_change");
                }
            }
        }

        public static RequestDetails GetRequest(string requestID, bool isEdit = false, bool getChannels = true)
        {
            RequestDetails details = new RequestDetails();
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = $"select * from {Constants.DatabaseSchema}microservice_twitch_request where {(isEdit ? "edit_request_id" : "request_id")} = @request_id";
                    command.Parameters.AddWithValue("request_id", requestID);


                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "AmpQueryGetRequestQ1"))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            details.Request = ReadRequest(reader);
                        }
                    }
                    if(details.Request == null)
                    {
                        return null;
                    }
                    byte[] decryptedKey = details.Request.DataKey == null ? null : AwsEncryption.DecryptKey(Convert.FromBase64String(details.Request.DataKey), Constants.AppConfig.Application.KmsArn);

                    details.Channels = new List<RequestChannel>();
                    if (getChannels)
                    {
                        command.CommandText = $"select * from {Constants.DatabaseSchema}microservice_twitch_request_channel where request_id = @request_id and status != '{RequestChannelStatus.Deleted}'";
                        using (var wrappedreader = new DataReaderWithMeasurements(command, null, "AmpQueryGetRequestChannels"))
                        {
                            var reader = wrappedreader.MysqlReader;
                            while (reader.Read())
                            {
                                RequestChannel channel = new RequestChannel();
                                channel.ChannelID = (int)reader["channel_id"];
                                channel.ChannelLogin = (string)reader["channel_login"];
                                channel.Status = (string)reader["status"];
                                channel.ProfileImage = (string)reader["profile_image"];
                                channel.ContactInfo = (reader["contact_info"] == System.DBNull.Value || decryptedKey == null) ? null : AwsEncryption.DecryptData((string)reader["contact_info"],decryptedKey);
                                channel.Rating = reader["rating"] == System.DBNull.Value ? null : (int?)reader["rating"];
                                channel.Comment = reader["comment"] == System.DBNull.Value ? null : (string)reader["comment"];
                                channel.CreatorStatus = reader["creator_status"] == System.DBNull.Value ? null : (string)reader["creator_status"];
                                channel.ResponseDate = reader["response_date"] == System.DBNull.Value ? null : (DateTime?)reader["response_date"];
                                channel.Priority = reader["priority"] == System.DBNull.Value ? null : (int?)reader["priority"];
                                channel.SortOrder = reader["sort_order"] == System.DBNull.Value ? null : (int?)reader["sort_order"];
                                channel.NominationReason = reader["nomination_reason"] == DBNull.Value ? null : (string)reader["nomination_reason"];
                                channel.DeclinedReason = reader["declined_reason"] == DBNull.Value ? null : (string)reader["declined_reason"];

                                details.Channels.Add(channel);
                            }
                        }
                    }
                    
                    details.AuthorizedUsers = new List<RequestUser>();
                    command.CommandText = $"select * from {Constants.DatabaseSchema}microservice_twitch_request_authorized_users where request_id = @request_id";
                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "AmpQueryGetRequestAuthorizedUsers"))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            details.AuthorizedUsers.Add(new RequestUser() { UserID = (string)reader["user_id"], UserName = (string)reader["user_name"] });
                        }
                    }
                }
            }
            return details;
        }

        public static Request ReadRequest(MySqlDataReader reader)
        {
            Request request = new Request();
            request.ID = (string)reader["request_id"];
            request.CreatedBy = (string)reader["created_by"];
            request.CreatedDate = (DateTime)reader["created_date"];
            request.Status = (string)reader["status"];
            object participantCount = reader["participant_count"];
            if (participantCount != System.DBNull.Value)
            {
                request.ParticipantCount = (int)participantCount;
            }
            object type = reader["type"];
            if (type != System.DBNull.Value)
            {
                request.Type = (string)type;
            }
            object dueDate = reader["due_date"];
            if (dueDate != System.DBNull.Value)
            {
                request.DueDate = (DateTime)dueDate;
            }
            object name = reader["name"];
            if (name != System.DBNull.Value)
            {
                request.Name = (string)name;
            }
            object description = reader["description"];
            if (description != System.DBNull.Value)
            {
                request.Description = (string)description;
            }
            object publicStatement = reader["public_statement"];
            if (publicStatement != System.DBNull.Value)
            {
                request.PublicStatement = (string)publicStatement;
            }
            object contactMethod = reader["contact_method"];
            if (contactMethod != System.DBNull.Value)
            {
                request.ContactMethod = (string)contactMethod;
            }
            object region = reader["region"];
            if (region != System.DBNull.Value)
            {
                string regions = (string)region;
                request.Region = regions.Split(',');
            }
            else
            {
                request.Region = new string[] { };
            }
            object language = reader["language"];
            if (language != System.DBNull.Value)
            {
                request.Language = (string)language;
            }
            object country = reader["country"];
            if (country != System.DBNull.Value)
            {
                string countries = (string)country;
                request.Country = countries.Split(',');
            }
            else
            {
                request.Country = new string[] { };
            }
            object privacy = reader["privacy"];
            if (privacy != System.DBNull.Value)
            {
                request.RequestPrivacy = (string)privacy;
            }
            object timeRequirementType = reader["time_requirement_type"];
            if (timeRequirementType != System.DBNull.Value)
            {
                request.TimeRequiredType = (string)timeRequirementType;
            }
            object timeRequirementCount = reader["time_requirement_count"];
            if (timeRequirementCount != System.DBNull.Value)
            {
                request.TimeRequiredCount = (float)timeRequirementCount;
            }
            request.EditID = reader["edit_request_id"] == System.DBNull.Value ? null : (string)reader["edit_request_id"];
            request.EngagementType = reader["engagement_type"] == System.DBNull.Value ? null : (string)reader["engagement_type"];
            request.EngagementDateMin = reader["engagement_date_min"] == System.DBNull.Value ? null : (DateTime?)reader["engagement_date_min"];
            request.EngagementDateMax = reader["engagement_date_max"] == System.DBNull.Value ? null : (DateTime?)reader["engagement_date_max"];
            request.PaidOpportunity = reader["paid_opportunity"] == System.DBNull.Value ? null : (bool?)reader["paid_opportunity"];
            request.Flood = reader["flood"] == System.DBNull.Value ? null : (bool?)((bool)reader["flood"]);
            request.AllowAMSuggestions = reader["allow_am_suggestions"] == System.DBNull.Value ? false : (bool)((bool)reader["allow_am_suggestions"]);
            request.RevisedBy = reader["revised_by"] == System.DBNull.Value ? null : (string)reader["revised_by"];
            request.DataKey = reader["data_key"] == System.DBNull.Value ? null : (string)reader["data_key"];
            object filters = reader["listing_filters"];
            request.Filters = filters == System.DBNull.Value ? null : JsonConvert.DeserializeObject<ListingFilter>((string)filters);

            return request;
        }

        public static IEnumerable<Request> GetRequestsByProfile(string profileName, string statusToCheck = null)
        {
            List<Request> requests = new List<Request>();
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = $"select * from {Constants.DatabaseSchema}microservice_twitch_request where created_by = @created_by and edit_request_id is null";
                    if(statusToCheck != null)
                    {
                        command.CommandText += " and status = @status";
                        command.Parameters.AddWithValue("status", statusToCheck);
                    }
                    else
                    {
                        command.CommandText += $" and status in ('{RequestStatus.PendingApproval}','{RequestStatus.Created}')";
                    }
                    command.Parameters.AddWithValue("created_by", profileName);
                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "AmpQueryGetRequests"))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            Request request = ReadRequest(reader);
                            requests.Add(request);
                        }
                    }
                }
            }
            return requests;
        }

        public static IEnumerable<Request> GetRequests(string statusToCheck = null)
        {
            List<Request> requests = new List<Request>();
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = $"select * from {Constants.DatabaseSchema}microservice_twitch_request where edit_request_id is null";
                    if (statusToCheck != null)
                    {
                        command.CommandText += " and status = @status";
                        command.Parameters.AddWithValue("status", statusToCheck);
                    }
                    else
                    {
                        command.CommandText += $" and status in ('{RequestStatus.PendingApproval}','{RequestStatus.Created}')";
                    }
                    using (var wrappedreader = new DataReaderWithMeasurements(command, null,  "AmpQueryGetRequests"))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            Request request = ReadRequest(reader);
                            requests.Add(request);
                        }
                    }
                }
            }
            return requests;
        }

        public static UserListingOverviewModel GetUserListingOverview(ref ListingFilter filter)
        {
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                UserListingOverviewModel model = new UserListingOverviewModel();
                filter.Columns = new[] { "channelid", "maxccu", "channelconcurrentsminuteswatchedtotal", "channelconcurrentsminutesbroadcasttotal", "uniquedaysbroadcast", "topgamebroadcast", "ccutier", "region", "averageccu" };

                var userListingQueryInfo = GetUserListingFilter(ref filter);                

                Stopwatch watch = new Stopwatch();
                watch.Start();

                using (var command = conn.GetCommand())
                {
                    string finalSql =
                    $@"
                        set session transaction isolation level read uncommitted;
                        drop temporary table if exists user_listing_population;
                        create temporary table user_listing_population(channelid bigint,maxccu bigint, channelconcurrentsminuteswatchedtotal bigint, channelconcurrentsminutesbroadcasttotal bigint, uniquedaysbroadcast int, topgamebroadcast varchar(50), ccutier varchar(50), region varchar(5), averageccu float8);
                        insert into user_listing_population {userListingQueryInfo.Item1}
                        limit {ChannelLimit};
                        ";

                    command.CommandText = finalSql;
                    if (userListingQueryInfo.Item2 != null && userListingQueryInfo.Item2.Count > 0)
                    {
                        foreach (var param in userListingQueryInfo?.Item2)
                        {
                            command.Parameters.AddWithValue(param.Key, param.Value);
                        }
                    }
                    command.ExecuteNonQueryWithMeasurements("GetUserListingOverview");

                    command.CommandText =
                    $@"
                        select 
                            cast(coalesce(max(maxccu), 0) as signed integer) as MaxCCU, 
                            cast(coalesce(avg(averageccu), 0.0) as decimal) as AverageCCU, 
                            count(*) as TotalChannels, 
                            cast(coalesce(sum(channelconcurrentsminuteswatchedtotal), 0) as unsigned integer) as TotalMinutesWatched, 
                            cast(coalesce(avg(channelconcurrentsminuteswatchedtotal), 0.0) as decimal) as AverageMinutesWatched, 
                            cast(coalesce(sum(channelconcurrentsminutesbroadcasttotal), 0) as unsigned integer) as TotalMinutesBroadcast, 
                            cast(coalesce(avg(channelconcurrentsminutesbroadcasttotal), 0.0) as decimal) as AverageMinutesBroadcast, 
                            cast(coalesce(avg(uniquedaysbroadcast), 0.0) as decimal) as AverageDaysBroadcast
                        from user_listing_population
                    ";
                    
                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "AmpQueryGetUserListingOverviewQ1"))
                    {
                        var reader = wrappedreader.MysqlReader;
                        if (reader.Read())
                        {
                            model.MaxCCU = (long)reader["MaxCCU"];
                            model.TotalChannels = (int)(long)reader["TotalChannels"];
                            model.TotalHoursBroadcast = ((UInt64)reader["TotalMinutesBroadcast"] / 60.0f);
                            model.AverageHoursBroadcast = ((float)(decimal)reader["AverageMinutesBroadcast"] / 60.0f);
                            model.TotalHoursWatched = ((UInt64)reader["TotalMinutesWatched"] / 60.0f);
                            model.AverageHoursWatched = ((float)(decimal)reader["AverageMinutesWatched"] / 60.0f);
                            model.AverageDaysBroadcast = (int)(float)(decimal)reader["AverageDaysBroadcast"];
                            model.AverageCCU = (float)(decimal)reader["AverageCCU"];
                        }
                    }
                    command.CommandText = 
                    @"
                        select count(*) as Count, topgamebroadcast as Game 
                        from user_listing_population 
                        group by 2 
                        order by 1 desc 
                        limit 0,5
                    ";
                    model.ChannelsPerGame = new List<dynamic>();
                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "AmpQueryGetUserListingOverviewQ2"))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            int count = (int)(long)reader["Count"];
                            model.ChannelsPerGame.Add(new { count = count, game = (string)reader["Game"], percent = (count / (model.TotalChannels * 1.0f)) * 100.0f });
                        }
                    }
                    command.CommandText = 
                    @"
                        select count(*) as Count, region as Region 
                        from user_listing_population 
                        where Region is not null 
                        group by 2 
                        order by 1 desc 
                        limit 0,5
                    ";
                    model.ChannelsPerRegion = new List<dynamic>();
                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "AmpQueryGetUserListingOverviewQ3"))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            int count = (int)(long)reader["Count"];
                            model.ChannelsPerRegion.Add(new { count = (int)(long)reader["Count"], region = (string)reader["Region"], percent = (count / (model.TotalChannels * 1.0f)) * 100.0f });
                        }
                    }

                    command.CommandText =
                    @"
                        select count(*) as Count, ccutier as Tier 
                        from user_listing_population 
                        group by 2 
                        order by 1 desc
                    ";
                    model.ChannelsPerTier = new List<dynamic>();

                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "AmpQueryGetUserListingOverviewQ4"))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            int count = (int)(long)reader["Count"];
                            model.ChannelsPerTier.Add(new { count = (int)(long)reader["Count"], ccutier = (string)reader["Tier"], percent = (count / (model.TotalChannels * 1.0f)) * 100.0f });
                        }
                    }
                }
                watch.Stop();
                Log.Verbose($"Took {watch.ElapsedMilliseconds} MS");
                
                return model;
            }
            
        }

        internal static Tuple<string, Dictionary<string, dynamic>> GetUserListingFilter(ref ListingFilter filter, string overrideColumns = null)
        {
            if (filter == null)
            {
                throw new ArgumentNullException("filter");
            }

            if (filter.Limit <= 0 || filter.Page < 0 || filter.AggregateType == Constants.AggregationType.Unknown)
            {
                throw new ArgumentOutOfRangeException("filter");
            }

            var columns = string.Join(",", filter.Columns.Intersect(TwitchUserListingColumnDefinition.Default()));
            if (string.IsNullOrWhiteSpace(columns))
            {
                throw new ArgumentOutOfRangeException("filter.Columns");
            }

            var validAggregations = new Constants.AggregationType[5]
            {
                    Constants.AggregationType.Week,
                    Constants.AggregationType.ThirtyDay,
                    Constants.AggregationType.SixtyDay,
                    Constants.AggregationType.NintyDay,
                    Constants.AggregationType.BankerYear
            };
            if (!validAggregations.Contains(filter.AggregateType))
            {
                throw new ArgumentOutOfRangeException("filter.AggregationType");
            }

            StringBuilder queryFilter = new StringBuilder();
            Dictionary<string, dynamic> param = null;
            queryFilter.AppendLine($"where AggregationType = {(int)filter.AggregateType}");
            var validColumns = TwitchUserListingColumnDefinition.Default();

            if (filter.QueryFilters != null && filter.QueryFilters.Count() > 0)
            {
                param = new Dictionary<string, dynamic>();
                var dataTypes = new ResonanceTrackedClassModel(typeof(TwitchUserListingModel));
                foreach (var qf in filter.QueryFilters.Where(x => x != null && !string.IsNullOrWhiteSpace(x.Key)))
                {
                    if (validColumns.Contains(qf.Key))
                    {
                        if (qf.FilterType == FilterType.Range)
                        {
                            if (qf.Value != null && qf.Value2 == null)
                            {
                                qf.FilterType = FilterType.GreaterThan;
                            }
                            if (qf.Value == null && qf.Value2 != null)
                            {
                                qf.FilterType = FilterType.LessThan;
                                qf.Value = qf.Value2;
                                qf.Value2 = null;
                            }
                        }

                        var validDataTypes = dataTypes.Fields.Where(x => x.Name.ToLower() == qf.Key.ToLower()).Select(x => x.ValidFilters).FirstOrDefault();
                        if (validDataTypes != null && validDataTypes.Length > 0)
                        {
                            if (validDataTypes.Any(x => x.HasFlag(qf.FilterType)))
                            {
                                var queryFilterResult = FilterHelperMysql.ProcessFilterQuery(qf);
                                queryFilter.AppendLine(queryFilterResult.SqlSnippet);
                                foreach (var p in queryFilterResult?.Parameters)
                                {
                                    param.Add(p.Key, p.Value);
                                }
                            }
                        }
                        
                    }
                    else
                    {
                        Log.Verbose($@"Skipping invalid query filter: {qf.Key}-{qf.Value}");
                    }
                }
            }

            // Filter name, Valid column field list
            var validExternalQueries = new Dictionary<string, List<string>>()
            {
                { "game-name", new List<string>() { "gamename" } },
                { "twitch-game-id", new List<string>() { "gameid" } },
                { "giantbomb-id", new List<string>() { "giantbombid" } },
                
            };

            if(filter.ExternalQueryFilters != null && filter.ExternalQueryFilters.Count() > 0)
            {
                param.Add("@externalParamAggregation", (int)filter.AggregateType);

                foreach (var eqf in filter.ExternalQueryFilters)
                {
                    if (validExternalQueries.Keys.Contains(eqf.Key))
                    {

                        var externalQueryFilter = new StringBuilder();
                        externalQueryFilter.AppendLine
                        ($@"and exists
                            (
	                            select 1
                                from {Constants.DatabaseSchema}microservice_twitch_user_game_stats
                                where 
		                            microservice_twitch_user_listing_past_{(int)filter.AggregateType}_days.ChannelID = microservice_twitch_user_game_stats.ChannelID
                                    and microservice_twitch_user_game_stats.AggregationType = @externalParamAggregation");

                        foreach(var eqfFilter in eqf.Value)
                        {
                            foreach(var eqfQueryFilter in eqfFilter.QueryFilters)
                            {
                                if (validExternalQueries[eqf.Key].Contains(eqfQueryFilter.Key))
                                {
                                    var externalQueryFilterResult = FilterHelperMysql.ProcessFilterQuery(eqfQueryFilter);
                                    externalQueryFilter.AppendLine(externalQueryFilterResult.SqlSnippet);
                                    foreach (var p in externalQueryFilterResult?.Parameters)
                                    {
                                        param.Add(p.Key, p.Value);
                                    }
                                }
                                else
                                {
                                    Log.Warn($@"Skipping invalid external query filter column {eqfQueryFilter.Key} :: {eqfQueryFilter.Value}");
                                }
                            }
                        }

                        externalQueryFilter.AppendLine($@")");
                        queryFilter.AppendLine(externalQueryFilter.ToString());
                    }
                    else
                    {
                        Log.Warn($@"Skipping invalid external query filter: {eqf.Key} :: {eqf.Value}");
                    }
                }
            }

            StringBuilder sort = new StringBuilder();
            if (filter != null && filter.SortOrder != null && filter.SortOrder.Length > 0)
            {
                bool isFirst = true;
                foreach (var sortitem in filter.SortOrder.OrderBy(x => x.Ordinal).ToArray())
                {
                    if (validColumns.Contains(sortitem.Column))
                    {
                        var asckey = sortitem.Ascending ? "ASC" : "DESC";
                        string first = isFirst ? "order by " : ",";
                        sort.AppendLine($@"{first}{sortitem.Column} {asckey}");
                        isFirst = false;
                    }
                }
            }

            var baseSql = 
            $@"
                select {overrideColumns ?? columns}
                from {Constants.DatabaseSchema}microservice_twitch_user_listing_past_{(int)filter.AggregateType}_days
                {queryFilter.ToString()}
                {sort.ToString()}
            ";
            return new Tuple<string, Dictionary<string, dynamic>>(baseSql, param);
        }

        internal static IEnumerable<dynamic> GetUserListing(HttpContext context, ref ListingFilter filter)
        {
            IEnumerable<dynamic> result = null;
            try
            {
                var userListingQuery = GetUserListingFilter(ref filter);
                var param = userListingQuery.Item2;
                string baseSql = userListingQuery.Item1;

                baseSql = $@"select tbl.* from ({baseSql} limit {ChannelLimit}) tbl";

                var sql = baseSql +
                $@"
                    limit {filter.Limit}
                    offset {filter.Page * filter.Limit}
                    ;
                ";

                result = DBManagerMysql.GetDynamicSqlData(context, sql, "get_user_listing", parameters: param);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        internal static bool? RequestHasRevision(string requestID)
        {
            bool? hasRevision = null;

            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandText = $"select request_id from {Constants.DatabaseSchema}microservice_twitch_request where edit_request_id=@request_id";
                        command.Parameters.AddWithValue("request_id", requestID);

                        using (var reader = new DataReaderWithMeasurements(command, null, "request_has_revision").MysqlReader)
                        {
                            if (reader.Read())
                            {
                                hasRevision = true;
                            }
                            else
                            {
                                hasRevision = false;
                            }
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }

            return hasRevision;
        }

        internal static bool? RequestIsRevision(string requestID)
        {
            bool? isRevision = null;

            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandText = $"select request_id from {Constants.DatabaseSchema}microservice_twitch_request where request_id=@request_id and edit_request_id is not null";
                        command.Parameters.AddWithValue("request_id", requestID);

                        using (var reader = new DataReaderWithMeasurements(command, null, "request_is_revision").MysqlReader)
                        {
                            if (reader.Read())
                            {
                                isRevision = true;
                            }
                            else
                            {
                                isRevision = false;
                            }
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }

            return isRevision;
        }
    }
}
