﻿using Resonance.Core;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Models.ServiceModels.AtlasModels;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Text;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using Resonance.Core.Models.ServiceModels.ExplorerModels;
using static Resonance.Core.Constants;
using Resonance.Core.Helpers.FilterHelpers;
using Resonance.Core.Models.ApiModels.AtlasModels;
using Resonance.Core.Helpers.ApiHelpers;
using Newtonsoft.Json;
using System.Diagnostics;
using Resonance.Core.Models.ApiModels;
using System.Collections.Concurrent;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models.ApiModels.TwitchModels;
using MySql.Data.MySqlClient;
using Resonance.Microservices.ListingHelpers;
using Microsoft.AspNetCore.Http;
using Resonance.Core.Helpers.AuthHelpers;
using Resonance.Core.Helpers.AwsHelpers;

namespace Resonance.Microservices.Queries
{
    /// <summary>
    /// TODO: NOTE FROM TOURNYMASTERBOT: We need to lift AtlasQuery so that Initialize takes in context / metrics / auth user tokens
    /// </summary>
    internal class AtlasQuery
    {
        internal void Initialize()
        {

        }

        public IEnumerable<string> AutocompleteGames(HttpContext context, string name)
        {
            var stopwatch = new Stopwatch();
            List<string> gameNames = new List<string>();
            try
            {
                stopwatch.Start();
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandText = $"select game_name from {Constants.DatabaseSchema}microservice_twitch_atlas_game_names where game_name like @game order by game_name asc limit 20";
                        command.Parameters.AddWithValue("@game", $"%{name}%");
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_autocomplete_games").MysqlReader)
                        {
                            while (reader.Read())
                            {
                                string gameName = (string)reader["game_name"];
                                gameNames.Add(gameName);
                            }
                        }
                    }
                }
            }
            catch (Exception)
            {

            }
            finally
            {
                stopwatch.Stop();
            }
            return gameNames;
        }

        internal IEnumerable<AtlasAutoCompleteResult> AutocompleteProducts(HttpContext context, string name, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var stopwatch = new Stopwatch();
            List<AtlasAutoCompleteResult> products = new List<AtlasAutoCompleteResult>();
            try
            {
                stopwatch.Start();
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.CreateCommand())
                    {
                        command.Parameters.AddWithValue("@productName", $"%{name}%");
                        command.CommandText = $"select product_id, product_name from {Constants.DatabaseSchema}microservice_twitch_atlas_product where product_name like @productName order by locate(@productName, product_name), length(product_name), product_name limit 0,10;";
                        using (var wrappedreader = new DataReaderWithMeasurements(command, context, "AtlasQueryAutocompleteProducts"))
                        {
                            var reader = wrappedreader.MysqlReader;
                            while (reader.Read())
                            {
                                string productName = (string)reader["product_name"];
                                int productID = (int)reader["product_id"];
                                products.Add(new AtlasAutoCompleteResult() { ID = productID, Name = productName });
                            }
                        }
                    }
                }
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "auto_complete_product",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }

            return products;
        }

        internal List<int> GetPremiumContentCreatorIDsFromQuery(HttpContext context, string query, Dictionary<string, dynamic> paramList)
        {
            var pccIDs = new List<int>();
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText = query;
                        foreach(var param in paramList)
                        {
                            command.Parameters.AddWithValue($"@{param.Key}", param.Value);
                        }
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_pcc_id_from_query").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    var premium_content_creator_id = reader.GetInt32(0);
                                    pccIDs.Add(premium_content_creator_id);
                                }
                            }
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                pccIDs = null;
                Log.Error(ex);
            }
            return pccIDs;
        }

        internal AtlasListing GetListing(HttpContext context, ref ListingFilter filter, ConcurrentBag<ElapsedTimeModel> metrics, string customSelect = null, string customCountSelect = null, string extraMetricName = null, string overrideCountColumn = null, string groupBy = null)
        {
            var stopwatch = new Stopwatch();
            AtlasListing listing = new AtlasListing();
            try
            {
                stopwatch.Start();
                if (filter == null)
                {
                    throw new ArgumentNullException("filter");
                }

                if (filter.Limit <= 0 || filter.Page < 0 || filter.EventType == Constants.AtlasInternalEventType.Unknown)
                {
                    throw new ArgumentOutOfRangeException("filter");
                }

                try
                {
                    string sql = "";
                    string countSql = "";
                    Dictionary<string, dynamic> param = new Dictionary<string, dynamic>();
                    if (customSelect != null)
                    {
                        sql = customSelect +
                        $@"
                            limit {filter.Limit}
                            offset {filter.Page * filter.Limit}
                        ";
                        countSql = customCountSelect;
                    }
                    else
                    {

                        var query = Filter(context, ref filter, metrics, useAlternateFieldNames: true, overrideCountColumn: overrideCountColumn, groupBy: groupBy);
                        param = query.Item2;
                        countSql = query.Item3;
                        string baseSql = query.Item1;

                        sql = baseSql +
                        $@"
                            limit {filter.Limit}
                            offset {filter.Page * filter.Limit}
                            ;
                        ";
                    }
                    using (var conn = DBManagerMysql.GetConnection())
                    {
                        using (var command = conn.CreateCommand())
                        {
                            command.CommandText = countSql;
                            if (param != null && param.Count > 0)
                            {
                                foreach (var parameter in param)
                                {
                                    command.Parameters.AddWithValue(parameter.Key, parameter.Value);
                                }
                            }
                            listing.TotalPages = (long)Math.Ceiling((command.ExecuteScalarWithMeasurements<long>($"get_listing_count{extraMetricName ?? ""}") * 1.0) / filter.Limit);
                        }
                    }
                    listing.Items = DBManagerMysql.GetDynamicSqlData(context, sql, $"get_listing{extraMetricName ?? ""}", parameters: param);
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
            }
            finally
            {
                stopwatch.Stop();
                if (metrics != null)
                {
                    if (!string.IsNullOrWhiteSpace(extraMetricName))
                    {
                        extraMetricName = $@"_{extraMetricName}";
                    }
                    metrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = $"get_listing{extraMetricName ?? ""}",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }
            return listing;
        }

        internal Tuple<string, Dictionary<string, dynamic>, string> Filter(HttpContext context, ref ListingFilter filter, ConcurrentBag<ElapsedTimeModel> metrics, string overrideColumns = null, bool useAlternateFieldNames = false, string extraMetricName = null, string groupBy = null, string overrideCountColumn = null)
        {
            var stopwatch = new Stopwatch();

            try
            {
                stopwatch.Start();
                if (filter == null)
                {
                    throw new ArgumentNullException("filter");
                }

                if (filter.Limit <= 0 || filter.Page < 0 || filter.AggregateType == Constants.AggregationType.Unknown)
                {
                    throw new ArgumentOutOfRangeException("filter");
                }

                string[] checkColumns = null;
                StringBuilder queryFilter = new StringBuilder();
                Dictionary<string, dynamic> param = null;
                if (filter.ActiveOnly)
                {
                    queryFilter.AppendLine($"where is_active=true");
                }
                else
                {
                    queryFilter.AppendLine($"where 1=1");
                }
                string[] validColumns = null;
                ResonanceTrackedClassModel dataTypes = null;
                string table = null;

                switch (filter.EventType)
                {
                    case AtlasInternalEventType.Event:
                        {
                            table = "vw_atlas_event_details";
                            validColumns = AtlasListingDefaults.EventDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(AtlasEvent));
                            checkColumns = AtlasListingDefaults.EventDefault();
                            break;
                        }
                    case AtlasInternalEventType.Product:
                        {
                            table = "vw_atlas_product_details";
                            validColumns = AtlasListingDefaults.ProductDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(ViewAtlasProduct));
                            checkColumns = AtlasListingDefaults.ProductDefault();
                            break;
                        }
                    case AtlasInternalEventType.Season:
                        {
                            table = "vw_atlas_season_details";
                            validColumns = AtlasListingDefaults.SeasonDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(ViewAtlasSeason));
                            checkColumns = AtlasListingDefaults.SeasonDefault();
                            break;
                        }
                    case AtlasInternalEventType.Stream:
                        {
                            table = "microservice_twitch_atlas_stream";
                            validColumns = AtlasListingDefaults.StreamDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(AtlasStream));
                            checkColumns = AtlasListingDefaults.StreamDefault();
                            break;
                        }
                    case AtlasInternalEventType.AccountManager:
                        {
                            table = "microservice_twitch_atlas_account_manager";
                            validColumns = AtlasListingDefaults.AccountManagerDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(AtlasAccountManager));
                            checkColumns = AtlasListingDefaults.AccountManagerDefault();
                            break;
                        }
                    case AtlasInternalEventType.PremiumContentCreator:
                        {
                            table = "vw_atlas_premium_content_creator";
                            validColumns = AtlasListingDefaults.PremiumContentCreatorDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(ViewPremiumContentCreator));
                            checkColumns = AtlasListingDefaults.PremiumContentCreatorDefault();
                            break;
                        }
                    case AtlasInternalEventType.PccToAmMap:
                        {
                            table = "vw_atlas_pcc_to_am_map";
                            validColumns = AtlasListingDefaults.PccToAmMapDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(ViewPremiumContentCreatorToAccountManagerMap));
                            checkColumns = AtlasListingDefaults.PccToAmMapDefault();
                            break;
                        }
                    case AtlasInternalEventType.PccToChannelMap:
                        {
                            table = "vw_atlas_pcc_to_channel_map";
                            validColumns = AtlasListingDefaults.PccToChannelMapDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(ViewPremiumContentCreatorToChannelMap));
                            checkColumns = AtlasListingDefaults.PccToChannelMapDefault();

                            break;
                        }
                    case AtlasInternalEventType.Contract:
                        {
                            table = "microservice_twitch_atlas_contracts";
                            validColumns = AtlasListingDefaults.ContractDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(AtlasContract));
                            checkColumns = AtlasListingDefaults.ContractDefault();

                            break;
                        }
                    case AtlasInternalEventType.ContractChannel:
                        {
                            table = "microservice_twitch_atlas_contract_channels";
                            validColumns = AtlasListingDefaults.ContractChannelDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(AtlasContractChannel));
                            checkColumns = AtlasListingDefaults.ContractChannelDefault();

                            break;
                        }
                    case AtlasInternalEventType.ContractAccountManagerMap:
                        {
                            table = "vw_atlas_contract_account_manager_details";
                            validColumns = AtlasListingDefaults.ContractAccountManagerMapDefault();
                            dataTypes = new ResonanceTrackedClassModel(typeof(ViewAtlasContractAccountManager));
                            checkColumns = AtlasListingDefaults.ContractAccountManagerMapDefault();

                            break;
                        }
                    default:
                        {
                            throw new NotImplementedException($@"{filter.EventType} is not implemented in Atlas Query");
                        }
                }

                var columns = string.Join(",", filter.Columns.Intersect(checkColumns));
                if (string.IsNullOrWhiteSpace(columns) && overrideColumns == null)
                {
                    throw new ArgumentOutOfRangeException("filter.Columns");
                }

                var validAggregations = new Constants.AtlasInternalEventType[11]
                {
                    Constants.AtlasInternalEventType.Event,
                    Constants.AtlasInternalEventType.Product,
                    Constants.AtlasInternalEventType.Season,
                    Constants.AtlasInternalEventType.Stream,
                    Constants.AtlasInternalEventType.AccountManager,
                    Constants.AtlasInternalEventType.PremiumContentCreator,
                    Constants.AtlasInternalEventType.PccToAmMap,
                    Constants.AtlasInternalEventType.PccToChannelMap,
                    Constants.AtlasInternalEventType.Contract,
                    Constants.AtlasInternalEventType.ContractChannel,
                    Constants.AtlasInternalEventType.ContractAccountManagerMap
                };
                if (!validAggregations.Contains(filter.EventType))
                {
                    throw new ArgumentOutOfRangeException("filter.EventType");
                }


                if (filter.QueryFilters != null && filter.QueryFilters.Count() > 0)
                {
                    param = new Dictionary<string, dynamic>();
                    foreach (var qf in filter.QueryFilters.Where(x => x != null && !string.IsNullOrWhiteSpace(x.Key)))
                    {
                        if (validColumns.Contains(qf.Key))
                        {
                            if (qf.FilterType == FilterType.Range)
                            {
                                if ((qf.Value != null && qf.Value != "") && (qf.Value2 == null || qf.Value2 == ""))
                                {
                                    qf.FilterType = FilterType.GreaterThan;
                                }
                                if ((qf.Value == null || qf.Value == "") && (qf.Value2 != null && qf.Value2 != ""))
                                {
                                    qf.FilterType = FilterType.LessThan;
                                    qf.Value = qf.Value2;
                                    qf.Value2 = null;
                                }
                            }

                            var validFields = dataTypes.Fields.Where(x => x.Name.ToLower() == qf.Key.ToLower()).Select(x => x.ValidFilters).FirstOrDefault();
                            // If the class doesn't return any data, check the name from the custom attribute to see if it yields any results
                            if (validFields == null || validFields.Length == 0 && useAlternateFieldNames)
                            {
                                try
                                {
                                    var fields = new Dictionary<string, ResonanceFieldModel>();
                                    foreach(var field in dataTypes.Fields)
                                    {
                                        foreach(var customAttribute in field.CustomAttributes)
                                        {
                                            foreach(var namedAttribute in customAttribute.NamedArguments)
                                            {
                                                switch (namedAttribute.MemberName)
                                                {
                                                    // Newtonsoft Json custom name
                                                    case "PropertyName":
                                                    {
                                                        if (namedAttribute.TypedValue.Value.ToString() == qf.Key.ToLower())
                                                        {
                                                            if (!fields.ContainsKey(namedAttribute.TypedValue.Value.ToString()))
                                                            {
                                                                fields.Add(namedAttribute.TypedValue.Value.ToString(), field);
                                                            }
                                                        }
                                                        break;
                                                    }
                                                }
                                            }
                                            foreach(var namedAttribute in customAttribute.ConstructorArguments)
                                            {
                                                if(namedAttribute.Value.ToString() == qf.Key.ToLower())
                                                {
                                                    if (!fields.ContainsKey(namedAttribute.Value.ToString()))
                                                    {
                                                        fields.Add(namedAttribute.Value.ToString(), field);
                                                    }
                                                }
                                            }
                                        }
                                    }
                                    validFields = fields.Values.Select(x => x.ValidFilters).FirstOrDefault();
                                }
                                catch (Exception) { }
                            }

                            if (validFields != null && validFields.Length > 0)
                            {
                                if (validFields.Any(x => x.HasFlag(qf.FilterType)))
                                {
                                    var queryFilterResult = FilterHelperMysql.ProcessFilterQuery(qf);
                                    queryFilter.AppendLine(queryFilterResult.SqlSnippet);
                                    foreach (var p in queryFilterResult?.Parameters)
                                    {
                                        param.Add(p.Key, p.Value);
                                    }
                                }
                            }
                        }
                        else if(qf.Key == "channels" && table == "vw_atlas_event_details" && qf.FilterType == FilterType.In)
                        {
                            long[] channelIDs = qf.ValueArray.Select(x => (long)x).ToArray();
                            queryFilter.AppendLine
                            ($@"and exists
                                (
                                    select 1
                                    from {Constants.DatabaseSchema}microservice_twitch_atlas_stream
                                    where 
                                        {table}.event_id = microservice_twitch_atlas_stream.event_id
                                        and microservice_twitch_atlas_stream.channel_id in ({string.Join(",", channelIDs)})
                                )
                            ");
                        }
                        else
                        {
                            Log.Verbose($@"Skipping invalid query filter: {qf.Key}-{qf.Value}");
                        }
                    }
                }

                StringBuilder sort = new StringBuilder();
                if (filter != null && filter.SortOrder != null && filter.SortOrder.Length > 0)
                {
                    bool isFirst = true;
                    foreach (var sortitem in filter.SortOrder.OrderBy(x => x.Ordinal).ToArray())
                    {
                        if (validColumns.Contains(sortitem.Column))
                        {
                            var asckey = sortitem.Ascending ? "ASC" : "DESC";
                            string first = isFirst ? "order by " : ",";
                            sort.AppendLine($@"{first}{sortitem.Column} {asckey}");
                            isFirst = false;
                        }
                    }
                }

                var baseSql =
                $@"
                    select {overrideColumns ?? columns}
                    from {Constants.DatabaseSchema}{table}
                    {queryFilter.ToString()}
                    {sort.ToString()}
                    {(groupBy != null ? $"group by {groupBy}" : "")}
                ";

                var countSql =
                $@"
                    select {overrideCountColumn ?? "count(*)"}
                    from {Constants.DatabaseSchema}{table}
                    {queryFilter.ToString()}
                    {(groupBy != null ? $"group by {groupBy}" : "")}
                ";
                return new Tuple<string, Dictionary<string, dynamic>, string>(baseSql, param, countSql);
            }
            finally
            {
                stopwatch.Stop();
                if (!string.IsNullOrWhiteSpace(extraMetricName))
                {
                    extraMetricName = $@"_{extraMetricName}";
                }

                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = $"generate_filter{extraMetricName ?? ""}",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
        }

        internal int GetPremiumContentCreatorIDByName(HttpContext context, string premiumContentCreatorName)
        {
            int pccID = 0;
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText = $@"select premium_content_Creator_id from {Constants.DatabaseSchema}microservice_twitch_atlas_premium_content_creator where premium_content_creator_name = @pccName;";
                        command.Parameters.AddWithValue($@"pccName", premiumContentCreatorName);
                        pccID = Convert.ToInt32(command.ExecuteScalar());
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            return pccID;
        }

        /// <summary>
        /// Null represents an error scenario
        /// </summary>
        internal bool? DoesPccExist(HttpContext context, int premiumContentCreatorID, string pccName)
        {
            bool? pccExists = null;

            try
            {
                if(premiumContentCreatorID <= 0 && string.IsNullOrWhiteSpace(pccName))
                {
                    return pccExists;
                }
                var whereClause = new StringBuilder();
                whereClause.AppendLine($@"where 1=1 and (");
                if(premiumContentCreatorID > 0)
                {
                    whereClause.AppendLine($@"premium_content_creator_id = @pccID");
                    if (!string.IsNullOrWhiteSpace(pccName))
                    {
                        whereClause.AppendLine($@"or premium_content_creator_name = @pccName");
                    }
                }
                else if(!string.IsNullOrWhiteSpace(pccName))
                {
                    whereClause.AppendLine($@"premium_content_creator_name = @pccName");
                }
                // This should be impossible, but just in case this gets edited in the future
                else
                {
                    return pccExists;
                }
                whereClause.AppendLine($@")");

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText =
                        $@"
                            select case when count(*) > 0 then True else False end as pcc_exists from {Constants.DatabaseSchema}microservice_twitch_atlas_premium_content_creator {whereClause.ToString()};
                        ";
                        command.Parameters.AddWithValue("@pccID", premiumContentCreatorID);
                        command.Parameters.AddWithValue("@pccName", pccName);
                        pccExists = Convert.ToBoolean((long)command.ExecuteScalar());
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
                pccExists = null;
            }
            return pccExists;
        }

        internal List<PremiumContentCreator> GetOwnedCreators(HttpContext context, string ldap, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            List<PremiumContentCreator> creators = new List<PremiumContentCreator>();
            var stopwatch = new Stopwatch();
            try
            {
                using (var conn = DBManagerMysql.GetConnection())
                {
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandText =
                        $@"
                            select distinct(microservice_twitch_atlas_premium_content_creator.premium_content_creator_id), premium_content_creator_name from {Constants.DatabaseSchema}microservice_twitch_atlas_premium_content_creator
                            left join {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_am_map 
                            on microservice_twitch_atlas_premium_content_creator.premium_content_creator_id = microservice_twitch_atlas_pcc_to_am_map.premium_content_creator_id 
                           left join  {Constants.DatabaseSchema}microservice_twitch_atlas_account_manager 
                            on microservice_twitch_atlas_account_manager.account_manager_id = microservice_twitch_atlas_pcc_to_am_map.account_manager_id
                            {(ldap == "*" ? "" : "where account_manager_ldap_name = @ldap")}
                            order by premium_content_creator_name asc
";
                        command.Parameters.AddWithValue("@ldap", ldap);
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_owned_creators").MysqlReader)
                        {
                            while (reader.Read())
                            {
                                PremiumContentCreator creator = new PremiumContentCreator();
                                creator.PremiumContentCreatorName = (string)reader["premium_content_creator_name"];
                                creator.PremiumContentCreatorID = (int)reader["premium_content_creator_id"];
                                creators.Add(creator);

                            }
                        }
                    }
                }


            }
            catch(Exception)
            {

            }
            return creators;
        }

        internal GroupedOwnedChannelList GetOwnedChannels(HttpContext context, int pccID, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var stopwatch = new Stopwatch();
            GroupedOwnedChannelList channelList = new GroupedOwnedChannelList() { GroupedChannels = new List<GroupedOwnedChannel>() };

            try
            {
                stopwatch.Start();
                Dictionary<long, OwnedChannel> channelLookup = new Dictionary<long, OwnedChannel>();
                Dictionary<string, GroupedOwnedChannel> productLookup = new Dictionary<string, GroupedOwnedChannel>();
                using (var conn = DBManagerMysql.GetConnection())
                {
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandText =
                        $@"
                            select
                                product_name, 
                                product_owner,
                                microservice_twitch_atlas_pcc_to_channel_map.channel_id
                            from 
                                {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_channel_map 
                                inner join {Constants.DatabaseSchema}microservice_twitch_atlas_stream
                                    on microservice_twitch_atlas_stream.channel_id = microservice_twitch_atlas_pcc_to_channel_map.channel_id
                                inner join {Constants.DatabaseSchema}microservice_twitch_atlas_event
                                    on microservice_twitch_atlas_event.event_id = microservice_twitch_atlas_stream.event_id 
                                inner join {Constants.DatabaseSchema}microservice_twitch_atlas_product
                                    on microservice_twitch_atlas_product.product_id = microservice_twitch_atlas_event.product_id
                            where microservice_twitch_atlas_pcc_to_channel_map.premium_content_creator_id = @pccID
                                and microservice_twitch_atlas_product.premium_content_creator_id = @pccID
                        ";
                        command.Parameters.AddWithValue("@pccID", pccID);
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_owned_channels").MysqlReader)
                        {
                            while (reader.Read())
                            {
                                string productName = (string)reader["product_name"];
                                string productOwner = (string)reader["product_owner"];
                                long channelID = (long)reader["channel_id"];
                                if (!productLookup.ContainsKey(productName))
                                {
                                    productLookup.Add(productName, new GroupedOwnedChannel() { ProductName = productName, ProductOwner = productOwner, Channels = new List<OwnedChannel>() });
                                }
                                OwnedChannel channel = new OwnedChannel() { ChannelID = channelID, ChannelStatus = "Offline" };
                                if (channelLookup.ContainsKey(channelID))
                                {
                                    channel = channelLookup[channelID];
                                }
                                else
                                {
                                    channelLookup[channelID] = channel;
                                }
                                if (!productLookup[productName].Channels.Any(c => c.ChannelID == channel.ChannelID))
                                {
                                    productLookup[productName].Channels.Add(channel);
                                }
                            }
                        }
                    }
                }
                var livelineResults = TwitchLivelineAPIHelper.GetChannelResponseByChannelIDs(channelLookup.Select(x => x.Key.ToString()));
                if (livelineResults != null && livelineResults.Streams != null)
                {
                    foreach (dynamic stream in livelineResults.Streams)
                    {
                        long? channelID = stream.channel_id;
                        if (channelID.HasValue && channelLookup.ContainsKey(channelID.Value))
                        {
                            int? viewers = stream.viewcount_data.viewcount;
                            string category = stream.channel_data.category_name_cased;
                            channelLookup[channelID.Value].ChannelStatus = "Online";
                            channelLookup[channelID.Value].Viewers = viewers ?? 0;
                            channelLookup[channelID.Value].ChannelCategory = category;
                        }
                    }
                }
                TwitchUsersServiceGetUsersResponse usersServiceResponse = TwitchUsersServiceHelpers.GetUsers(context, channelLookup.Keys);
                if (usersServiceResponse != null && usersServiceResponse.Results != null)
                {
                    foreach (var channel in usersServiceResponse.Results)
                    {
                        long channelID = channel.ID;
                        string channelName = channel.Login;
                        string profileImage = channel.ProfileImageUrl.FormatUsersServiceImageUrl();
                        channelLookup[channelID].ChannelLogin = channelName;
                        channelLookup[channelID].ChannelProfileImage = profileImage;
                    }

                }

                if(productLookup != null)
                {
                    foreach (var product in productLookup)
                    {
                        channelList.GroupedChannels.Add(product.Value);
                    }
                }
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "get_owned_channels_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return channelList;
        }

        internal IEnumerable<TopicListEntry> GetTopicsAndFormats(HttpContext context, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var stopwatch = new Stopwatch();
            Dictionary<int, TopicListEntry> topicLookup = new Dictionary<int, TopicListEntry>();
            Dictionary<int, List<FormatListEntry>> formatLookup = new Dictionary<int, List<FormatListEntry>>();
            try
            {
                stopwatch.Start();
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText =
                        $@"select * from {Constants.DatabaseSchema}microservice_twitch_atlas_topics where is_active = 1";
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_topics").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    int topicID = (int)reader["topic_id"];
                                    string topic = (string)reader["topic_name"];
                                    if (!topicLookup.ContainsKey(topicID))
                                    {
                                        topicLookup.Add(topicID, new TopicListEntry() { TopicID = topicID, Topic = topic, Formats = new List<FormatListEntry>(), SubTopics = new List<string>() });
                                    }
                                }
                            }
                        }

                        command.CommandText =
                        $@"select * from {Constants.DatabaseSchema}microservice_twitch_atlas_sub_topics where is_active = 1";
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_sub_topics").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    int topicID = (int)reader["topic_id"];
                                    string subTopic = (string)reader["sub_topic_name"];
                                    if (topicLookup.ContainsKey(topicID))
                                    {
                                        if (!topicLookup[topicID].SubTopics.Contains(subTopic))
                                        {
                                            topicLookup[topicID].SubTopics.Add(subTopic);
                                        }
                                    }
                                }
                            }
                        }
                        command.CommandText =
                        $@"select * from {Constants.DatabaseSchema}microservice_twitch_atlas_formats where is_active = 1";
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_formats").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    int topicID = (int)reader["topic_id"];
                                    int formatID = (int)reader["format_id"];
                                    string format = (string)reader["format_name"];
                                    if (topicLookup.ContainsKey(topicID))
                                    {
                                        FormatListEntry formatListEntry = new FormatListEntry() { Format = format, FormatID = formatID, SubFormats = new List<string>() };
                                        if (!formatLookup.ContainsKey(formatID))
                                        {
                                            formatLookup.Add(formatID, new List<FormatListEntry>());
                                        }
                                        formatLookup[formatID].Add(formatListEntry);
                                        topicLookup[topicID].Formats.Add(formatListEntry);
                                    }
                                }
                            }
                        }
                        command.CommandText =
                        $@"select * from {Constants.DatabaseSchema}microservice_twitch_atlas_sub_formats where is_active = 1";
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_sub_formats").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    int formatID = (int)reader["format_id"];
                                    string subFormat = (string)reader["sub_format_name"];
                                    if (formatLookup.ContainsKey(formatID))
                                    {
                                        foreach (FormatListEntry format in formatLookup[formatID])
                                        {
                                            format.SubFormats.Add(subFormat);
                                        }
                                    }
                                }
                            }
                        }
                    }

                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "get_topics_formats_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }

            var sorted = new List<TopicListEntry>();
            foreach(var topic in topicLookup.OrderBy(x => x.Value.Topic).ToArray())
            {
                topic.Value.SubTopics = topic.Value.SubTopics.OrderBy(x => x).ToList();
                topic.Value.Formats = topic.Value.Formats.OrderBy(x => x.Format).ToList()
                    .Select(x => new FormatListEntry
                    {
                        FormatID = x.FormatID,
                        Format = x.Format,
                        SubFormats = x.SubFormats.OrderBy(y => y).ToList()
                    }).ToList();
                sorted.Add(topic.Value);
            }

            return sorted;
        }

        internal Tuple<int, Dictionary<int, List<dynamic>>> GetSeasonEventList(IEnumerable<int> seasonIDs, ListingFilter existingFilter, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            Dictionary<int, List<dynamic>> seasonEvents = new Dictionary<int, List<dynamic>>();

            var statuscode = 200;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                ListingFilter eventFilter = new ListingFilter();
                eventFilter.EventType = AtlasInternalEventType.Event;
                eventFilter.Columns = AtlasListingDefaults.EventDefault();
                eventFilter.Limit = 1000;
                eventFilter.Page = 0;
                eventFilter.SortOrder = new SortFilter[] { new SortFilter() { Ordinal = 0, Column = "end_time", Ascending = false } };
                List<QueryFilter> existingFilters = existingFilter.QueryFilters.ToList();
                existingFilters.Add(new QueryFilter() { Key = "season_id", FilterType = FilterType.In, ValueArray = seasonIDs.Select(x => (dynamic)x).ToArray() });
                eventFilter.QueryFilters = existingFilters;
                var eventResults = GetListing(context, ref eventFilter, metrics);
                statuscode = ListingHelper.PostProcessAtlasEventListing(ref eventResults, metrics, context);
                if (eventResults.Items != null)
                {
                    foreach (var result in eventResults.Items)
                    {
                        int season_id = result.season_id;
                        if (!seasonEvents.ContainsKey(season_id))
                        {
                            seasonEvents.Add(season_id, new List<dynamic>());
                        }
                        seasonEvents[season_id].Add(result);
                    }
                }

            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "get_season_event_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return new Tuple<int, Dictionary<int, List<dynamic>>>(statuscode, seasonEvents);
        }

        internal Dictionary<long, List<dynamic>> GetChannelStreamList(HttpContext context, IEnumerable<long> channelIDs, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            Dictionary<long, List<dynamic>> channelEvents = new Dictionary<long, List<dynamic>>();
            ListingFilter eventFilter = new ListingFilter();
            eventFilter.EventType = AtlasInternalEventType.Stream;
            eventFilter.Columns = new[] { "event_id", "channel_id" };
            eventFilter.Limit = 1000;
            eventFilter.Page = 0;
            eventFilter.SortOrder = new SortFilter[] { new SortFilter() { Ordinal = 0, Column = "end_time", Ascending = false } };
            eventFilter.QueryFilters = new[] { new QueryFilter() { Key = "channel_id", FilterType = FilterType.In, ValueArray = channelIDs.Select(x => (dynamic)x).ToArray() } };
            var eventResults = GetListing(context, ref eventFilter, metrics);
            foreach (var result in eventResults.Items)
            {
                long channel_id = result.channel_id;
                if (!channelEvents.ContainsKey(channel_id))
                {
                    channelEvents.Add(channel_id, new List<dynamic>());
                }
                channelEvents[channel_id].Add(result);
            }
            return channelEvents;
        }

        internal Tuple<int, IEnumerable<dynamic>> GetEventListByIDs(IEnumerable<int> eventIDs, ConcurrentBag<ElapsedTimeModel> metrics, HttpContext context)
        {
            ListingFilter eventFilter = new ListingFilter();
            eventFilter.EventType = AtlasInternalEventType.Event;
            eventFilter.Columns = AtlasListingDefaults.EventDefault();
            eventFilter.Limit = 1000;
            eventFilter.Page = 0;
            eventFilter.SortOrder = new SortFilter[] { new SortFilter() { Ordinal = 0, Column = "end_time", Ascending = false } };
            eventFilter.QueryFilters = new[] { new QueryFilter() { Key = "event_id", FilterType = FilterType.In, ValueArray = eventIDs.Select(x => (dynamic)x).ToArray() } };
            AtlasListing listing = GetListing(context, ref eventFilter, metrics);
            var statuscode = ListingHelper.PostProcessAtlasEventListing(ref listing, metrics, context);
            return new Tuple<int, IEnumerable<dynamic>>(statuscode, listing.Items);
        }

        internal Tuple<int, IEnumerable<dynamic>> GetEventListByGameNames(IEnumerable<string> gameNames, ConcurrentBag<ElapsedTimeModel> metrics, ListingFilter filter, HttpContext context)
        {
            var statuscode = 200;
            ListingFilter eventFilter = new ListingFilter();
            eventFilter.EventType = AtlasInternalEventType.Event;
            eventFilter.Columns = AtlasListingDefaults.EventDefault();
            eventFilter.Limit = 1000;
            eventFilter.Page = 0;
            List<QueryFilter> filters = filter.QueryFilters.ToArray().ToList();
            filters.Add(new QueryFilter() { Key = "game_name", FilterType = FilterType.In, ValueArray = gameNames.Select(x => (dynamic)x).ToArray() });
            eventFilter.QueryFilters = filters;
            AtlasListing listingResponse = GetListing(context, ref eventFilter, metrics);
            Stopwatch postProcessStopWatch = new Stopwatch();
            postProcessStopWatch.Restart();
            statuscode = ListingHelper.PostProcessAtlasEventListing(ref listingResponse, metrics, context);
            postProcessStopWatch.Stop();
            metrics.Add(new ElapsedTimeModel()
            {
                MetricName = "post-process-event-listing",
                ElapsedMS = postProcessStopWatch.ElapsedMilliseconds
            });
            return new Tuple<int, IEnumerable<dynamic>>(statuscode, listingResponse.Items);
        }

        public IEnumerable<dynamic> GetEventListByTopics(HttpContext context, IEnumerable<string> topics, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            ListingFilter eventFilter = new ListingFilter();
            eventFilter.EventType = AtlasInternalEventType.Event;
            eventFilter.Columns = AtlasListingDefaults.EventDefault();
            eventFilter.Limit = 1000;
            eventFilter.Page = 0;
            eventFilter.QueryFilters = new[] { new QueryFilter() { Key = "topic", FilterType = FilterType.In, ValueArray = topics.Select(x => (dynamic)x).ToArray() } };
            return GetListing(context, ref eventFilter, metrics).Items;
        }

        public IEnumerable<dynamic> GetEventListByFormats(HttpContext context, IEnumerable<string> formats, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            ListingFilter eventFilter = new ListingFilter();
            eventFilter.EventType = AtlasInternalEventType.Event;
            eventFilter.Columns = AtlasListingDefaults.EventDefault();
            eventFilter.Limit = 1000;
            eventFilter.Page = 0;
            eventFilter.QueryFilters = new[] { new QueryFilter() { Key = "format", FilterType = FilterType.In, ValueArray = formats.Select(x => (dynamic)x).ToArray() } };
            return GetListing(context, ref eventFilter, metrics).Items;
        }

        internal Tuple<int, int> EditProduct(HttpContext context, UserAuthData tokenData, ProductEditModal data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            int statusCode = 500;
            int productID = data.ProductID;
            var stopwatch = new Stopwatch();

            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        if (data.ProductID > 0)
                        {
                            sql.AppendLine("");
                            command.Parameters.AddWithValue("@productId", data.ProductID);
                            sql.AppendLine
                            ($@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_product
                                set 
                                    product_name = @productName,
                                    product_description = @product_description,
                                    product_owner = @productOwner, 
                                    topic = @topic,
                                    sub_topic = @sub_topic,
                                    format = @format,
                                    sub_format = @sub_format,
                                    is_active = @is_active,
                                    hash = @hash,
                                    premium_content_creator_id = @premium_content_creator_id,
                                    updated_time = @updatedTime,
                                    updated_user = @updatedUser,
                                    contract_id = @contractID
                                where product_id = @productId
                                ;
                                select case when coalesce((select product_id from {Constants.DatabaseSchema}microservice_twitch_atlas_product where product_id = @productId), 0) > 0 then 200 else 404 end as content_exists;
                            ");

                        }
                        else
                        {
                            sql.AppendLine
                            ($@"
                                insert into {Constants.DatabaseSchema}microservice_twitch_atlas_product (product_name, product_description, product_owner, topic, sub_topic, format, sub_format, is_active, created_time, created_user, hash, premium_content_creator_id, updated_time, updated_user, contract_id)
                                select 
                                    @productName as product_name,
                                    @product_description as product_description,
                                    @productOwner as product_owner, 
                                    @topic as topic,
                                    @sub_topic as sub_topic,
                                    @format as format,
                                    @sub_format as sub_format,
                                    @is_active as is_active,
                                    @createdTime as created_time, 
                                    @createdUser as created_user,
                                    @hash as hash,
                                    @premium_content_creator_id as premium_content_creator_id,
                                    @updatedTime as updated_time,
                                    @updatedUser as updated_user,
                                    @contractID as contract_id
                                ;
                                select 200 as content_exists;
                            ");
                        }

                        command.CommandText = sql.ToString();
                        command.Parameters.AddWithValue("@productName", data.ProductName);
                        command.Parameters.AddWithValue("@product_description", data.ProductDescription);
                        command.Parameters.AddWithValue("@productOwner", data.ProductOwner);
                        command.Parameters.AddWithValue("@topic", data.Topic);
                        command.Parameters.AddWithValue("@sub_topic", data.SubTopic);
                        command.Parameters.AddWithValue("@format", data.Format);
                        command.Parameters.AddWithValue("@sub_format", data.SubFormat);
                        command.Parameters.AddWithValue("@is_active", data.IsActive);
                        command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.Parameters.AddWithValue("@premium_content_creator_id", data.PremiumContentCreatorID);
                        command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@contractId", data.ContractID);
                        statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "atlas_edit_product", context: context);
                        if (statusCode == 200 && data.ProductID == 0)
                        {
                            command.CommandText = "select LAST_INSERT_ID()";
                            productID = (int)command.ExecuteScalarWithMeasurements<ulong>("get_product_id", context: context);
                        }
                        else if (statusCode == 200)
                        {
                            productID = data.ProductID;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_product_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }

            // Handle asset
            if(statusCode == 200 && productID > 0)
            {
                if(data.FrontPageBanner == null || data.FrontPageBanner.Length == 0)
                {
                    try
                    {
                        using (var conn = DBManagerMysql.GetConnection(true))
                        {
                            using (var command = conn.GetCommand())
                            {
                                command.CommandText = $@"delete from {Constants.DatabaseSchema}microservice_twitch_atlas_product_assets where product_id = @product_id and asset_location = 'front-page-promotional-banner-slot';";
                                command.Parameters.AddWithValue("@product_id", productID);
                                command.ExecuteNonQueryWithMeasurements("delete_asset_fpb");
                            }
                        }
                    }
                    catch(Exception ex)
                    {
                        Log.Error(ex);
                    }
                }
            }

            return new Tuple<int, int>(statusCode, productID);
        }

        internal int EditSeason(HttpContext context, UserAuthData tokenData, AtlasSeason data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();

            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {

                        if (data.SeasonID > 0)
                        {
                            sql.AppendLine("");
                            command.Parameters.AddWithValue("@seasonId", data.SeasonID);
                            sql.AppendLine
                            ($@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_season
                                set 
                                    product_id = @productId, 
                                    season_name = @seasonName, 
                                    start_time = @startTime, 
                                    end_time = @endTime,
                                    pledged_hours_broadcast = @pledgedHoursBroadcast,
                                    prize_pool = @prizePool,
                                    is_active = @is_active,
                                    hash = @hash,
                                    previous_season_id = @previous_season_id,
                                    previous_season_name = @previous_season_name,
                                    updated_time = @updatedTime,
                                    updated_user = @updatedUser
                                where season_id = @seasonId
                                ;
                                select case when coalesce((select season_id from {Constants.DatabaseSchema}microservice_twitch_atlas_season where season_id = @seasonId), 0) > 0 then 200 else 404 end as content_exists;
                            ");

                        }
                        else
                        {
                            sql.AppendLine
                            ($@"
                                insert into {Constants.DatabaseSchema}microservice_twitch_atlas_season (product_id, season_name, start_time, end_time, pledged_hours_broadcast, prize_pool, 
                                    is_active, created_time, created_user, hash,
                                    previous_season_id, previous_season_name, updated_time, updated_user)
                                select 
                                    @productId as product_id, 
                                    @seasonName as season_name, 
                                    @startTime as start_time, 
                                    @endTime as end_time, 
                                    @pledgedHoursBroadcast as pledged_hours_broadcast,
                                    @prizePool as prize_pool,
                                    @is_active as is_active,
                                    @createdTime as created_time,
                                    @createdUser as created_user,
                                    @hash as hash,
                                    @previous_season_id as previous_season_id,
                                    @previous_season_name as previous_season_name,
                                    @updatedTime as updated_time,
                                    @updatedUser as updated_user
                                ;
                                select 200 as content_exists;
                            ");
                        }

                        command.CommandText = sql.ToString();
                        command.Parameters.AddWithValue("@productId", data.ProductID);
                        command.Parameters.AddWithValue("@seasonName", data.SeasonName);
                        command.Parameters.AddWithValue("@startTime", data.StartTime);
                        command.Parameters.AddWithValue("@endTime", data.EndTime);
                        command.Parameters.AddWithValue("@pledgedHoursBroadcast", data.PledgedHoursBroadcast);
                        command.Parameters.AddWithValue("@prizePool", data.PrizePool);
                        command.Parameters.AddWithValue("@is_active", data.IsActive);
                        command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.Parameters.AddWithValue("@previous_season_id", data.PreviousSeasonID);
                        command.Parameters.AddWithValue("@previous_season_name", data.PreviousSeasonName);
                        command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                        statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "atlas_edit_season", context: context);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "season_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        internal Tuple<int, int> EditEvent(HttpContext context, UserAuthData tokenData, AtlasEvent data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            int eventID = data.EventID;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {

                        if (data.EventID > 0)
                        {
                            sql.AppendLine("");
                            command.Parameters.AddWithValue("@eventId", data.EventID);
                            sql.AppendLine
                            ($@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_event
                                set 
                                    season_id = @seasonId, 
                                    product_id = @productId, 
                                    event_name = @eventName,
                                    premium_content_creator_id = @premium_content_creator_id,
                                    start_time = @startTime, 
                                    end_time = @endTime,
                                    topic = @topic,
                                    sub_topic = @sub_topic,
                                    format = @format,
                                    sub_format = @sub_format,
                                    event_type = @eventType,
                                    estimated_average_ccv = @estimatedAvgCcv,
                                    pledged_hours_broadcast = @pledgedHoursBroadcast,
                                    is_active = @is_active,
                                    hash = @hash,
                                    game_name = @game_name,
                                    distribution = @distribution,
                                    twitch_involvement = @twitch_involvement,
                                    costreaming_settings = @costreaming_settings,
                                    updated_time = @updatedTime,
                                    updated_user = @updatedUser
                                where event_id = @eventId
                                ;
                                select case when coalesce((select event_id from {Constants.DatabaseSchema}microservice_twitch_atlas_event where event_id = @eventId), 0) > 0 then 200 else 404 end as content_exists;
                            ");

                        }
                        else
                        {
                            sql.AppendLine
                            ($@"
                                insert into {Constants.DatabaseSchema}microservice_twitch_atlas_event (season_id, product_id, event_name, premium_content_creator_id, start_time, end_time, topic, sub_topic, format, sub_format, event_type, estimated_average_ccv, pledged_hours_broadcast, is_active, created_time, created_user, hash, game_name, distribution, twitch_involvement, costreaming_settings, updated_time, updated_user)
                                select 
                                    @seasonId as season_id, 
                                    @productId as product_id, 
                                    @eventName as event_name,
                                    @premium_content_creator_id as premium_content_creator_id,
                                    @startTime as start_time, 
                                    @endTime as end_time,
                                    @topic as topic,
                                    @sub_topic as sub_topic,
                                    @format as format,
                                    @sub_format as sub_format,
                                    @eventType as event_type,
                                    @estimatedAvgCcv as estimated_average_ccv,
                                    @pledgedHoursBroadcast as pledged_hours_broadcast,
                                    @is_active as is_active,
                                    @createdTime as created_time, 
                                    @createdUser as created_user,
                                    @hash as hash,
                                    @game_name as game_name,
                                    @distribution as distribution,
                                    @twitch_involvement as twitch_involvement,
                                    @costreaming_settings as costreaming_settings,
                                    @updatedTime as updated_time,
                                    @updatedUser as updated_user
                                ;
                                select 200 as content_exists;
                            ");
                        }

                        command.CommandText = sql.ToString();
                        command.Parameters.AddWithValue("@seasonId", data.SeasonID);
                        command.Parameters.AddWithValue("@productId", data.ProductID);
                        command.Parameters.AddWithValue("@eventName", data.EventName);
                        command.Parameters.AddWithValue("@premium_content_creator_id", data.PremiumContentCreatorID);
                        command.Parameters.AddWithValue("@startTime", data.StartTime);
                        command.Parameters.AddWithValue("@endTime", data.EndTime);
                        command.Parameters.AddWithValue("@topic", data.Topic);
                        command.Parameters.AddWithValue("@sub_topic", data.SubTopic);
                        command.Parameters.AddWithValue("@format", data.Format);
                        command.Parameters.AddWithValue("@sub_format", data.SubFormat);
                        command.Parameters.AddWithValue("@eventType", data.EventType);
                        command.Parameters.AddWithValue("@game_name", data.GameName);
                        command.Parameters.AddWithValue("@distribution", data.Distribution);
                        command.Parameters.AddWithValue("@twitch_involvement", data.TwitchInvolvement);
                        command.Parameters.AddWithValue("@costreaming_settings", data.CostreamingSettings);
                        command.Parameters.AddWithValue("@estimatedAvgCcv", data.EstimatedAverageCCV);
                        command.Parameters.AddWithValue("@pledgedHoursBroadcast", data.StartTime.HasValue ? ( data.EndTime.Value - data.StartTime.Value).TotalHours : 0);
                        command.Parameters.AddWithValue("@is_active", data.IsActive);
                        command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                        statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "atlas_edit_event", context: context);
                        if (statusCode == 200 && data.EventID == 0)
                        {
                            command.CommandText = "select LAST_INSERT_ID()";
                            eventID = (int)command.ExecuteScalarWithMeasurements<ulong>("get_event_id", context: context);
                        }
                        else
                        {
                            eventID = data.EventID;
                        }
                    }
                }
                CloudwatchHelper.EnqueueMetricRequest("edit_event_statuscode", statusCode, context);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                CloudwatchHelper.EnqueueMetricRequest("edit_event_error", 1, context);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "event_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return new Tuple<int, int>(statusCode, eventID);
        }

        internal Dictionary<int, string> GetProductNames(HttpContext context, IEnumerable<int> productIDs, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            Dictionary<int, string> productNames = new Dictionary<int, string>();
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                if (!productIDs.Any())
                {
                    return productNames;
                }
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText = $"select microservice_twitch_atlas_product.product_name,microservice_twitch_atlas_product.product_id from {Constants.DatabaseSchema}microservice_twitch_atlas_product where product_id in ({string.Join(",", productIDs)})";
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_product_names").MysqlReader)
                        {
                            while (reader.Read())
                            {
                                int productID = (int)reader["product_id"];
                                string productName = (string)reader["product_name"];
                                productNames[productID] = productName;
                            }
                        }
                    }
                }

            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "get_product_name_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return productNames;
        }

        internal int EditStream(HttpContext context, UserAuthData tokenData, AtlasStream data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {

                        if (data.StreamID > 0)
                        {
                            sql.AppendLine("");
                            command.Parameters.AddWithValue("@streamId", data.StreamID);
                            sql.AppendLine
                            ($@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_stream
                                set 
                                    event_id = @eventId, 
                                    channel_id = @channelId,
                                    stream_login = @streamLogin,
                                    game_name = @gameName,
                                    start_time = @startTime, 
                                    end_time = @endTime,
                                    channel_type = @channelType,
                                    is_active = @is_active,
                                    hash = @hash,
                                    updated_time = @updatedTime,
                                    updated_user = @updatedUser
                                where stream_id = @streamId
                                ;
                                select case when coalesce((select stream_id from {Constants.DatabaseSchema}microservice_twitch_atlas_stream where stream_id = @streamId), 0) > 0 then 200 else 404 end as content_exists;
                            ");

                        }
                        else
                        {
                            sql.AppendLine
                            ($@"
                                insert into {Constants.DatabaseSchema}microservice_twitch_atlas_stream (event_id, channel_id, stream_login, game_name, start_time, end_time, channel_type, is_active, created_time, created_user, hash, updated_time, updated_user)
                                select 
                                    @eventId as event_id, 
                                    @channelId as channel_id,
                                    @streamLogin as stream_login,
                                    @gameName as game_name,
                                    @startTime as start_time, 
                                    @endTime as end_time, 
                                    @channelType as channel_type,
                                    @is_active as is_active,
                                    @createdTime as created_time, 
                                    @createdUser as created_user,
                                    @hash as hash,
                                    @updatedTime as updated_time,
                                    @updatedUser as updated_user
                                ;
                                select 200 as content_exists;
                            ");
                        }

                        command.CommandText = sql.ToString();
                        command.Parameters.AddWithValue("@eventId", data.EventID);
                        command.Parameters.AddWithValue("@channelId", data.ChannelID);
                        command.Parameters.AddWithValue("@streamLogin", data.StreamLogin);
                        command.Parameters.AddWithValue("@gameName", data.GameName);
                        command.Parameters.AddWithValue("@startTime", data.StartTime);
                        command.Parameters.AddWithValue("@endTime", data.EndTime);
                        command.Parameters.AddWithValue("@channelType", data.ChannelType);
                        command.Parameters.AddWithValue("@is_active", data.IsActive);
                        command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                        statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "atlas_edit_stream", context: context);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_stream_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        internal void DeleteStreams(HttpContext context, IEnumerable<int> streamIDs)
        {
            if (!streamIDs.Any())
            {
                return;
            }
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"delete from {Constants.DatabaseSchema}microservice_twitch_atlas_stream where stream_id in ({string.Join(",", streamIDs.ToArray())})";
                    command.ExecuteNonQueryWithMeasurements("delete_atlas_stream");
                }
            }
        }

        internal void DeleteContractChannels(HttpContext context, IEnumerable<int> contractChannelIDs)
        {
            if (!contractChannelIDs.Any())
            {
                return;
            }
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"delete from {Constants.DatabaseSchema}microservice_twitch_atlas_contract_channels where contract_channel_id in ({string.Join(",", contractChannelIDs.ToArray())})";
                    command.ExecuteNonQueryWithMeasurements("delete_contract_channels");
                }
            }
        }

        internal void DeletePCCAms(HttpContext context, IEnumerable<int> pccAMIDs)
        {
            if (!pccAMIDs.Any())
            {
                return;
            }
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"delete from {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_am_map where pcc_am_map_id in ({string.Join(",", pccAMIDs.ToArray())})";
                    command.ExecuteNonQueryWithMeasurements("delete_pcc_am_map");
                }
            }
        }

        internal void DeletePCCChannels(HttpContext context, IEnumerable<int> pccChannelIDs)
        {
            if (!pccChannelIDs.Any())
            {
                return;
            }
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"delete from {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_channel_map where pcc_channel_map_id in ({string.Join(",", pccChannelIDs.ToArray())})";
                    command.ExecuteNonQueryWithMeasurements("delete_pcc_channel_map");
                }
            }
        }

        internal Tuple<int, int?> EditPremiumContentCreator(HttpContext context, UserAuthData tokenData, PremiumContentCreator data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            int? pccID = null;
            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                bool shouldContinue = !string.IsNullOrWhiteSpace(data.PremiumContentCreatorName);
                bool? pccNameExists = null;

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    if (string.IsNullOrWhiteSpace(data.PremiumContentCreatorName))
                    {
                        pccNameExists = null;
                    }
                    else
                    {
                        try
                        {
                            using (var command = conn.GetCommand())
                            {
                                command.CommandText = $@"select case when count(*) > 0 then True else False end from {Constants.DatabaseSchema}microservice_twitch_atlas_premium_content_creator where premium_content_creator_name = @pccName;";
                                command.Parameters.AddWithValue("@pccName", data.PremiumContentCreatorName);
                                pccNameExists = Convert.ToBoolean((long)command.ExecuteScalar());
                            }
                        }
                        catch (Exception ex)
                        {
                            Log.Error(ex);
                            pccNameExists = null;
                        }
                    }

                    if
                    (
                        shouldContinue
                        &&
                        (
                            // Update existing entry
                            data.PremiumContentCreatorID > 0
                            // New entry, and name does not exist
                            || (data.PremiumContentCreatorID == 0 && pccNameExists == false)
                        )
                    )
                    {
                        using (var transaction = conn.BeginTransaction())
                        {
                            try
                            {
                                using (var command = conn.GetCommand())
                                {
                                    command.Transaction = transaction;
                                    if (data.PremiumContentCreatorID > 0)
                                    {
                                        sql.AppendLine("");
                                        command.Parameters.AddWithValue("@pccID", data.PremiumContentCreatorID);
                                        sql.AppendLine
                                        ($@"
                                        update {Constants.DatabaseSchema}microservice_twitch_atlas_premium_content_creator
                                        set
                                            premium_content_creator_name = @pccName,
                                            premium_content_creator_type = @pccType,
                                            is_active = @isActive,
                                            am_name = @amName,
                                            hash = @hash,
                                            updated_time = @updatedTime,
                                            updated_user = @updatedUser
                                        where premium_content_creator_id = @pccID
                                        ;
                                        select case when coalesce((select premium_content_creator_id from {Constants.DatabaseSchema}microservice_twitch_atlas_premium_content_creator where premium_content_creator_id = @pccID), 0) > 0 then 200 else 404 end as content_exists;
                                    ");
                                    }
                                    else
                                    {
                                        sql.AppendLine
                                        ($@"
                                        insert into {Constants.DatabaseSchema}microservice_twitch_atlas_premium_content_creator (premium_content_creator_name, premium_content_creator_type, is_active, created_time, created_user, hash, am_name, updated_time, updated_user)
                                        select 
                                            @pccName as premium_content_creator_name, 
                                            @pccType as premium_content_creator_type, 
                                            @isActive as is_active,
                                            @createdTime as created_time, 
                                            @createdUser as created_user,
                                            @hash as hash,
                                            @amName as am_name,
                                            @updatedTime as updated_time,
                                            @updatedUser as updated_user
                                        ;
                                        insert into {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_am_map
                                        (premium_content_creator_id, account_manager_id, is_active, created_time, created_user, hash)
                                        select
                                            (select premium_content_creator_id from {Constants.DatabaseSchema}microservice_twitch_atlas_premium_content_creator where premium_content_creator_name = @pccName limit 1) as premium_content_creator_id,
                                            (select account_manager_id from {Constants.DatabaseSchema}microservice_twitch_atlas_account_manager where account_manager_ldap_name = @amName limit 1) as account_manager_id,
                                            true as is_active,
                                            @createdTime as created_time,
                                            @createdUser as created_user,
                                            @hash as hash
                                        ;
                                        select 200 as content_exists;
                                    ");
                                    }
                                    command.CommandText = sql.ToString();
                                    command.Parameters.AddWithValue("@pccName", data.PremiumContentCreatorName);
                                    command.Parameters.AddWithValue("@pccType", data.PremiumContentCreatorType);
                                    command.Parameters.AddWithValue("@amName", tokenData.UserID);
                                    command.Parameters.AddWithValue("@isActive", data.IsActive);
                                    command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                                    command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                                    command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                                    command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                                    command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                                    statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "atlas_edit_pcc", context: context);
                                    transaction.Commit();
                                }
                            }
                            catch (Exception ex)
                            {
                                Log.Error(ex);
                                statusCode = 500;
                                transaction.Rollback();
                            }

                            if (statusCode == 200 && data.PremiumContentCreatorID == 0)
                            {
                                try
                                {
                                    using (var command = conn.GetCommand())
                                    {
                                        command.CommandText = $"select premium_content_creator_id from {Constants.DatabaseSchema}microservice_twitch_atlas_premium_content_creator where premium_content_creator_name = @pccName limit 1;";
                                        command.Parameters.AddWithValue("@pccName", data.PremiumContentCreatorName);
                                        pccID = command.ExecuteScalarWithMeasurements<int>("get_pcc_id_from_name", context: context);
                                        if(pccID != null && pccID > 0)
                                        {
                                            data.PremiumContentCreatorID = pccID.Value;
                                        }
                                    }
                                }
                                catch (Exception ex)
                                {
                                    statusCode = 500;
                                    Log.Error(ex);
                                }
                            }
                            else
                            {
                                pccID = data.PremiumContentCreatorID;
                            }

                        }
                    }
                    else
                    {
                        statusCode = 400;
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_pcc_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return new Tuple<int, int?>(statusCode, pccID);
        }

        internal int EditPremiumContentCreatorChannels(HttpContext context, UserAuthData tokenData, PremiumContentCreatorToChannelMap data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        if (data.PccChannelMapId > 0)
                        {
                            sql.AppendLine("");
                            command.Parameters.AddWithValue("@pccChannelMapID", data.PccChannelMapId);
                            sql.AppendLine
                            ($@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_channel_map
                                set
                                    premium_content_creator_id = @pccID,
                                    channel_id = @channelID,
                                    is_active = @isActive,
                                    hash = @hash,
                                    channel_ownership = @channel_ownership,
                                    channel_vertical = @channel_vertical,
                                    updated_time = @updatedTime,
                                    updated_user = @updatedUser
                                where pcc_channel_map_id = @pccChannelMapID
                                ;
                                select case when coalesce((select premium_content_creator_id from {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_channel_map where pcc_channel_map_id = @pccChannelMapID), 0) > 0 then 200 else 404 end as content_exists;
                            ");

                        }
                        else
                        {
                            sql.AppendLine
                            ($@"
                                insert into {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_channel_map (premium_content_creator_id, channel_id, is_active, created_time, created_user, hash, channel_ownership, channel_vertical, updated_time, updated_user)
                                select 
                                    @pccID as premium_content_creator_id, 
                                    @channelID as channel_id, 
                                    @isActive as is_active,
                                    @createdTime as created_time, 
                                    @createdUser as created_user,
                                    @hash as hash,
                                    @channel_ownership as channel_ownership,
                                    @channel_vertical as channel_vertical,
                                    @updatedTime as updated_time,
                                    @updatedUser as updated_user
                                ;
                                select 200 as content_exists;
                            ");
                        }

                        command.CommandText = sql.ToString();
                        command.Parameters.AddWithValue("@pccID", data.PremiumContentCreatorID);
                        command.Parameters.AddWithValue("@channelID", data.ChannelID);
                        command.Parameters.AddWithValue("@isActive", data.IsActive);
                        command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@channel_ownership", data.ChannelOwnership);
                        command.Parameters.AddWithValue("@channel_vertical", data.ChannelVertical);
                        command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "atlas_edit_pcc_channel_map", context: context);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_pcc_channel_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }

            return statusCode;
        }

        internal int EditPremiumContentCreatorAccountManager(HttpContext context, UserAuthData tokenData, PremiumContentCreatorToAccountManagerMap data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        if (data.PccToAmMapID > 0)
                        {
                            sql.AppendLine("");
                            command.Parameters.AddWithValue("@pccAmMapID", data.PccToAmMapID);
                            sql.AppendLine
                            ($@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_am_map
                                set
                                    premium_content_creator_id = @pccID,
                                    account_manager_id = @amID,
                                    is_active = @isActive,
                                    hash = @hash,
                                    updated_time = @updatedTime,
                                    updated_user = @updatedUser
                                where pcc_am_map_id = @pccAmMapID
                                ;
                                select case when coalesce((select pcc_am_map_id from {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_am_map where pcc_am_map_id = @pccAmMapID), 0) > 0 then 200 else 404 end as content_exists;
                            ");

                        }
                        else
                        {
                            sql.AppendLine
                            ($@"
                                insert into {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_am_map (premium_content_creator_id, account_manager_id, is_active, created_time, created_user, hash, updated_time, updated_user)
                                select 
                                    @pccID as premium_content_creator_id, 
                                    @amID as account_manager_id, 
                                    @isActive as is_active,
                                    @createdTime as created_time, 
                                    @createdUser as created_user,
                                    @hash as hash,
                                    @updatedTime as updated_time,
                                    @updatedUser as updated_user
                                ;
                                select 200 as content_exists;
                            ");
                        }

                        command.CommandText = sql.ToString();
                        command.Parameters.AddWithValue("@pccID", data.PremiumContentCreatorID);
                        command.Parameters.AddWithValue("@amID", data.AccountManagerID);
                        command.Parameters.AddWithValue("@isActive", data.IsActive);
                        command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                        statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "atlas_edit_pcc_am_map", context: context);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_pcc_am_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        internal int EditAccountManager(HttpContext context, UserAuthData tokenData, AtlasAccountManager data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        if (data.AccountManagerID > 0)
                        {
                            sql.AppendLine("");
                            command.Parameters.AddWithValue("@amID", data.AccountManagerID);
                            sql.AppendLine
                            ($@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_account_manager
                                set
                                    account_manager_ldap_name = @accountManagerLdapName,
                                    account_manager_first_name = @accountManagerFirstName,
                                    account_manager_last_name = @accountManagerLastName,
                                    account_manager_email = @accountManagerEmail,
                                    approving_manager_amazon_id = @approvingManagerAmazonID,
                                    approving_manager_name = @approvingManagerAmazonName,
                                    is_active = @isActive,                                    
                                    hash = @hash,
                                    updated_time = @updatedTime,
                                    updated_user = @updatedUser
                                where account_manager_id = @amID
                                ;
                                select case when coalesce((select account_manager_id from {Constants.DatabaseSchema}microservice_twitch_atlas_account_manager where account_manager_id = @amID), 0) > 0 then 200 else 404 end as content_exists;
                            ");

                        }
                        else
                        {
                            sql.AppendLine
                            ($@"
                                replace into {Constants.DatabaseSchema}microservice_twitch_atlas_account_manager (account_manager_ldap_name, account_manager_first_name, account_manager_last_name, account_manager_email, approving_manager_amazon_id, approving_manager_name, is_active, created_time, created_user, hash, updated_time, updated_user)
                                select 
                                    @accountManagerLdapName as account_manager_ldap_name, 
                                    @accountManagerFirstName as account_manager_first_name, 
                                    @accountManagerLastName as account_manager_last_name,
                                    @accountManagerEmail as account_manager_email,
                                    @approvingManagerAmazonID as approving_manager_amazon_id,
                                    @approvingManagerAmazonName as approving_manager_name,
                                    @isActive as is_active,
                                    @createdTime as created_time, 
                                    @createdUser as created_user,
                                    @hash as hash,
                                    @updatedTime as updated_time,
                                    @updatedUser as updated_user
                                ;
                                select 200 as content_exists;
                            ");
                        }

                        command.CommandText = sql.ToString();
                        command.Parameters.AddWithValue("@accountManagerLdapName", data.AccountManagerLdapName);
                        command.Parameters.AddWithValue("@accountManagerFirstName", data.AccountManagerFirstName);
                        command.Parameters.AddWithValue("@accountManagerLastName", data.AccountManagerLastName);
                        command.Parameters.AddWithValue("@accountManagerEmail", data.AccountManagerEmail);
                        command.Parameters.AddWithValue("@approvingManagerAmazonID", data.ApprovingManagerAmazonID);
                        command.Parameters.AddWithValue("@approvingManagerAmazonName", data.ApprovingManagerName);
                        command.Parameters.AddWithValue("@isActive", data.IsActive);
                        command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                        statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "atlas_edit_account_manager", context: context);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_am_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return statusCode;
        }

        internal Tuple<int, int> EditContract(HttpContext context, UserAuthData tokenData, AtlasContract data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            int contractID = data.ContractID;

            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {

                        if (data.ContractID > 0)
                        {
                            sql.AppendLine("");
                            command.Parameters.AddWithValue("@contractID", data.ContractID);
                            sql.AppendLine
                            ($@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_contracts
                                set 
                                    contract_name = @contractName,
                                    contract_type = @contractType, 
                                    premium_content_creator_id = @pccID,
                                    start_date = @startDate,
                                    end_date = @endDate,
                                    revenue_share_enabled = @revShareEnabled,
                                    ad_revenue_contract_type = @adRevContractType,
                                    ad_revenue_share_type_tier_1 = @adRevShareTypeTier1,
                                    ad_revenue_share_amount_tier_1 = @adRevShareAmountTier1,
                                    ad_revenue_share_type_tier_2 = @adRevShareTypeTier2,
                                    ad_revenue_share_amount_tier_2 = @adRevShareAmountTier2,
                                    ad_revenue_share_type_tier_3 = @adRevShareTypeTier3,
                                    ad_revenue_share_amount_tier_3 = @adRevShareAmountTier3,
                                    bits_revenue_share_percent = @bitsRevSharePercent,
                                    subs_revenue_share_percent = @subsRevSharePercent,
                                    ad_density_per_hour = @adDensityPerHour,
                                    exclusivity_enabled = @exclusivity,
                                    ad_prerolls_enabled = @prerolls,
                                    sponsorship_enabled = @sponsorship,
                                    additional_notes = @additional_notes,
                                    document_link = @document_link,
                                    auto_renew = @auto_renew,
                                    is_active = @is_active,
                                    hash = @hash,
                                    updated_time = @updatedTime,
                                    updated_user = @updatedUser
                                where contract_id = @contractID
                                ;
                                select case when coalesce((select contract_id from {Constants.DatabaseSchema}microservice_twitch_atlas_contracts where contract_id = @contractID), 0) > 0 then 200 else 404 end as content_exists;
                            ");

                        }
                        else
                        {
                            sql.AppendLine
                            ($@"
                                insert into {Constants.DatabaseSchema}microservice_twitch_atlas_contracts (contract_name, contract_type, premium_content_creator_id,
                                start_date, end_date, revenue_share_enabled,
                                ad_revenue_contract_type, ad_revenue_share_type_tier_1, ad_revenue_share_amount_tier_1, 
                                ad_revenue_share_type_tier_2, ad_revenue_share_amount_tier_2,
                                ad_revenue_share_type_tier_3, ad_revenue_share_amount_tier_3,
                                bits_revenue_share_percent, subs_revenue_share_percent, ad_density_per_hour,
                                exclusivity_enabled, ad_prerolls_enabled, sponsorship_enabled,additional_notes, document_link, auto_renew,
                                is_active, created_time, created_user, hash, updated_time, updated_user)
                                select 
                                    @contractName as contract_name, 
                                    @contractType as contract_type, 
                                    @pccID as premium_content_creator_id,
                                    @startDate as start_date,
                                    @endDate as end_date,
                                    @revShareEnabled as revenue_share_enabled,
                                    @adRevContractType as ad_revenue_contract_type,
                                    @adRevShareTypeTier1 as ad_revenue_share_type_tier_1,
                                    @adRevShareAmountTier1 as ad_revenue_share_amount_tier_1,
                                    @adRevShareTypeTier2 as ad_revenue_share_type_tier_2,
                                    @adRevShareAmountTier2 as ad_revenue_share_amount_tier_2,
                                    @adRevShareTypeTier3 as ad_revenue_share_type_tier_3,
                                    @adRevShareAmountTier3 as ad_revenue_share_amount_tier_3,
                                    @bitsRevSharePercent as bits_revenue_share_percent,
                                    @subsRevSharePercent as subs_revenue_share_percent,
                                    @adDensityPerHour as ad_density_per_hour,
                                    @exclusivity as exclusivity_enabled,
                                    @prerolls as ad_prerolls_enabled,
                                    @sponsorship as sponsorship_enabled,
                                    @additional_notes as additional_notes,
                                    @document_link as document_link,
                                    @auto_renew as auto_renew,
                                    @is_active as is_active,
                                    @createdTime as created_time, 
                                    @createdUser as created_user,
                                    @hash as hash,
                                    @updatedTime as updated_time,
                                    @updatedUser as updated_user
                                
                                ;
                                select 200 as content_exists;
                            ");
                        }

                        command.CommandText = sql.ToString();
                        command.Parameters.AddWithValue("@contractName", data.ContractName);
                        command.Parameters.AddWithValue("@contractType", data.ContractType);
                        command.Parameters.AddWithValue("@pccID", data.PremiumContentCreatorID);
                        command.Parameters.AddWithValue("@startDate", data.StartDate);
                        command.Parameters.AddWithValue("@endDate", data.EndDate);
                        command.Parameters.AddWithValue("@revShareEnabled", data.RevenueShareEnabled);
                        command.Parameters.AddWithValue("@adRevContractType", data.AdRevenueContractType);
                        command.Parameters.AddWithValue("@adRevShareTypeTier1", data.AdRevenueShareTypeTier1);
                        command.Parameters.AddWithValue("@adRevShareAmountTier1", data.AdRevenueShareAmountTier1);
                        command.Parameters.AddWithValue("@adRevShareTypeTier2", data.AdRevenueShareTypeTier2);
                        command.Parameters.AddWithValue("@adRevShareAmountTier2", data.AdRevenueShareAmountTier2);
                        command.Parameters.AddWithValue("@adRevShareTypeTier3", data.AdRevenueShareTypeTier3);
                        command.Parameters.AddWithValue("@adRevShareAmountTier3", data.AdRevenueShareAmountTier3);
                        command.Parameters.AddWithValue("@bitsRevSharePercent", data.BitsRevenueSharePercent);
                        command.Parameters.AddWithValue("@subsRevSharePercent", data.SubsRevenueSharePercent);
                        command.Parameters.AddWithValue("@adDensityPerHour", data.AdDensityPerHour);
                        command.Parameters.AddWithValue("@exclusivity", data.ExclusivityEnabled);
                        command.Parameters.AddWithValue("@prerolls", data.AdPrerollsEnabled);
                        command.Parameters.AddWithValue("@sponsorship", data.SponsorshipEnabled);
                        command.Parameters.AddWithValue("@additional_notes", data.AdditionalNotes);
                        command.Parameters.AddWithValue("@document_link", data.DocumentLink);
                        command.Parameters.AddWithValue("@auto_renew", data.AutoRenew);
                        command.Parameters.AddWithValue("@is_active", data.IsActive);
                        command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                        statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "atlas_edit_contract", context: context);
                        if (statusCode == 200 && data.ContractID == 0)
                        {
                            command.CommandText = "select LAST_INSERT_ID()";
                            contractID = (int)command.ExecuteScalarWithMeasurements<ulong>("get_contract_id", context: context);
                        }
                        else if (statusCode == 200)
                        {
                            contractID = data.ContractID;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_contract_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return new Tuple<int, int>(statusCode, contractID);
        }

        internal int EditContractChannel(HttpContext context, UserAuthData tokenData, AtlasContractChannel data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {

                        if (data.ChannelContractID > 0)
                        {
                            sql.AppendLine("");
                            command.Parameters.AddWithValue("@contract_channel_id", data.ChannelContractID);
                            sql.AppendLine
                            ($@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_contract_channels
                                set 
                                    contract_id = @contractID,
                                    channel_id = @channelID,
                                    custom_start_date = @customStartDate,
                                    custom_end_date = @customEndDate,
                                    is_active = @is_active,
                                    hash = @hash,
                                    updated_time = @updatedTime,
                                    updated_user = @updatedUser
                                where contract_channel_id = @contract_channel_id
                                ;
                                select case when coalesce((select contract_channel_id from {Constants.DatabaseSchema}microservice_twitch_atlas_contract_channels where contract_channel_id = @contract_channel_id), 0) > 0 then 200 else 404 end as content_exists;
                            ");

                        }
                        else
                        {
                            sql.AppendLine
                            ($@"
                                insert into {Constants.DatabaseSchema}microservice_twitch_atlas_contract_channels (contract_id, channel_id, custom_start_date, custom_end_date,
                                    is_active, created_time, created_user, hash, updated_time, updated_user)
                                select 
                                    @contractID as contract_name, 
                                    @channelID as channel_id,
                                    @customStartDate as custom_start_date,
                                    @customEndDate as custom_end_date,
                                    @is_active as is_active,
                                    @createdTime as created_time, 
                                    @createdUser as created_user,
                                    @hash as hash,
                                    @updatedTime as updated_time,
                                    @updatedUser as updated_user
                                ;
                                select 200 as content_exists;
                            ");
                        }

                        command.CommandText = sql.ToString();
                        command.Parameters.AddWithValue("@contractID", data.ContractID);
                        command.Parameters.AddWithValue("@channelID", data.ChannelID);
                        command.Parameters.AddWithValue("@customStartDate", data.CustomStartDate);
                        command.Parameters.AddWithValue("@customEndDate", data.CustomEndDate);
                        command.Parameters.AddWithValue("@is_active", data.IsActive);
                        command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                        statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "atlas_edit_contract_channels", context: context);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_contract_channel_query",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });

            }
            return statusCode;
        }

        internal int EditContractAccountManagerMap(HttpContext context, UserAuthData tokenData, AtlasContractAccountManagerMap data, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var statusCode = 500;
            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                var sql = new StringBuilder();

                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {

                        if (data.ContractAccountManagerMapID > 0)
                        {
                            sql.AppendLine("");
                            command.Parameters.AddWithValue("@contract_account_manager_map_id", data.ContractAccountManagerMapID);
                            sql.AppendLine
                            ($@"
                                update {Constants.DatabaseSchema}microservice_twitch_atlas_contract_account_manager_map
                                set 
                                    contract_id = @contract_id,
                                    account_manager_id = @account_manager_id,
                                    is_active = @is_active,
                                    hash = @hash,
                                    updated_time = @updatedTime,
                                    updated_user = @updatedUser
                                where contract_account_manager_map_id = @contract_account_manager_map_id
                                ;
                                select case when coalesce((select contract_account_manager_map_id from {Constants.DatabaseSchema}microservice_twitch_atlas_contract_account_manager_map where contract_account_manager_map_id = @contract_account_manager_map_id), 0) > 0 then 200 else 404 end as content_exists;
                            ");

                        }
                        else
                        {
                            sql.AppendLine
                            ($@"
                                insert into {Constants.DatabaseSchema}microservice_twitch_atlas_contract_account_manager_map (contract_id, account_manager_id,
                                    is_active, created_time, created_user, hash, updated_time, updated_user)
                                select 
                                    @contract_id as contract_id, 
                                    @account_manager_id as account_manager_id,
                                    @is_active as is_active,
                                    @createdTime as created_time, 
                                    @createdUser as created_user,
                                    @hash as hash,
                                    @updatedTime as updated_time,
                                    @updatedUser as updated_user
                                ;
                                select 200 as content_exists;
                            ");
                        }

                        command.CommandText = sql.ToString();
                        command.Parameters.AddWithValue("@contract_id", data.ContractID);
                        command.Parameters.AddWithValue("@account_manager_id", data.AccountManagerID);
                        command.Parameters.AddWithValue("@is_active", data.IsActive);
                        command.Parameters.AddWithValue("@createdTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@createdUser", tokenData.UserID);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.Parameters.AddWithValue("@updatedTime", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@updatedUser", tokenData.UserID);
                        statusCode = (int)command.ExecuteScalarWithMeasurements<long>(logname: "edit_atlas_contract_account_manager_map", context: context);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                statusCode = (int)HttpStatusCode.UnprocessableEntity;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "edit_atlas_contract_account_manager_map",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });

            }
            return statusCode;
        }

        internal CalendarEventListByDate GetActiveEventsForCalendar(HttpContext context, int pccID)
        {
            CalendarEventListByDate data = new CalendarEventListByDate();

            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $@"select e.event_id,e.event_name,e.start_time,e.end_time,e.topic,e.sub_topic,e.`format`,e.sub_format,e.event_type,p.product_name from microservice_twitch_atlas_event e
                        left outer join {Constants.DatabaseSchema}microservice_twitch_atlas_product p on p.product_id = e.product_id
                        where e.is_active = 1 and e.end_time >= current_date() and e.premium_content_creator_id = @pccID;";
                    command.Parameters.AddWithValue("pccID", pccID);
                    using (var reader = new DataReaderWithMeasurements(command, context, "get_calendar_active_events").MysqlReader)
                    {
                        List<EventData> list = new List<EventData>();
                        while (reader.Read())
                        {
                            int eventid = (int)reader["event_id"];
                            string name = (string)reader["event_name"];
                            DateTime starttime = (DateTime)reader["start_time"];
                            DateTime endtime = (DateTime)reader["end_time"];
                            string topic = reader["topic"] != DBNull.Value ? (string)reader["topic"] : "";
                            string subtopic = reader["sub_topic"] != DBNull.Value ? (string)reader["sub_topic"] : "";
                            string format = reader["format"] != DBNull.Value ? (string)reader["format"] : "";
                            string subformat = reader["sub_format"] != DBNull.Value ? (string)reader["sub_format"] : "";
                            string eventtype = (string)reader["event_type"];
                            string productname = reader["product_name"] != DBNull.Value ? (string)reader["product_name"] : "";

                            EventData item = new EventData
                            {
                                EventID = eventid,
                                EventName = name,
                                StartTime = starttime,
                                EndTime = endtime,
                                Topic = topic,
                                SubTopic = subtopic,
                                Format = format,
                                SubFormat = subformat,
                                EventType = eventtype,
                                ProductName = productname,
                            };

                            list.Add(item);
                        }

                        data.EventsByDay = list.GroupBy(x => x.StartTime.ToString("yyyy-MM-dd"))
                            .ToDictionary(y => y.Key, z => z.OrderBy(x => x.StartTime)
                            .OrderBy(x => x.StartTime)
                            .ToList());
                    }
                }
            }

            return data;
        }

        internal IEnumerable<PccStats> GetAggregateProductStats(HttpContext context, int[] pccIds)
        {
            if (pccIds == null || pccIds.Length == 0)
            {
                throw new ArgumentNullException(nameof(pccIds));
            }

            IList<PccStats> list = new List<PccStats>();
            string pccList = string.Join(',', pccIds);
            string query =
                $@"select a.premium_content_creator_id, a.measurement_date, a.total_revenue, a.hours_broadcast, a.hours_watched
                    from {Constants.DatabaseSchema}microservice_twitch_atlas_aggregate_pcc_stats a
                    where a.premium_content_creator_id in ({pccList})
                ;";

            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = query;

                    using (var reader = new DataReaderWithMeasurements(command, context, "get_aggregate_product_stats").MysqlReader)
                    {
                        while (reader.Read())
                        {
                            int premium_content_creator_id = (int)reader["premium_content_creator_id"];
                            DateTime measurement_date = (DateTime)reader["measurement_date"];
                            float total_revenue = (float)reader["total_revenue"];
                            float hours_broadcast = (float)reader["hours_broadcast"];
                            float hours_watched = (float)reader["hours_watched"];

                            PccStats pcc = new PccStats
                            {
                                PremiumContentCreatorID = premium_content_creator_id,
                                MeasurementDate = measurement_date,
                                TotalRevenue = total_revenue,
                                HoursBroadcast = hours_broadcast,
                                HoursWatched = hours_watched
                            };

                            list.Add(pcc);
                        }
                    }
                }
            }


            return list;
        }

        internal IList<long> GetChannelIDsFromEvent(HttpContext context, int eventID)
        {
            if (eventID <= 0)
            {
                throw new ArgumentException("Value must be a positive integer", nameof(eventID));
            }

            IList<long> channelIDs = new List<long>();
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"select distinct channel_id from {Constants.DatabaseSchema}microservice_twitch_atlas_stream where event_id = @event_id and channel_id is not null";
                    command.Parameters.AddWithValue("event_id", eventID);

                    using (var reader = new DataReaderWithMeasurements(command, context, "get_event_channel_ids").MysqlReader)
                    {
                        while (reader.Read())
                        {
                            long channel_id = (long)reader["channel_id"];
                            channelIDs.Add(channel_id);
                        }
                    }
                }
            }

            return channelIDs;
        }

        internal IList<long> GetChannelsByPcc(HttpContext context, int pccID)
        {
            if (pccID <= 0)
            {
                return new List<long>(new long[] { });
            }

            IList<long> channelIDs = new List<long>();

            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"select channel_id from {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_channel_map where premium_content_creator_id = @pcc_id;";
                    command.Parameters.AddWithValue("pcc_id", pccID);

                    using (var reader = new DataReaderWithMeasurements(command, context, "get_channels_by_pcc").MysqlReader)
                    {
                        while (reader.Read())
                        {
                            long channel_id = (long)reader["channel_id"];
                            channelIDs.Add(channel_id);
                        }
                    }
                }
            }

            return channelIDs;
        }

        internal IList<ContractsByPCC> GetContractsByPremiumContentCreator(HttpContext context)
        {
            IList<ContractsByPCC> contracts = new List<ContractsByPCC> ();

            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText = $"select contract_id, contract_name, premium_content_creator_id from {Constants.DatabaseSchema}microservice_twitch_atlas_contracts order by contract_name;";

                    using (var reader = new DataReaderWithMeasurements(command, context, "get_contracts_by_pcc").MysqlReader)
                    {
                        while (reader.Read())
                        {
                            int contract_id = (int)reader["contract_id"];
                            string contract_name = (string)reader["contract_name"];
                            int pcc_id = (int)reader["premium_content_creator_id"];

                            contracts.Add(new ContractsByPCC { ContractID = contract_id, ContractName = contract_name, PremiumContentCreatorID = pcc_id });
                        }
                    }
                }
            }

            return contracts;
        }

        internal IList<EventChannelMap> GetChannelInfoForEventByPCC(int pccID)
        {
            IList<EventChannelMap> list = new List<EventChannelMap>();

            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText =
                        $@"select e.event_id,s.channel_id,s.stream_login from microservice_twitch_atlas_event e
                         left outer join microservice_twitch_atlas_stream s on s.event_id = e.event_id 
                         where e.premium_content_creator_id = @pcc_id
                         order by e.event_id";
                    command.Parameters.AddWithValue("pcc_id", pccID);

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            int? eventID = reader["event_id"] != DBNull.Value ? (int)reader["event_id"] : (int?)null;
                            long? channelID = reader["channel_id"] != DBNull.Value ? (long)reader["channel_id"] : (long?)null;
                            string channelName = reader["stream_login"] != DBNull.Value ? (string)reader["stream_login"] : null;

                            if (eventID != null && channelID != null)
                            {
                                list.Add(new EventChannelMap { EventID = eventID.Value, ChannelID = channelID.Value, ChannelName = channelName });
                            }
                        }
                    }
                }
            }
            
            return list;
        }

        internal IList<EventCsvRecord> GetEventsByPcc(int pccID)
        {
            IList<EventCsvRecord> list = new List<EventCsvRecord>();

            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandText =
                        $@"select e.event_id,e.event_name,e.season_id,season.season_name,e.product_id,product.product_name,e.premium_content_creator_id,pcc.premium_content_creator_name,e.start_time,e.end_time,e.topic,e.sub_topic,

                        e.format, e.sub_format,e.event_type,e.estimated_average_ccv,e.pledged_hours_broadcast,e.is_active,e.created_time,e.created_user,
                        e.game_name,e.distribution,e.twitch_involvement,e.costreaming_settings,e.updated_time,e.updated_user,s.channel_id,
                        e.created_time,e.created_user,e.updated_time,e.updated_user
                            from microservice_twitch_atlas_event e

                            left outer join microservice_twitch_atlas_stream s on s.event_id = e.event_id
                            left outer join microservice_twitch_atlas_season season on season.season_id = e.season_id
                            left outer join microservice_twitch_atlas_product product on product.product_id = e.product_id
                            left outer join microservice_twitch_atlas_premium_content_creator pcc on pcc.premium_content_creator_id = e.premium_content_creator_id

                            where e.premium_content_creator_id = @pcc_id

                            order by event_id
                        ; ";

                    command.Parameters.AddWithValue("pcc_id", pccID);

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            int event_id = (int)reader["event_id"];
                            string event_name = (string)reader["event_name"];
                            int season_id = reader["season_id"] != DBNull.Value ? (int)reader["season_id"] : 0;
                            string season_name = reader["season_name"] != DBNull.Value ? (string)reader["season_name"] : null;
                            int product_id = reader["product_id"] != DBNull.Value ? (int)reader["product_id"] : 0;
                            string product_name = reader["product_name"] != DBNull.Value ? (string)reader["product_name"] : null;
                            int premium_content_creator_id = (int)reader["premium_content_creator_id"];
                            string premium_content_creator_name = reader["premium_content_creator_name"] != DBNull.Value ? (string)reader["premium_content_creator_name"] : null;
                            DateTime start_time = (DateTime)reader["start_time"];                            
                            DateTime end_time = (DateTime)reader["end_time"];                            
                            string topic = reader["topic"] != DBNull.Value ? (string)reader["topic"] : null;
                            string sub_topic = reader["sub_topic"] != DBNull.Value ? (string)reader["sub_topic"] : null;
                            string format = reader["format"] != DBNull.Value ? (string)reader["format"] : null;
                            string sub_format = reader["sub_format"] != DBNull.Value ? (string)reader["sub_format"] : null;
                            string event_type = (string)reader["event_type"];
                            float? estimated_average_ccv = reader["estimated_average_ccv"] != DBNull.Value ? (float?)reader["estimated_average_ccv"] : (float?)null;
                            float? pledged_hours_broadcast = reader["pledged_hours_broadcast"] != DBNull.Value ? (float?)reader["pledged_hours_broadcast"] : (float?)null;
                            bool is_active = (bool)reader["is_active"];
                            string game_name = reader["game_name"] != DBNull.Value ? (string)reader["game_name"] : null;
                            bool? distribution = reader["distribution"] != DBNull.Value ? (bool?)reader["distribution"] : (bool?)null;
                            string twitch_involvement = reader["twitch_involvement"] != DBNull.Value ? (string)reader["twitch_involvement"] : null;
                            string costreaming_settings = reader["costreaming_settings"] != DBNull.Value ? (string)reader["costreaming_settings"] : null;
                            DateTime createdTime = (DateTime)reader["created_time"];
                            string createdUser = (string)reader["created_user"];
                            DateTime? updatedTime = reader["updated_time"] != DBNull.Value ? (DateTime?)reader["updated_time"] : (DateTime?)null;
                            string updatedUser = reader["updated_user"] != DBNull.Value ? (string)reader["updated_user"] : null;

                            list.Add(new EventCsvRecord
                            {
                                EventID = event_id,
                                SeasonID = season_id,
                                SeasonName = season_name,
                                ProductID = product_id,
                                ProductName = product_name,
                                EventName = event_name,
                                PremiumContentCreatorID = premium_content_creator_id,
                                PremiumContentCreatorName = premium_content_creator_name,
                                StartTime = start_time,
                                EndTime = end_time,
                                Topic = topic,
                                SubTopic = sub_topic,
                                Format = format,
                                SubFormat = sub_format,
                                EventType = event_type,
                                EstimatedAverageCCV = estimated_average_ccv,
                                PledgedHoursBroadcast = pledged_hours_broadcast,
                                IsActive = is_active,
                                GameName = game_name,
                                Distribution = distribution,
                                TwitchInvolvement = twitch_involvement,
                                CostreamingSettings = costreaming_settings,
                                CreatedTime = createdTime,
                                CreatedUser = createdUser,
                                UpdatedTime = updatedTime,
                                UpdatedUser = updatedUser,
                            });
                        }
                    }
                }
            }

            return list;
        }
    }
}
