﻿using Resonance.Core;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using System;
using System.Collections.Generic;
using System.Text;
using static Resonance.Core.Models.FilterModels.CustomTwitchDirectFilters;

namespace Resonance.Microservices.Queries
{
    public class TwitchDirectQuery
    {
        internal void Initialize()
        {

        }

        internal List<TwitchDirectArbitrarySummaryModel> GetArbitrarySummaryData(ArbitrarySummaryFilter filter)
        {
            var items = new List<TwitchDirectArbitrarySummaryModel>();
            var sqlBuilder = new StringBuilder();
            try
            {
                using (var conn = DBManagerRedshift.TahoeConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        // Build required parameters for query
                        foreach (var channel in filter.ChannelTimeRangeRequest)
                        {
                            var channelParam = $"@p{Guid.NewGuid().ToString("N")}";
                            command.Parameters.AddWithValue($"{channelParam}", channel.Key);

                            foreach (var timeRange in channel.Value)
                            {
                                // Verify time range is valid
                                if (timeRange.StartDate <= timeRange.EndDate)
                                {
                                    var startParam = $"@p{Guid.NewGuid().ToString("N")}";
                                    var endParam = $"@p{Guid.NewGuid().ToString("N")}";
                                    command.Parameters.AddWithValue(startParam, timeRange.StartDate);
                                    command.Parameters.AddWithValue(endParam, timeRange.EndDate);
                                    if(sqlBuilder.Length == 0)
                                    {
                                        sqlBuilder.AppendLine($@"(channel_id = {channelParam} and day >= {startParam} and day < {endParam})");
                                    }
                                    else
                                    {
                                        sqlBuilder.AppendLine($@"or (channel_id = {channelParam} and day >= {startParam} and day < {endParam})");
                                    }
                                }
                            }
                        }

                        command.CommandText =
                        $@"
                            select
                                channel_id::varchar,
                                day::datetime,
                                sum(coalesce(minutes_watched_total, 0))::float as daily_total_minutes_watched,
                                sum(coalesce(minutes_broadcast_total, 0))::float as daily_total_minutes_broadcast,
                                sum(
                                    coalesce(creator_ad_revenue, 0.0)
                                    + coalesce(creator_sub_revenue_total, 0.0)
                                    + coalesce(creator_bits_revenue_total, 0.0)
                                    + coalesce(creator_fuel_revenue, 0.0)
                                    + coalesce(creator_bounty_board_revenue, 0.0)
                                )::float as CreatorRevenueTotal
                            from cubes.affiliates_partners_daily_channel_summary 
                            where
                                1 = 1
                                and ({sqlBuilder.ToString()})
                            group by
                                channel_id,
                                day
                            ;
                        ";

                        Log.Verbose($@"{command.CommandText}");
                        if(Constants.AppConfig.Application.Environment == "Development")
                        {
                            var paramBuilder = new StringBuilder();
                            foreach(Npgsql.NpgsqlParameter param in command.Parameters)
                            {
                                paramBuilder.AppendLine($@"{param.ParameterName} :: {param.Value}");
                            }
                            Log.Verbose(paramBuilder.ToString());
                        }

                        using (var reader = new DataReaderWithMeasurements(command, null, "get_arbitrary_summary_data").NpgReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    var item = new TwitchDirectArbitrarySummaryModel()
                                    {
                                        TwitchChannelID = reader.GetString(0),
                                        Day = reader.GetDateTime(1),
                                        HoursWatched = (float)reader.GetDouble(2),
                                        HoursBroadcast = (float)reader.GetDouble(3),
                                        TotalRevenue = (float)reader.GetDouble(4)
                                    };
                                    items.Add(item);
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return items;
        }
    }
}
