﻿using System;
using System.Collections.Generic;
using System.Text;
using Resonance.Core;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ServiceModels.TwitchModels;

namespace Resonance.Microservices.Queries
{
    /// <summary>
    /// INTERNAL methods to handle processing data
    /// </summary>
    internal class TwitchUserQuery
    {
        private enum filterType
        {
            Unknown = 0,
            TwitchUserID = 1,
            TwitchLogin = 2
        }

        internal string GetBaseQuery()
        {
            return $@"select twitch_user_id, twitch_login, is_active, inactive_status from {Constants.DatabaseSchema}microservice_twitch_user_lookup";
        }

        internal void Initialize()
        {
        }

        internal TwitchUserModel GetModel(long? twitchUserID = null, string twitchLogin = null)
        {
            TwitchUserModel result = null;

            try
            {
                if (twitchUserID == null && string.IsNullOrWhiteSpace(twitchLogin))
                {
                    throw new ArgumentNullException("One parameter must be passed. `twitchUserID`, `twitchLogin`");
                }
                filterType filter = filterType.Unknown;
                string whereQueryFilter = string.Empty;
                if (twitchUserID != null)
                {
                    filter = filterType.TwitchUserID;
                    whereQueryFilter = @"twitch_user_id = @twitchuserid limit 1;";
                }
                else if (twitchLogin != null)
                {
                    filter = filterType.TwitchLogin;
                    whereQueryFilter = @"twitch_login = @twitchlogin limit 1;";
                }
                else
                {
                    throw new NotImplementedException("Unspecified `whereQueryFilter` for TwitchUser GetModel");
                }

                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 60;
                        command.CommandText =
                        $@"
                            {this.GetBaseQuery()}
                            where
                            {whereQueryFilter}
                        ";
                        switch (filter)
                        {
                            case filterType.TwitchUserID:
                            {
                                command.Parameters.AddWithValue("@twitchuserid", twitchUserID.Value);
                                break;
                            }
                            case filterType.TwitchLogin:
                            {
                                command.Parameters.AddWithValue("@twitchlogin", twitchLogin);
                                break;
                            }
                            default:
                            {
                                break;
                            }
                        }
                        using (var wrappedreader = new DataReaderWithMeasurements(command, null, "TwitchUserQueryGetModel"))
                        {
                            var reader = wrappedreader.NpgReader;
                            if (reader.HasRows)
                            {
                                reader.Read();
                                result = new TwitchUserModel()
                                {
                                    TwitchUserID = reader.GetInt64(0),
                                    TwitchLogin = reader.GetValue(1) as string,
                                    IsActive = reader.GetBoolean(2),
                                    InactiveReason = (Constants.TwitchInactiveStatus)reader.GetInt32(3)
                                };
                            }
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        internal IEnumerable<TwitchUserModel> GetListModel(long[] twitchUserIDs = null, string[] twitchLogins = null)
        {
            List<TwitchUserModel> result = null;

            try
            {
                if ((twitchUserIDs == null || twitchUserIDs.Length == 0)
                    && (twitchLogins == null || twitchLogins.Length == 0))
                {
                    throw new ArgumentNullException("One parameter must be passed. `twitchUserIDs`, `twitchLogins`");
                }

                filterType filter = filterType.Unknown;
                string whereQueryFilter = string.Empty;

                if (twitchUserIDs != null)
                {
                    filter = filterType.TwitchUserID;

                    var parameters = new List<string>();
                    for (var i = 0; i < twitchUserIDs.Length; i++)
                    {
                        parameters.Add($"@twitchuserid_{i}");
                    }
                    whereQueryFilter = $"twitch_user_id in ({string.Join(",", parameters)});";
                }
                else if (twitchLogins != null)
                {
                    filter = filterType.TwitchLogin;
                    var parameters = new List<string>();
                    for (var i = 0; i < twitchLogins.Length; i++)
                    {
                        parameters.Add($"@twitchlogin_{i}");
                    }
                    whereQueryFilter = $"twitch_login in ({string.Join(",", parameters)});";
                }
                else
                {
                    throw new NotImplementedException("Unspecified `whereQueryFilter` for TwitchUser GetListModel");
                }

                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 60;

                        command.CommandText =
                        $@"
                            {this.GetBaseQuery()}
                            where
                            {whereQueryFilter}
                        ";

                        if (filter == filterType.TwitchUserID)
                        {
                            for (var i = 0; i < twitchUserIDs.Length; i++)
                            {
                                command.Parameters.AddWithValue($"@twitchuserid_{i}", twitchUserIDs[i]);
                            }
                        }
                        else if (filter == filterType.TwitchLogin)
                        {
                            for (var i = 0; i < twitchLogins.Length; i++)
                            {
                                command.Parameters.AddWithValue($"@twitchlogin_{i}", twitchLogins[i]);
                            }
                        }
                        else
                        {
                            throw new NotImplementedException("Unspecified `whereQueryFilter` for TwitchUserLink GetListModel");
                        }

                        using (var wrappedreader = new DataReaderWithMeasurements(command, null, "TwitchUserQueryGetListModel"))
                        {
                            var reader = wrappedreader.NpgReader;
                            if (reader.HasRows)
                            {
                                result = new List<TwitchUserModel>();
                                while (reader.Read())
                                {
                                    var link = new TwitchUserModel()
                                    {
                                        TwitchUserID = reader.GetInt64(0),
                                        TwitchLogin = reader.GetValue(1) as string,
                                        IsActive = reader.GetBoolean(2),
                                        InactiveReason = (Constants.TwitchInactiveStatus)reader.GetInt32(3)
                                    };
                                    result.Add(link);
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }

            return result;
        }

        internal bool? UpdateInactiveUser(ref string schemaprefix, ref TwitchBannedOrDeletedUserModel data)
        {
            try
            {
                if (data == null || data.UserID <= 0)
                {
                    return null;
                }

                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 600;
                        command.CommandText = $@"update {schemaprefix}microservice_twitch_user_lookup set is_active = False, inactive_status = @inactiveStatus where twitch_user_id = @twitchUserID;";
                        command.Parameters.AddWithValue("@twitchUserID", data.UserID);
                        command.Parameters.AddWithValue("@inactiveStatus", (int)data.Reason);
                        command.ExecuteNonQueryWithMeasurements("UpdateInactiveUser");
                    }
                }
                return true;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return false;
        }

        internal bool? InsertUpdateTwitchUser(ref string schemaprefix, ref TwitchUserModel user)
        {
            try
            {
                if(user == null || user.TwitchUserID <= 0)
                {
                    return null;
                }
                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {

                        command.CommandTimeout = 600;
                        command.CommandText =
                        $@"
                            update {schemaprefix}microservice_twitch_user_lookup
                                set twitch_login = @twitchlogin,
                                is_active = @isactive,
                                inactive_status = @inactivereason
                            from {schemaprefix}microservice_twitch_user_lookup as a
                            where 
                                twitch_user_id = @twitchUserID
                                and 
                                (
                                    isnull(a.twitch_user_id, 0) <> 0 
                                    and isnull(a.twitch_login, '') <> isnull(@twitchLogin, '')
                                )
                            ;

                            insert into {schemaprefix}microservice_twitch_user_lookup
                            select @twitchUserID, @twitchLogin, @isactive, @inactivereason
                            where not exists
                            (
                                select 1
                                from {schemaprefix}microservice_twitch_user_lookup as b
                                where b.twitch_user_id = @twitchUserID
                            );
                        ";
                        command.Parameters.AddWithValue("@twitchUserID", user.TwitchUserID);
                        command.Parameters.AddWithValue("@twitchLogin", user.TwitchLogin);
                        command.Parameters.AddWithValue("@isactive", user.IsActive);
                        command.Parameters.AddWithValue("@inactivereason", (int)user.InactiveReason);
                        command.ExecuteNonQueryWithMeasurements("InsertUpdateTwitchUser");
                        command.Parameters.Clear();
                        return true;
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return false;
        }

        internal long? InsertUpdateBulkTwitchUsers(ref string s3Bucket, ref string filepath, ref string schemaprefix, ref string workerIdentifier, ref string batchID, ref TwitchUserModel[] data)
        {
            long? result = 0;
            try
            {
                DBManagerRedshift.ImportToRedshift(s3Bucket, filepath, $"{schemaprefix}microservice_staging_twitch_user_lookup", dateFormat: Constants.RedshiftDateFormat, compressionType: "gzip", maxerrors: 0);
                Log.Info($"Import To Redshift: {filepath}");
                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 600;
                        var tempTableName = $"tmp_tul_{workerIdentifier}_{batchID}";
                        command.CommandText =
                        $@"
                            create temp table {tempTableName}
                            (
                                twitch_user_id bigint not null,
                                twitch_login varchar(50) not null,
                                is_active bool not null,
                                inactive_status int null,
                                WorkerIdentifier varchar(255) not null,
                                BatchID varchar(100) not null,
                                BatchStart datetime not null
                            );

                            insert into {tempTableName}(twitch_user_id, twitch_login, is_active, inactive_status, WorkerIdentifier, BatchID, BatchStart)
                            select 
                                a.twitch_user_id, a.twitch_login, a.is_active, a.inactive_status, @WorkerIdentifier, @BatchID, max(b.MaxBatch)
                            from {schemaprefix}microservice_staging_twitch_user_lookup as a
                            inner join
                            (
                                select twitch_user_id, max(BatchStart) as MaxBatch 
                                from {schemaprefix}microservice_staging_twitch_user_lookup
                                where WorkerIdentifier = @WorkerIdentifier
                                and BatchID = @BatchID
                                group by twitch_user_id
                            ) as b
                            on a.twitch_user_id = b.twitch_user_id
                                and a.BatchStart = b.MaxBatch
                            where 
                                WorkerIdentifier = @WorkerIdentifier
                                and BatchID = @BatchID
                            group by a.twitch_user_id, a.twitch_login, a.is_active, a.inactive_status;

                            update {schemaprefix}microservice_twitch_user_lookup
                                set twitch_login = b.twitch_login,
                                is_active = b.is_active,
                                inactive_status = b.inactive_status
                            from {schemaprefix}microservice_twitch_user_lookup as a
                            inner join {tempTableName} as b
                                on a.twitch_user_id = b.twitch_user_id
                            where 
                                (isnull(a.twitch_user_id, 0) <> 0
                                and 
                                (
                                    isnull(a.twitch_login, '') <> isnull(b.twitch_login, ''))
                                    or a.is_active <> b.is_active
                                    or a.inactive_status <> b.inactive_status
                                )
                            ;

                            insert into {schemaprefix}microservice_twitch_user_lookup
                            select a.twitch_user_id, a.twitch_login, a.is_active, a.inactive_status
                            from {tempTableName} as a
                            where not exists
                            (
                                select 1
                                from {schemaprefix}microservice_twitch_user_lookup as b
                                where a.twitch_user_id = b.twitch_user_id
                            );

                            drop table {tempTableName};

                            delete from {schemaprefix}microservice_staging_twitch_user_lookup where WorkerIdentifier = @WorkerIdentifier and BatchID = @BatchID;
                        ";
                        command.Parameters.AddWithValue("@WorkerIdentifier", workerIdentifier);
                        command.Parameters.AddWithValue("@BatchID", batchID);
                        command.ExecuteNonQueryWithMeasurements("InsertUpdateBulkTwitchUsers");
                        command.Parameters.Clear();
                        Log.Info($"Redshift Import Complete: {filepath}");
                    }
                }
                result = data.Length;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        internal long? UpdateInactiveUsers(ref string schemaprefix, ref List<TwitchBannedOrDeletedUserModel> batch)
        {
            long? resultCount = null;

            try
            {
                if(batch == null || batch.Count == 0)
                {
                    return null;
                }

                var queryBuilder = new StringBuilder();
                var queryParams = new Dictionary<string, Constants.TwitchInactiveStatus>();
                var userParams = new Dictionary<string, long>();
                var counter = 0;
                foreach(var user in batch)
                {
                    queryBuilder.AppendLine($@"update {schemaprefix}microservice_twitch_user_lookup set is_active = False, inactive_status = @inactiveStatus_{counter} where twitch_user_id = @twitchUser_{counter};");
                    queryParams.Add($@"@inactiveStatus_{counter}", user.Reason);
                    userParams.Add($@"@twitchUser_{counter}", user.UserID);
                    counter++;
                }

                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 600;
                        command.CommandText = queryBuilder.ToString();
                        foreach(var queryParam in queryParams)
                        {
                            command.Parameters.AddWithValue(queryParam.Key, (int)queryParam.Value);
                        }
                        foreach(var userParam in userParams)
                        {
                            command.Parameters.AddWithValue(userParam.Key, userParam.Value);
                        }
                        command.ExecuteNonQueryWithMeasurements("UpdateInactiveUsers");
                        resultCount += batch.Count;
                        queryParams.Clear();
                        userParams.Clear();
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }

            return resultCount;
        }
    }
}
