﻿using Resonance.Core;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using System;
using System.Collections.Generic;
using System.Linq;
using Resonance.Core.Models.ServiceModels.TwitchModels;

namespace Resonance.Microservices.Queries
{
    /// <summary>
    /// INTERNAL methods to handle processing data
    /// </summary>
    internal class TwitchVerifiedLinkQuery
    {
        internal void Initialize()
        {
        }

        private enum filterType
        {
            Unknown = 0,
            YoutubeTag = 1,
            TwitchUserID = 2,
            TwitchLogin = 3,
            GetAllRecords = 99
        }

        /// <summary>
        /// Select base query. Does not include 'Where' by default
        /// </summary>
        internal string GetBaseQuery()
        {
            return
            $@"
                select
                  TwitchUserID,
                  YoutubeTag,
                  TwitchLogin,
                  AddedBy,
                  LastUpdated
                from {Constants.DatabaseSchema}microservice_twitch_verified_link
            ";
        }

        /// <summary>
        /// Delete base query, includes 'Where' in the query by default. Don't you dare remove it.
        /// </summary>
        /// <returns></returns>
        internal string DeleteBaseQuery()
        {
            return 
            $@"
                delete from {Constants.DatabaseSchema}microservice_twitch_verified_link
                where
            ";
        }

        internal TwitchVerifiedLinkModel GetModel(long? twitchUserID = null, string twitchLogin = null, string youtubeTag = null)
        {
            TwitchVerifiedLinkModel result = null;
            try
            {
                if(twitchUserID == null && string.IsNullOrWhiteSpace(twitchLogin) && string.IsNullOrWhiteSpace(youtubeTag))
                {
                    throw new ArgumentNullException("One parameter must be passed. `twitchUserID`, `twitchLogin`, `youtubeTag`");
                }
                filterType filter = filterType.Unknown;
                string whereQueryFilter = string.Empty;
                if(youtubeTag != null)
                {
                    filter = filterType.YoutubeTag;
                    whereQueryFilter = @"YoutubeTag = @youtubetag limit 1;";
                }
                else if(twitchUserID != null)
                {
                    filter = filterType.TwitchUserID;
                    whereQueryFilter = @"TwitchUserID = @twitchuserid limit 1;";
                }
                else if(twitchLogin != null)
                {
                    filter = filterType.TwitchLogin;
                    whereQueryFilter = @"TwitchLogin = @twitchlogin limit 1;";
                }
                else
                {
                    throw new NotImplementedException("Unspecified `whereQueryFilter` for TwitchVerifiedLink GetModel");
                }

                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 60;
                        command.CommandText =
                        $@"
                            {GetBaseQuery()}
                            where
                            {whereQueryFilter}
                        ";
                        switch (filter)
                        {
                            case filterType.YoutubeTag:
                            {
                                command.Parameters.AddWithValue("@youtubetag", youtubeTag);
                                break;
                            }
                            case filterType.TwitchUserID:
                            {
                                command.Parameters.AddWithValue("@twitchuserid", twitchUserID.Value);
                                break;
                            }
                            case filterType.TwitchLogin:
                            {
                                command.Parameters.AddWithValue("@twitchlogin", twitchLogin);
                                break;
                            }
                            default:
                            {
                                break;
                            }
                        }
                        using (var wrappedreader = new DataReaderWithMeasurements(command, null, "TwitchVerifiedLinkQueryGetModel"))
                        {
                            var reader = wrappedreader.NpgReader;
                            if (reader.HasRows)
                            {
                                reader.Read();
                                result = new TwitchVerifiedLinkModel()
                                {
                                    TwitchUserID = reader.GetValue(0) as long?,
                                    YoutubeTag = reader.GetString(1),
                                    TwitchLogin = reader.GetString(2),
                                    AddedBy = reader.GetString(3),
                                    LastUpdated = reader.GetDateTime(4)
                                };
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        internal IEnumerable<TwitchVerifiedLinkModel> GetListModel(long[] twitchUserIDs = null, string[] twitchLogins = null, string[] youtubeTags = null, bool getAllRecords = false)
        {
            List<TwitchVerifiedLinkModel> result = null;
            try
            {
                if 
                (
                    (twitchUserIDs == null || twitchUserIDs.Length == 0)
                    && (twitchLogins == null || twitchLogins.Length == 0)
                    && (youtubeTags == null || youtubeTags.Length == 0)
                )
                {
                    if (!getAllRecords)
                    {
                        throw new ArgumentNullException("One parameter must be passed. `twitchUserIDs`, `twitchLogins`, `youtubeTags`");
                    }
                }
                filterType filter = filterType.Unknown;
                string whereQueryFilter = string.Empty;
                if (youtubeTags != null)
                {
                    filter = filterType.YoutubeTag;

                    var parameters = new List<string>();
                    for (var i = 0; i < youtubeTags.Length; i++)
                    {
                        parameters.Add($"@youtubetag_{i}");
                    }
                    whereQueryFilter = $"YoutubeTag in ({string.Join(",", parameters)});";
                }
                else if (twitchUserIDs != null)
                {
                    filter = filterType.TwitchUserID;

                    var parameters = new List<string>();
                    for (var i = 0; i < twitchUserIDs.Length; i++)
                    {
                        parameters.Add($"@twitchuserid_{i}");
                    }
                    whereQueryFilter = $"TwitchUserID in ({string.Join(",", parameters)});";
                }
                else if (twitchLogins != null)
                {
                    filter = filterType.TwitchLogin;
                    var parameters = new List<string>();
                    for (var i = 0; i < twitchLogins.Length; i++)
                    {
                        parameters.Add($"@twitchlogin_{i}");
                    }
                    whereQueryFilter = $"TwitchLogin in ({string.Join(",", parameters)});";

                }
                else if (getAllRecords == true)
                {
                    filter = filterType.GetAllRecords;
                }
                else
                {
                    throw new NotImplementedException("Unspecified `whereQueryFilter` for TwitchVerifiedLink GetListModel");
                }

                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 60;
                        if (getAllRecords)
                        {
                            command.CommandText = $@"{GetBaseQuery()}";
                        }
                        else
                        {
                            command.CommandText =
                            $@"
                                {GetBaseQuery()}
                                where
                                {whereQueryFilter}
                            ";
                        }
                        
                        if (filter == filterType.YoutubeTag)
                        {
                            for (var i = 0; i < youtubeTags.Length; i++)
                            {
                                command.Parameters.AddWithValue($"@youtubetag_{i}", youtubeTags[i]);
                            }
                        }
                        else if (filter == filterType.TwitchUserID)
                        {
                            for (var i = 0; i < twitchUserIDs.Length; i++)
                            {
                                command.Parameters.AddWithValue($"@twitchuserid_{i}", twitchUserIDs[i]);
                            }
                        }
                        else if (filter == filterType.TwitchLogin)
                        {
                            for (var i = 0; i < twitchLogins.Length; i++)
                            {
                                command.Parameters.AddWithValue($"@twitchlogin_{i}", twitchLogins[i]);
                            }
                        }
                        else
                        {
                            if (!getAllRecords)
                            {
                                throw new NotImplementedException("Unspecified `whereQueryFilter` for TwitchVerifiedLink GetModel");
                            }
                        }

                        using (var wrappedreader = new DataReaderWithMeasurements(command, null, "TwitchVerifiedLinkQueryGetListModel"))
                        {
                            var reader = wrappedreader.NpgReader;
                            if (reader.HasRows)
                            {
                                result = new List<TwitchVerifiedLinkModel>();
                                while (reader.Read())
                                {
                                    var link = new TwitchVerifiedLinkModel()
                                    {
                                        TwitchUserID = reader.GetValue(0) as long?,
                                        YoutubeTag = reader.GetString(1),
                                        TwitchLogin = reader.GetString(2),
                                        AddedBy = reader.GetString(3),
                                        LastUpdated = reader.GetDateTime(4)
                                    };
                                    result.Add(link);
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return result;
        }

        internal bool InsertUpdateBulkTwitchVerifiedLink(string s3Bucket, string filepath, string schemaprefix, string workerIdentifier, string batchID)
        {
            try
            {
                DBManagerRedshift.ImportToRedshift(s3Bucket, filepath, $"{schemaprefix}microservice_staging_twitch_verified_link", dateFormat: Constants.RedshiftDateFormat, compressionType: "gzip", maxerrors: 0);
                Log.Info($"Import To Redshift: {filepath}");
                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {

                        command.CommandTimeout = 600;
                        var tempTableName = $"tmp_tvl_{workerIdentifier}_{batchID}";
                        command.CommandText =
                        $@"
                            create temp table {tempTableName}
                            (
                                TwitchUserID bigint null, 
                                YoutubeTag varchar(100) not null, 
                                TwitchLogin varchar(100) null,
                                AddedBy varchar(100) not null, 
                                LastUpdated datetime not null,
                                WorkerIdentifier varchar(255) not null,
                                BatchID varchar(100) not null,
                                BatchStart datetime not null
                            );
                            insert into {tempTableName}(TwitchUserID, YoutubeTag, TwitchLogin, AddedBy, LastUpdated, WorkerIdentifier, BatchID, BatchStart)
                            select 
                                a.TwitchUserID, a.YoutubeTag, a.TwitchLogin, a.AddedBy, 
                                max(a.LastUpdated) as LastUpdated,
                                a.WorkerIdentifier, a.BatchID, max(b.MaxBatch) as batchStart
                            from {schemaprefix}microservice_staging_twitch_verified_link as a
                            inner join
                            (
                                select YoutubeTag, max(BatchStart) as MaxBatch 
                                from {schemaprefix}microservice_staging_twitch_verified_link
                                where WorkerIdentifier = @WorkerIdentifier
                                and BatchID = @BatchID
                                group by YoutubeTag
                            ) as b
                            on a.YoutubeTag = b.YoutubeTag
                                and a.BatchStart = b.MaxBatch
                            where 
                                WorkerIdentifier = @WorkerIdentifier
                                and BatchID = @BatchID
                            group by a.TwitchUserID, a.YoutubeTag, a.TwitchLogin, a.AddedBy, a.WorkerIdentifier, a.BatchID;

                            update {schemaprefix}microservice_twitch_verified_link
                                set TwitchUserID = b.TwitchUserID,
                                TwitchLogin = b.TwitchLogin,
                                AddedBy = b.AddedBy,
                                LastUpdated = b.LastUpdated
                            from {schemaprefix}microservice_twitch_verified_link as a
                            inner join {tempTableName} as b
                                on a.YoutubeTag = b.YoutubeTag
                            where 
                                (isnull(a.TwitchUserID, 0) <> 0 
                                and isnull(a.TwitchLogin, '') <> isnull(b.TwitchLogin, ''))
                            ;

                            insert into {schemaprefix}microservice_twitch_verified_link
                            select a.TwitchUserID, a.YoutubeTag, a.TwitchLogin, a.AddedBy, a.LastUpdated
                            from {tempTableName} as a
                            where not exists
                            (
                                select 1
                                from {schemaprefix}microservice_twitch_verified_link as b
                                where a.YoutubeTag = b.YoutubeTag
                            );

                            drop table {tempTableName};

                            delete from {schemaprefix}microservice_staging_twitch_verified_link where WorkerIdentifier = @WorkerIdentifier and BatchID = @BatchID;
                        ";
                        command.Parameters.AddWithValue("@WorkerIdentifier", workerIdentifier);
                        command.Parameters.AddWithValue("@BatchID", batchID);
                        command.ExecuteNonQueryWithMeasurements("InsertUpdateBulkTwitchVerifiedLink");
                        command.Parameters.Clear();
                        Log.Info($"Redshift Import Complete: {filepath}");
                    }
                }
                return true;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return false;
        }

        internal bool DeleteByTwitchUserIDs(TwitchDeletedUserListModel deleteUsers)
        {
            try
            {
                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 60;
                        var deleteUserIDs = new List<string>();
                        command.CommandText = $@"{DeleteBaseQuery()} TwitchUserID in ({string.Join(",", deleteUsers.DeletedUsers.Select(x => x.TwitchUserID))})";
                        command.ExecuteNonQueryWithMeasurements("DeleteByTwitchUserIDs");
                    }
                }
                return true;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return false;
        }

        internal bool DeleteByYoutubeTags(TwitchDeletedUserListModel deleteUsers)
        {
            try
            {
                if(deleteUsers == null || deleteUsers.DeletedUsers == null || deleteUsers.DeletedUsers.Length == 0)
                {
                    return false;
                }

                var counter = 0;
                var batch = new Dictionary<string, string>();
                // Batch process deleted users
                foreach(var user in deleteUsers.DeletedUsers)
                {
                    if (user.YoutubeTag.StartsWith("UC"))
                    {
                        batch.Add($"@youtubeTag_{counter}", user.YoutubeTag);
                        counter++;
                        if (counter == 100)
                        {
                            ExecuteYoutubeTagBatchDelete(ref batch);
                            counter = 0;
                            batch.Clear();
                        }
                    }
                }
                // Tail cleanup
                if(batch.Count > 0)
                {
                    ExecuteYoutubeTagBatchDelete(ref batch);
                    counter = 0;
                    batch.Clear();
                }
                return true;
            }
            catch (Exception ex)
            {
                Log.Error( ex);
            }
            return false;
        }

        private void ExecuteYoutubeTagBatchDelete(ref Dictionary<string, string> batch)
        {
            try
            {
                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 60;
                        var deleteUserIDs = new List<string>();
                        command.CommandText = $@"{DeleteBaseQuery()} YoutubeTag in ({string.Join(",", batch.Keys)})";
                        foreach(var user in batch)
                        {
                            command.Parameters.AddWithValue(user.Key, user.Value);
                        }
                        command.ExecuteNonQueryWithMeasurements("ExecuteYoutubeTagBatchDelete");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }
    }
}
