﻿using MySql.Data.MySqlClient;
using Resonance.Core;
using Resonance.Core.Helpers.LoggingHelpers;
using System;
using static Resonance.Core.Constants;

namespace Resonance.Microservices.Queries
{
    public static class AmpQuerySql
    {
        /**** SERVICE QUERIES ****/
        public static MySqlCommand LoadFileFromS3(string bucket, string keypath, string table, string quotechar = "\"", string ignore = "", string fields = "", string lineterm = "\\r\\n", bool allowAutomaticLineTerminationChange = true)
        {
            Log.Verbose($@"Loading {bucket}/{keypath} to {table} {fields}");
            string manifest = "";
            if (keypath.EndsWith("manifest"))
            {
                Log.Info($"LoadFileFromS3 detected manifest file for {bucket}/{keypath}. Flagging as a manifest data load.");
                manifest = "manifest";
                if (allowAutomaticLineTerminationChange)
                {
                    Log.Info($"Changing to \\n line terminator");
                    lineterm = "\\n";
                }
            }
            MySqlCommand command = new MySqlCommand()
            {
                CommandText =
                $@"
                    load data from s3 {manifest} 's3://{bucket}/{keypath}'
                    replace
                    into table {table}
                    character set 'utf8'
                    fields terminated by ','
                    enclosed by '{quotechar}'
                    lines terminated by '{lineterm}'
                    {ignore}
                    {fields}
                    ;
                "
            };
            return command;
        }

        public static MySqlCommand InsertUpdateEtlTracking(string source, string target, string method, DateTime timestamp, long rowcount)
        {
            MySqlCommand command = new MySqlCommand()
            {
                CommandText =
                $@"
                    replace into {Constants.DatabaseSchema}microservice_etl_tracking (Source, Target, Method, LastUpdateDate, RowCount)
                    select
	                    @source as Source,
                        @target as Target,
                        @method as Method,
                        @timestamp as LastUpdateDate,
                        @rowcount as RowCount;
                ",
            };
            command.Parameters.AddWithValue("@source", source);
            command.Parameters.AddWithValue("@target", target);
            command.Parameters.AddWithValue("@method", method);
            command.Parameters.AddWithValue("@timestamp", timestamp);
            command.Parameters.AddWithValue("@rowcount", rowcount);
            return command;
        }

        public static MySqlCommand ResultCountTwitchUserListing(AggregationType aggregationType)
        {
            var ts = DateTime.UtcNow.Date.AddDays(-3);
            MySqlCommand command = new MySqlCommand()
            {
                CommandText = 
                $@"
                    select count(*) as cnt
                    from {Constants.DatabaseSchema}microservice_twitch_user_listing 
                    where
                        MaxDate >= @ts
                        and AggregationType = {(int)aggregationType}
                ;"
            };
            command.Parameters.AddWithValue("@ts", ts);
            return command;
        }

        /**** TWITCH GAMES ****/
        public static MySqlCommand DropTwitchGamesTempTable()
        {
            MySqlCommand command = new MySqlCommand()
            {
                CommandText = "drop table if exists temp_microservice_twitch_games;create temporary table temp_microservice_twitch_games(like development.microservice_twitch_games);"
            };
            return command;
        }

        public static MySqlCommand InsertTwitchGames()
        {
            MySqlCommand command = new MySqlCommand()
            {
                CommandText = 
                $@"
                    insert into {Constants.DatabaseSchema}microservice_twitch_games
                    select * from temp_microservice_twitch_games
                    where temp_microservice_twitch_games.game_id 
                    not in 
                    (
                        select game_id from {Constants.DatabaseSchema}microservice_twitch_games
                    );
                    drop table temp_microservice_twitch_games;
                "
            };
            return command;
        }
    }
}
