# AWS region for the S3 bucket
$region = "us-west-2"
# Name of the bucket
$bucket = "crsmcn-resonance-deploy"
# Prefix for the deployment archives
$keyPrefixArchive = "archives"
# Environment
$environment = $args[0]
# Service name
$serviceName = $args[1]
if ($serviceName -eq 'Atlas_Jobs') {
    $serviceName = 'Jobs\Atlas'
}
# Local path to download the archive to
$localPath = "C:\Resonance\"
# Filename of the local archive
$localFilename = "deploy.zip"
# Directory to deploy to
$deployToPath = "C:\Resonance"
# S3 key prefix for the deployment archive
$keyPrefix = $keyPrefixArchive + "/" + $environment + "/" + $serviceName.ToLower() + "/"
if ($serviceName -eq 'Jobs\Atlas') {
    $keyPrefix = $keyPrefixArchive + "/" + $environment + "/" + 'jobs/atlas' + "/"
}
# Local deployment archive
$localFile = $localPath + $localFilename
# IIS Application Pool name
$poolName = "Resonance " + $serviceName
# web.config path
$webConfigPath = "C:\Resonance\" + $environment + "\" + $serviceName + "\web.config"
# Initialize key var
$key = ""
# Max retries for waiting for IIS to stop/start
$maxRetries = 5

import-module WebAdministration

# Pull in the Web Administration services
[Void][Reflection.Assembly]::LoadWithPartialName("Microsoft.Web.Administration")

# Add a helper function to unzip the archive once we've downloaded it
Add-Type -AssemblyName System.IO.Compression.FileSystem
function Unzip
{
    param([string]$zipfile, [string]$outpath)
    [System.IO.Compression.ZipFile]::ExtractToDirectory($zipfile, $outpath)
}

# Grab the latest deployment key based on the last modified date of the objects in S3
$key = (Get-S3Object -BucketName $bucket -Key $keyPrefix | Sort-Object LastModified -Descending | Select-Object -First 1 | select Key | ft -hidetableheaders | Out-String).Trim()

# Set the directory we're deploying to
$deployTo = $deployToPath + "\" + $environment + "\" + $serviceName

# Download the archive
Write-Output ('Copying file from S3: {0}' -f $keyPrefix)
Copy-S3Object -BucketName $bucket -Key $key -LocalFile $localFile -Region $region

# Stop IIS Web App Pool, unless we're doing a Jobs deploy, where we stop Task Scheduler task instead
if ($serviceName -ne 'Jobs\Atlas') {
    if ((Get-WebAppPoolState -Name $poolName).Value -ne 'Stopped'){
        try {
            Write-Output ('Stopping Application Pool: {0}' -f $poolName)
            Stop-WebAppPool -Name $poolName
        } catch {
            Write-Output ("Restarting Credential Manager Service as IIS isn't responding to control messages: {0}" -f $poolName)
            Restart-Service VaultSvc
        }
    }
    
    $currentRetry = 0
    $success = $false
    
    do {
        $status = Get-WebAppPoolState -name $poolName
        if ($status.Value -eq "Stopped") {
            $success = $true;
            Write-Output ('Application Pool: {0} stopped.' -f $poolName)
        } else {
            Write-Output ('Waiting for Application Pool: {0} to stop.' -f $poolName)
            Start-Sleep -s 10
            $currentRetry = $currentRetry + 1;
        }
    } while (!$success -and $currentRetry -le $maxRetries)
} else {
    Write-Output ('Stopping Scheduled Task: "Resonance Jobs Atlas"')
    Stop-ScheduledTask -TaskName "Resonance Jobs Atlas" | Start-ScheduledTask
    Start-Sleep -s 5
}

# Check to see if there's an existing deploy of this version, delete it if so
if (Test-Path -path $deployTo) {
    Remove-Item -path $deployTo -Recurse -Force
}

# Unzip the archive from earlier
Unzip $localFile $deployTo

# Cleanup the downloaded archive
if (Test-Path $localFile) {
    Remove-Item $localFile
}

# Stop IIS Web App Pool, unless we're doing a Jobs deploy, where we stop Task Scheduler task instead
if ($serviceName -ne 'Jobs\Atlas') {
    # Also swap out environment in web config path if needed
    (Get-Content $webConfigPath) -replace '--environment=(\w+)', "--environment=$environment" | Out-File $webConfigPath -Encoding utf8
    
    if ((Get-WebAppPoolState -Name $poolName).Value -ne 'Started'){
        try {
            Write-Output ('Starting Application Pool: {0}' -f $poolName)
            Start-WebAppPool -Name $poolName
        } catch {
            Write-Output ("Restarting Credential Manager Service as IIS isn't responding to control messages: {0}" -f $poolName)
            Restart-Service VaultSvc
        }
    }
    
    $currentRetry = 0
    $success = $false
    
    do {
        $status = Get-WebAppPoolState -name $poolName
        if ($status.Value -eq "Started") {
            $success = $true;
            Write-Output ('Application Pool: {0} started.' -f $poolName)
        } else {
            Write-Output ('Waiting for Application Pool: {0} to start.' -f $poolName)
            Start-Sleep -s 10
            $currentRetry = $currentRetry + 1;
        }
    } while (!$success -and $currentRetry -le $maxRetries)
} else {
    Start-Sleep -s 5
    Write-Output ('Starting Scheduled Task: "Resonance Jobs Atlas"')
    Get-ScheduledTask -TaskName "Resonance Jobs Atlas" | Start-ScheduledTask
}