package v1

import "encoding/json"

// NullString is a custom type for JSON string fields that could be one of the following:
// - Not present in JSON
// - Present but "null" in JSON
// - Present with string value
type NullString struct {
	// Present indicates whether the field was present in the JSON string.
	Present bool

	// Value is the underlying value.
	Value *string
}

// MarshalJSON JSON-marshals the Value.
func (i *NullString) MarshalJSON() ([]byte, error) {
	return json.Marshal(i.Value)
}

// UnmarshalJSON distinguishes between the JSON field's
// being present but "null", or not present.
func (i *NullString) UnmarshalJSON(data []byte) error {
	i.Present = true

	if string(data) == "null" {
		return nil
	}

	var value string
	if err := json.Unmarshal(data, &value); err != nil {
		return err
	}

	i.Value = &value

	return nil
}
