package v1

import (
	"encoding/json"
	"testing"
)

func TestNullStringMarshalJSON(t *testing.T) {
	value := "1234567890"

	type wrapper struct {
		NullString *NullString `json:",omitempty"`
	}

	tests := []struct {
		NullString *NullString
		Expected   string
	}{
		{
			NullString: nil,
			Expected:   `{}`,
		},
		{
			NullString: &NullString{
				Present: true,
				Value:   nil,
			},
			Expected: `{"NullString":null}`,
		},
		{
			NullString: &NullString{
				Present: true,
				Value:   &value,
			},
			Expected: `{"NullString":"1234567890"}`,
		},
	}

	for idx, test := range tests {
		marshaled, err := json.Marshal(&wrapper{
			NullString: test.NullString,
		})
		if err != nil {
			t.Errorf("expected no error, got: [%s]", err)
		}

		actual := string(marshaled)

		if actual != test.Expected {
			t.Errorf("index: %d | expected: [%s], got: [%s]", idx, test.Expected, actual)
		}
	}
}

func TestNullStringUnmarshalJSON(t *testing.T) {
	value := "1234567890"

	type wrapper struct {
		NullString NullString
	}

	tests := []struct {
		JSON     string
		Expected NullString
	}{
		{
			JSON: `{}`,
			Expected: NullString{
				Present: false,
			},
		},
		{
			JSON: `{"NullString":null}`,
			Expected: NullString{
				Present: true,
				Value:   nil,
			},
		},
		{
			JSON: `{"NullString":"1234567890"}`,
			Expected: NullString{
				Present: true,
				Value:   &value,
			},
		},
	}

	for idx, test := range tests {
		w := wrapper{}

		err := json.Unmarshal([]byte(test.JSON), &w)
		if err != nil {
			t.Errorf("expected no error, got: [%s]", err)
		}

		if !equalNullStrings(test.Expected, w.NullString) {
			t.Errorf("index: %d | expected: [%v], got: [%v]", idx, test.Expected, w.NullString)
		}
	}
}

func equalNullStrings(a NullString, b NullString) bool {
	if a.Present != b.Present {
		return false
	}

	if a.Value == nil {
		return b.Value == nil
	}

	if a.Value != nil && b.Value == nil {
		return false
	}

	return *a.Value == *b.Value
}
