package roster

import (
	"context"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
)

// Client is the client interface for the Roster service.
type Client interface {
	// GetV1Teams makes a "GET /v1/teams" request with query parameter(s)
	// to filter teams.
	GetV1Teams(ctx context.Context, params v1.GetTeamsRequestParams, opts *twitchclient.ReqOpts) (*v1.GetTeamsResponse, error)

	// PostV1Teams makes a "POST /v1/teams" request to create a new team.
	PostV1Teams(ctx context.Context, params v1.PostTeamsRequestParams, opts *twitchclient.ReqOpts) (*v1.PostTeamsResponse, error)

	// GetV1Team makes a "GET /v1/teams/:team_id" request
	// to update a given team.
	GetV1Team(ctx context.Context, teamID string, opts *twitchclient.ReqOpts) (*v1.GetTeamResponse, error)

	// PatchV1Team makes a "PATCH /v1/teams/:team_id" request
	// to update a given team.
	PatchV1Team(ctx context.Context, params v1.PatchTeamRequestParams, opts *twitchclient.ReqOpts) (*v1.PatchTeamResponse, error)

	// ===== TEAM IMAGE UPLOAD =====

	// PostV1ImageUploads make a POST /v1/teams/:team_id/image_uploads" request
	PostV1ImageUploads(ctx context.Context, params v1.PostImageUploadsRequestParams, opts *twitchclient.ReqOpts) (*v1.PostImageUploadsResponse, error)

	// ===== TEAM MEMBERSHIP =====

	// GetV1TeamMemberships makes a "GET /v1/teams/:team_id/members" request
	// for a given team's members.
	GetV1TeamMemberships(ctx context.Context, params v1.GetTeamMembershipsRequestParams, opts *twitchclient.ReqOpts) (*v1.GetTeamMembershipsResponse, error)

	// GetV1TeamLiveMemberships makes a "GET /v1/teams/:team_id/live_memberships" request
	// for a given team's live members.
	GetV1TeamLiveMemberships(ctx context.Context, params v1.GetTeamLiveMembershipsRequestParams, opts *twitchclient.ReqOpts) (*v1.GetTeamLiveMembershipsResponse, error)

	// DeleteV1TeamMembership make a "DELETE /v1/teams/:team_id/channels/:channel_id/membership" request
	DeleteV1TeamMembership(ctx context.Context, params v1.DeleteMembershipRequestParams, opts *twitchclient.ReqOpts) error

	// PatchV1TeamMembership make a PATCH /v1/teams/:team_id/channels/:channel_id/membership" request
	PatchV1TeamMembership(ctx context.Context, params v1.PatchTeamMembershipRequestParams, opts *twitchclient.ReqOpts) error

	// ===== TEAM INVITATION =====

	// GetV1TeamInvitations make a "GET /v1/teams/:team_id/invitations" request
	GetV1TeamInvitations(ctx context.Context, teamID string, opts *twitchclient.ReqOpts) (*v1.GetTeamInvitationsResponse, error)

	// PostV1TeamInvitations make a "POST /v1/teams/:team_id/invitations" request.
	PostV1TeamInvitations(ctx context.Context, params v1.PostTeamInvitationsRequestParams, opts *twitchclient.ReqOpts) (*v1.PostTeamInvitationsResponse, error)

	// DeleteV1TeamInvitation make a "DELETE /v1/teams/:team_id/channels/:channel_id/invitation" request.
	DeleteV1TeamInvitation(ctx context.Context, params v1.DeleteTeamInvitationRequestParams, opts *twitchclient.ReqOpts) error

	// ===== TEAM FEATURED CHANNEL =====

	// GetV1FeaturedChannels make a "GET /v1/teams/:team_id/featured_channels" request.
	GetV1FeaturedChannels(ctx context.Context, params v1.GetFeaturedChannelsRequestParams, opts *twitchclient.ReqOpts) (*v1.GetFeaturedChannelsResponse, error)

	// GetV1LiveFeaturedChannels make a "GET /v1/teams/:team_id/live_featured_channels" request.
	GetV1LiveFeaturedChannels(ctx context.Context, params v1.GetLiveFeaturedChannelsRequestParams, reqOpts *twitchclient.ReqOpts) (*v1.GetLiveFeaturedChannelsResponse, error)

	// PatchV1FeaturedChannel make a PATCH /v1/teams/:team_id/featured_channels/:channel_id" request
	PatchV1FeaturedChannel(ctx context.Context, params v1.PatchFeaturedChannelRequestParams, opts *twitchclient.ReqOpts) error

	// PostV1FeaturedChannels make a "POST /v1/teams/:team_id/featured_channels" request.
	PostV1FeaturedChannels(ctx context.Context, params v1.PostFeaturedChannelsRequestParams, opts *twitchclient.ReqOpts) (*v1.PostFeaturedChannelsResponse, error)

	// DeleteV1FeaturedChannel make a "DELETE /v1/teams/:team_id/featured_channels/:channel_id" request.
	DeleteV1FeaturedChannel(ctx context.Context, params v1.DeleteFeaturedChannelRequestParams, opts *twitchclient.ReqOpts) error

	// ===== CHANNEL INVITATION =====

	// GetV1ChannelInvitations make a "GET /v1/channels/:channel_id/invitations" request.
	GetV1ChannelInvitations(ctx context.Context, params v1.GetChannelInvitationsRequestParams, opts *twitchclient.ReqOpts) (*v1.GetChannelInvitationsResponse, error)

	// PutV1ChannelInvitation make a "PUT /v1/channels/:channel_id/teams/:teams_id/invitation" request
	PutV1ChannelInvitation(ctx context.Context, params v1.PutChannelInvitationRequestParams, opts *twitchclient.ReqOpts) error

	// ===== CHANNEL MEMBERSHIP =====

	// GetV1ChannelMemberships make a "GET /v1/channels/:channel_id/memberships" request.
	GetV1ChannelMemberships(ctx context.Context, params v1.GetChannelMembershipsRequestParams, opts *twitchclient.ReqOpts) (*v1.GetChannelMembershipsResponse, error)

	// PutV1ChannelMembership make a "PUT /v1/channels/:channel_id/teams/:teams_id/membership" request
	PutV1ChannelMembership(ctx context.Context, params v1.PutChannelMembershipRequestParams, opts *twitchclient.ReqOpts) error

	// DeleteV1ChannelMembership make a "DELETE /v1/channels/:channel_id/teams/:team_id/membership" request
	DeleteV1ChannelMembership(ctx context.Context, params v1.DeleteMembershipRequestParams, opts *twitchclient.ReqOpts) error
}

type clientImpl struct {
	twitchclient.Client
}

// NewClient instantiates a new client for the Roster service.
func NewClient(config twitchclient.ClientConf) (Client, error) {
	if config.TimingXactName == "" {
		config.TimingXactName = "cb-roster"
	}

	twitchClient, err := twitchclient.NewClient(config)
	if err != nil {
		return nil, err
	}

	return &clientImpl{
		Client: twitchClient,
	}, nil
}
