package roster

import (
	"context"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("DeleteV1TeamInvitation", func() {
	var (
		server *ghttp.Server
		client Client
		err    error
	)

	BeforeEach(func() {
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo(HaveOccurred())
	})

	AfterEach(func() {
		server.Close()
	})

	It("fails with invalid parameters", func() {
		params := v1.DeleteTeamInvitationRequestParams{
			TeamID:    "%qq",
			ChannelID: "channel-id",
		}
		err := client.DeleteV1TeamInvitation(context.Background(), params, nil)

		Expect(err).To(HaveOccurred())
		Expect(err.Error()).To(ContainSubstring("invalid"))

	})

	It("calls out to the roster URL", func() {
		server.AppendHandlers(
			ghttp.VerifyRequest("DELETE", "/v1/teams/team-id/channels/channel-id/invitation"),
		)

		params := v1.DeleteTeamInvitationRequestParams{
			TeamID:    "team-id",
			ChannelID: "channel-id",
		}
		_ = client.DeleteV1TeamInvitation(context.Background(), params, nil)

		receivedRequests := server.ReceivedRequests()
		Expect(receivedRequests).To(HaveLen(1))
	})

	It("returns nil on request rountrip success", func() {
		server.AppendHandlers(
			ghttp.CombineHandlers(
				ghttp.VerifyRequest("DELETE", "/v1/teams/team-id/channels/channel-id/invitation"),
				ghttp.RespondWith(http.StatusNoContent, ""),
			),
		)
		params := v1.DeleteTeamInvitationRequestParams{
			TeamID:    "team-id",
			ChannelID: "channel-id",
		}
		err := client.DeleteV1TeamInvitation(context.Background(), params, nil)

		Expect(err).NotTo(HaveOccurred())
	})

	It("returns error on failed delete", func() {
		server.AppendHandlers(
			ghttp.CombineHandlers(
				ghttp.VerifyRequest("DELETE", "/v1/teams/team-id/channels/channel-id/invitation"),
				ghttp.RespondWith(http.StatusInternalServerError, "Surprise Error"),
			),
		)

		params := v1.DeleteTeamInvitationRequestParams{
			TeamID:    "team-id",
			ChannelID: "channel-id",
		}
		err := client.DeleteV1TeamInvitation(context.Background(), params, nil)

		Expect(err).To(HaveOccurred())
		Expect(err.Error()).To(ContainSubstring("Surprise Error"))
	})
})
