package roster

import (
	"context"
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("GetV1ChannelInvitations", func() {
	var (
		server *ghttp.Server
		client Client
		err    error
	)

	BeforeEach(func() {
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo(HaveOccurred())
	})

	AfterEach(func() {
		server.Close()
	})

	Context("when roster returns a 400", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/channels/invalid-channel-id/invitations"),
					ghttp.RespondWith(http.StatusBadRequest, ""),
				),
			)
		})

		It("returns an error", func() {
			params := v1.GetChannelInvitationsRequestParams{
				ChannelID: "invalid-channel-id",
			}

			resp, err := client.GetV1ChannelInvitations(context.Background(), params, nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when roster returns a 500", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/channels/123/invitations"),
					ghttp.RespondWith(http.StatusInternalServerError, ""),
				),
			)
		})

		It("returns an error", func() {
			params := v1.GetChannelInvitationsRequestParams{
				ChannelID: "123",
			}

			resp, err := client.GetV1ChannelInvitations(context.Background(), params, nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when roster returns a 200", func() {
		var (
			teamID      string
			teamName    string
			channelID   string
			payload     string
			teamOwnerID string
			teamLogoID  string
			teamLogoURL string
		)

		BeforeEach(func() {
			teamID = "123"
			teamName = "dat_team_name"
			channelID = "123"
			teamOwnerID = "01"
			teamLogoID = "2"
			teamLogoURL = "image.src.com"

			payloadFmt := `
					{
						"data": [
							{
								"team": {
									"id": "%s",
									"name": "%s",
									"user_id": "%s",
									"logo_id": "%s",
									"logo_url": "%s"
								}
							}
						]
					}
				`
			payload = fmt.Sprintf(payloadFmt, teamID, teamName, teamOwnerID, teamLogoID, teamLogoURL)

			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/channels/123/invitations"),
					ghttp.RespondWith(http.StatusOK, payload),
				),
			)
		})

		It("returns a list of invitations", func() {
			expectedResponse := v1.GetChannelInvitationsResponse{
				Data: []v1.GetChannelInvitationsData{
					{
						Team: v1.Team{
							ID:          teamID,
							Name:        teamName,
							UserID:      teamOwnerID,
							DisplayName: "",
							LogoID:      &teamLogoID,
							LogoURL:     &teamLogoURL,
						},
					},
				},
			}

			params := v1.GetChannelInvitationsRequestParams{
				ChannelID: channelID,
			}

			resp, err := client.GetV1ChannelInvitations(context.Background(), params, nil)
			Expect(err).NotTo(HaveOccurred())

			Expect(resp).NotTo(BeNil())
			Expect(*resp).To(Equal(expectedResponse))
		})
	})
})
