package roster

import (
	"context"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("GetV1ChannelMemberships()", func() {
	var (
		server       *ghttp.Server
		client       Client
		params       v1.GetChannelMembershipsRequestParams
		expectedPath string
		err          error
	)

	BeforeEach(func() {
		expectedPath = "/v1/channels/13/memberships"
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo(HaveOccurred())
	})

	AfterEach(func() {
		server.Close()
	})

	Context("with a bad URL", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", expectedPath),
					ghttp.RespondWith(http.StatusNoContent, ""),
				),
			)

			params = v1.GetChannelMembershipsRequestParams{
				ChannelID: "%nothex",
			}
		})

		It("returns an error without making a request", func() {
			_, err = client.GetV1ChannelMemberships(context.Background(), params, nil)

			Expect(server.ReceivedRequests()).To(BeEmpty())
			Expect(err).To(HaveOccurred())
			Expect(err.Error()).To(ContainSubstring("invalid"))
		})
	})

	Context("with a valid URL", func() {
		BeforeEach(func() {
			params = v1.GetChannelMembershipsRequestParams{
				ChannelID: "13",
			}
		})

		Context("when roster returns an error", func() {
			BeforeEach(func() {
				server.AppendHandlers(
					ghttp.CombineHandlers(
						ghttp.VerifyRequest("GET", expectedPath),
						ghttp.RespondWith(http.StatusTeapot, "I'm a web-scale teapot"),
					),
				)
			})

			It("returns the error code and description", func() {
				_, err = client.GetV1ChannelMemberships(context.Background(), params, nil)

				Expect(server.ReceivedRequests()).To(HaveLen(1))
				Expect(err).To(HaveOccurred())
				Expect(err.Error()).To(Equal("418: I'm a web-scale teapot"))
			})
		})

		Context("when roster returns success", func() {
			BeforeEach(func() {
				server.AppendHandlers(
					ghttp.CombineHandlers(
						ghttp.VerifyRequest("GET", expectedPath),
						ghttp.RespondWith(http.StatusOK, `{
							"data": [
								{
									"primary": true,
									"revenue_revealed": true,
									"stats_revealed": false,
									"team": {
										"id": "10",
										"name": "staff",
										"user_id": "teamowner",
										"display_name": "Twitch Staff",
										"logo_id": "123",
										"logo_url": "image.src.com/123"
									}
								},
								{
									"primary": false,
									"revenue_revealed": true,
									"stats_revealed": false,
									"team": {
										"id": "11",
										"name": "asdf",
										"user_id": "asdfowner",
										"display_name": "ASDF",
										"logo_id": null,
										"logo_url": null
									}
								}
							]
						}`),
					),
				)
			})

			It("returns no error", func() {
				logoID := "123"
				logoURL := "image.src.com/123"

				expectedResponse := v1.GetChannelMembershipsResponse{
					Data: []v1.GetChannelMembershipsData{
						{
							Primary:         true,
							RevenueRevealed: true,
							StatsRevealed:   false,
							Team: v1.Team{
								ID:          "10",
								Name:        "staff",
								UserID:      "teamowner",
								DisplayName: "Twitch Staff",
								LogoID:      &logoID,
								LogoURL:     &logoURL,
							},
						},
						{
							Primary:         false,
							RevenueRevealed: true,
							StatsRevealed:   false,
							Team: v1.Team{
								ID:          "11",
								Name:        "asdf",
								UserID:      "asdfowner",
								DisplayName: "ASDF",
								LogoID:      nil,
								LogoURL:     nil,
							},
						},
					},
				}

				response, err := client.GetV1ChannelMemberships(context.Background(), params, nil)

				Expect(server.ReceivedRequests()).To(HaveLen(1))
				Expect(err).NotTo(HaveOccurred())
				Expect(*response).To(Equal(expectedResponse))
			})
		})
	})
})
