package roster

import (
	"context"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("GetV1FeaturedChannels", func() {
	var (
		server *ghttp.Server
		client Client
		err    error
	)

	BeforeEach(func() {
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo(HaveOccurred())
	})

	AfterEach(func() {
		server.Close()
	})

	It("fails with invalid parameters", func() {
		params := v1.GetFeaturedChannelsRequestParams{
			TeamID: "%qq",
		}
		_, err := client.GetV1FeaturedChannels(context.Background(), params, nil)

		Expect(err).To(HaveOccurred())
		Expect(err.Error()).To(ContainSubstring("invalid"))

	})

	It("calls out to the roster URL", func() {
		server.AppendHandlers(
			ghttp.VerifyRequest("GET", "/v1/teams/team-id/featured_channels"),
		)

		params := v1.GetFeaturedChannelsRequestParams{
			TeamID: "team-id",
		}
		_, _ = client.GetV1FeaturedChannels(context.Background(), params, nil)

		receivedRequests := server.ReceivedRequests()
		Expect(receivedRequests).To(HaveLen(1))
	})

	It("returns error on failed get", func() {
		server.AppendHandlers(
			ghttp.CombineHandlers(
				ghttp.VerifyRequest("GET", "/v1/teams/team-id/featured_channels"),
				ghttp.RespondWith(http.StatusInternalServerError, "Surprise Error"),
			),
		)

		params := v1.GetFeaturedChannelsRequestParams{
			TeamID: "team-id",
		}
		_, err := client.GetV1FeaturedChannels(context.Background(), params, nil)

		Expect(err).To(HaveOccurred())
		Expect(err.Error()).To(ContainSubstring("Surprise Error"))
	})

	It("returns error on resource not found", func() {
		server.AppendHandlers(
			ghttp.CombineHandlers(
				ghttp.VerifyRequest("GET", "/v1/teams/team-id/featured_channels"),
				ghttp.RespondWith(http.StatusNotFound, "Team ID Not Found"),
			),
		)

		params := v1.GetFeaturedChannelsRequestParams{
			TeamID: "team-id",
		}
		_, err := client.GetV1FeaturedChannels(context.Background(), params, nil)

		Expect(err).To(HaveOccurred())
		Expect(err.Error()).To(ContainSubstring("Team ID Not Found"))
	})

	It("returns error on bad request", func() {
		server.AppendHandlers(
			ghttp.CombineHandlers(
				ghttp.VerifyRequest("GET", "/v1/teams/team-id/featured_channels"),
				ghttp.RespondWith(http.StatusBadRequest, "Poorly formatted request"),
			),
		)

		params := v1.GetFeaturedChannelsRequestParams{
			TeamID: "team-id",
		}
		_, err := client.GetV1FeaturedChannels(context.Background(), params, nil)

		Expect(err).To(HaveOccurred())
		Expect(err.Error()).To(ContainSubstring("Poorly formatted request"))
	})

	It("returns array of featured channels on request rountrip success", func() {
		responseBody := `
			{
			  "data": [
			    {
			      "team_id": "team-id",
			      "channel_id": "first-channel"
  			    },
			    {
			      "team_id": "team-id",
			      "channel_id": "second-channel"
			    }
			  ]
			}`
		server.AppendHandlers(
			ghttp.CombineHandlers(
				ghttp.VerifyRequest("GET", "/v1/teams/team-id/featured_channels"),
				ghttp.RespondWith(http.StatusOK, responseBody),
			),
		)
		params := v1.GetFeaturedChannelsRequestParams{
			TeamID: "team-id",
		}
		_, err := client.GetV1FeaturedChannels(context.Background(), params, nil)

		Expect(err).NotTo(HaveOccurred())
	})
})
