package roster

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("GetV1TeamInvitations", func() {
	var (
		server *ghttp.Server
		client Client
		err    error
	)

	BeforeEach(func() {
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo((HaveOccurred()))
	})

	AfterEach(func() {
		server.Close()
	})

	Context("when roster returns a 500", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/teams/123/invitations"),
					ghttp.RespondWith(http.StatusInternalServerError, ""),
				),
			)
		})

		It("returns an error", func() {
			resp, err := client.GetV1TeamInvitations(context.Background(), "123", nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when roster returns a malformed response", func() {
		var payload string

		BeforeEach(func() {
			payload = `
						{
							"meta": {},
							"data": []"bad json"
						}
					   `
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/teams/123/invitations"),
					ghttp.RespondWith(http.StatusOK, payload),
				),
			)
		})

		It("returns an error and the raw json", func() {
			resp, err := client.GetV1TeamInvitations(context.Background(), "123", nil)
			Expect(err).To(HaveOccurred())

			jsonResponse, err := json.Marshal(resp)
			Expect(err).NotTo(HaveOccurred())
			Expect(string(jsonResponse)).NotTo(Equal(payload))
		})
	})

	Context("when roster returns a set of invitations", func() {
		var (
			teamID    string
			channelID string
			payload   string
		)

		BeforeEach(func() {
			teamID = "123"
			channelID = "some channel id"

			payloadFmt := `
					{
						"data": [
							{
								"channel_id": "%s",
								"team_id": "%s"
							}
						]
					}
				`
			payload = fmt.Sprintf(payloadFmt, channelID, teamID)

			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/teams/123/invitations"),
					ghttp.RespondWith(http.StatusOK, payload),
				),
			)
		})

		It("returns the invitations", func() {
			resp, err := client.GetV1TeamInvitations(context.Background(), teamID, nil)
			Expect(err).NotTo(HaveOccurred())

			Expect(resp).NotTo(BeNil())
			Expect(resp.Data).To(HaveLen(1))
			Expect(resp.Data[0].ChannelID).To(Equal(channelID))
			Expect(resp.Data[0].TeamID).To(Equal(teamID))
		})
	})
})
