package roster

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("GetV1TeamLiveMemberships", func() {
	var (
		server *ghttp.Server
		client Client
		err    error
	)

	BeforeEach(func() {
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo((HaveOccurred()))
	})

	AfterEach(func() {
		server.Close()
	})

	Context("when roster returns a 500", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/teams/123/live_memberships"),
					ghttp.RespondWith(http.StatusInternalServerError, ""),
				),
			)
		})

		It("returns an error", func() {
			params := v1.GetTeamLiveMembershipsRequestParams{
				TeamID: "123",
			}
			resp, err := client.GetV1TeamLiveMemberships(context.Background(), params, nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when roster returns a malformed response", func() {
		var payload string

		BeforeEach(func() {
			payload = `
						{
							"meta": {},
							"data": []"bad json"
						}
					   `
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/teams/123/live_memberships"),
					ghttp.RespondWith(http.StatusOK, payload),
				),
			)
		})

		It("returns an error and the raw json", func() {
			params := v1.GetTeamLiveMembershipsRequestParams{
				TeamID: "123",
			}

			resp, err := client.GetV1TeamLiveMemberships(context.Background(), params, nil)
			Expect(err).To(HaveOccurred())

			jsonResponse, err := json.Marshal(resp)
			Expect(err).NotTo(HaveOccurred())
			Expect(string(jsonResponse)).NotTo(Equal(payload))
		})
	})

	Context("when roster returns a set of live members", func() {
		var (
			teamID          string
			teamOwnerID     string
			channelID       string
			revenueRevealed bool
			statsRevealed   bool
			payload         string
		)

		BeforeEach(func() {
			teamID = "123"
			teamOwnerID = "team owner"
			channelID = "some channel id"
			revenueRevealed = false
			statsRevealed = true

			payloadFmt := `
					{
						"meta": {
							"team": {
								"id": "%s",
								"user_id": "%s"
							}
						},
						"data": [
							{
								"channel_id": "%s",
								"team_id": "%s",
								"revenue_revealed": %t,
								"stats_revealed": %t
							}
						]
					}
				`
			payload = fmt.Sprintf(payloadFmt, teamID, teamOwnerID, channelID, teamID, revenueRevealed, statsRevealed)

			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/teams/123/live_memberships"),
					ghttp.RespondWith(http.StatusOK, payload),
				),
			)
		})

		It("returns the live members", func() {
			params := v1.GetTeamLiveMembershipsRequestParams{
				TeamID: teamID,
			}

			resp, err := client.GetV1TeamLiveMemberships(context.Background(), params, nil)
			Expect(err).NotTo(HaveOccurred())

			Expect(resp).NotTo(BeNil())
			Expect(resp.Meta.Team.ID).To(Equal(teamID))
			Expect(resp.Meta.Team.UserID).To(Equal(teamOwnerID))

			Expect(resp.Data).To(HaveLen(1))
			Expect(resp.Data[0].ChannelID).To(Equal(channelID))
			Expect(resp.Data[0].TeamID).To(Equal(teamID))
			Expect(resp.Data[0].RevenueRevealed).To(Equal(revenueRevealed))
			Expect(resp.Data[0].StatsRevealed).To(Equal(statsRevealed))
		})
	})
})
