package roster

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("GetV1Team", func() {
	var (
		server *ghttp.Server
		client Client
		err    error
	)

	BeforeEach(func() {
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo((HaveOccurred()))
	})

	AfterEach(func() {
		server.Close()
	})

	Context("when Roster returns a 500", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/teams/123"),
					ghttp.RespondWith(http.StatusInternalServerError, ""),
				),
			)
		})

		It("returns an error", func() {
			resp, err := client.GetV1Team(context.Background(), "123", nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when Roster returns a malformed response", func() {
		var payload string

		BeforeEach(func() {
			payload = `
				{
					"data": }"bad json"{
				}
			`
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/teams/123"),
					ghttp.RespondWith(http.StatusOK, payload),
				),
			)
		})

		It("returns an error and the raw json", func() {
			resp, err := client.GetV1Team(context.Background(), "123", nil)
			Expect(err).To(HaveOccurred())

			jsonResponse, err := json.Marshal(resp)
			Expect(err).NotTo(HaveOccurred())
			Expect(string(jsonResponse)).NotTo(Equal(payload))
		})
	})

	Context("when Roster returns the team", func() {
		var (
			teamID              string
			teamName            string
			teamOwnerID         string
			displayName         string
			descriptionHTML     string
			descriptionMarkdown string
			logoID              string
			logoURL             string
			bannerID            string
			bannerURL           string
			backgroundImageID   string
			backgroundImageURL  string
		)

		BeforeEach(func() {
			teamID = "123"
			teamName = "staff"
			teamOwnerID = "team owner id"
			displayName = "displayed Name"
			descriptionHTML = "<code>stuff</code>"
			descriptionMarkdown = "`stuff`"
			logoID = "logouid"
			logoURL = "https://static-cdn.jtvnw.net/jtv_user_pictures/team-Some Team-team_logo_image-logouid-600x600.src"
			bannerID = "banneruid"
			bannerURL = "https://static-cdn.jtvnw.net/jtv_user_pictures/team-Some Team-banner_image-banneruid-640x125.jpeg"
			backgroundImageID = "backgrounduid"
			backgroundImageURL = "https://static-cdn.jtvnw.net/jtv_user_pictures/team-Some Team-background_image-backgrounduid.bmp"

			payloadFmt := `
				{
					"data": {
						"id": "%s",
						"name": "%s",
						"user_id": "%s",
						"display_name": "%s",
						"description_html": "%s",
						"description_markdown": "%s",
						"logo_id": "%s",
						"logo_url": "%s",
						"banner_id": "%s",
						"banner_url": "%s",
						"background_image_id": "%s",
						"background_image_url": "%s"
					}
				}
			`

			payload := fmt.Sprintf(
				payloadFmt,
				teamID,
				teamName,
				teamOwnerID,
				displayName,
				descriptionHTML,
				descriptionMarkdown,
				logoID,
				logoURL,
				bannerID,
				bannerURL,
				backgroundImageID,
				backgroundImageURL,
			)

			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("GET", "/v1/teams/123"),
					ghttp.RespondWith(http.StatusOK, payload),
				),
			)
		})

		It("returns the team", func() {
			expectedTeam := v1.Team{
				ID:                  teamID,
				Name:                teamName,
				UserID:              teamOwnerID,
				DisplayName:         displayName,
				DescriptionHTML:     descriptionHTML,
				DescriptionMarkdown: descriptionMarkdown,
				LogoID:              &logoID,
				LogoURL:             &logoURL,
				BannerID:            &bannerID,
				BannerURL:           &bannerURL,
				BackgroundImageID:   &backgroundImageID,
				BackgroundImageURL:  &backgroundImageURL,
			}

			resp, err := client.GetV1Team(context.Background(), "123", nil)
			Expect(err).NotTo(HaveOccurred())
			Expect(resp).NotTo(BeNil())
			Expect(resp.Data).To(Equal(expectedTeam))
		})
	})
})
